/*
	vfdcmd.c

	Virtual Floppy Disk drive control program (console version)
    Copyright (C) 2003 Kenji Kato
*/

#define WIN32_LEAN_AND_MEAN
#define _CRTDBG_MAP_ALLOC
#include <windows.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <crtdbg.h>

#include "vfdctl.h"
#include "vfdutil.h"
#include "resource.h"

//
//	prototypes for local functions
//
static int	Install(char **args, DWORD current_state);
static int	Remove(DWORD current_state);
static int	Start(DWORD current_state);
static int	Stop(DWORD current_state);
static int	Mount(char **args, DWORD current_state);
static int	Unmount(DWORD current_state);
static int	Stat(DWORD current_state);
static int	PrintHelp(const char *topic);

static BOOL Retry_Callback(DWORD param);
static BOOL Continue_Callback(DWORD err);

static void PrintMessage(UINT msg, ...);
static void PrintError(DWORD err);

//
//	main
//
int main(int argc, char **argv)
{
	DWORD	driver_state;
	char	*cmd;
	DWORD	ret;

	//	Reports memory leaks at process termination

	_CrtSetDbgFlag(_CrtSetDbgFlag(_CRTDBG_REPORT_FLAG) | _CRTDBG_LEAK_CHECK_DF);

	//	At least one parameter (command) is required

	if (argc < 2) {
		PrintHelp(NULL);
		return -1;
	}

	//	Get Current Driver state

	if ((ret = VfdGetDriverState(&driver_state)) != ERROR_SUCCESS) {
		PrintMessage(IDS_GET_STAT_NG);
		PrintError(ret);
		return -1;
	}

	//	Parse command line arguments

	cmd = *(argv + 1);

	if (stricmp(cmd, "install") == 0) {
		if (argc > 4) {
			PrintMessage(IDS_TOO_MANY_ARGS);
			PrintHelp(cmd);
			return -1;
		}

		return Install(argv + 2, driver_state);
	}
	else if (stricmp(cmd, "start") == 0) {
		if (argc > 2) {
			PrintMessage(IDS_TOO_MANY_ARGS);
			PrintHelp(cmd);
			return -1;
		}

		return Start(driver_state);
	}
	else if (stricmp(cmd, "mount") == 0) {
		if (argc > 5) {
			PrintMessage(IDS_TOO_MANY_ARGS);
			PrintHelp(cmd);
			return -1;
		}
	 
		return Mount(argv + 2, driver_state);
	}
	else if (stricmp(cmd, "remove") == 0) {
		if (argc > 2) {
			PrintMessage(IDS_TOO_MANY_ARGS);
			PrintHelp(cmd);
			return -1;
		}

		return Remove(driver_state);
	}
	else if (stricmp(cmd, "stop") == 0) {
		if (argc > 2) {
			PrintMessage(IDS_TOO_MANY_ARGS);
			PrintHelp(cmd);
			return -1;
		}

		return Stop(driver_state);
	}
	else if (stricmp(cmd, "umount") == 0) {
		if (argc > 2) {
			PrintMessage(IDS_TOO_MANY_ARGS);
			PrintHelp(cmd);
			return -1;
		}

		return Unmount(driver_state);
	}
	else if (stricmp(cmd, "stat") == 0) {
		if (argc > 2) {
			PrintMessage(IDS_TOO_MANY_ARGS);
			PrintHelp(cmd);
			return -1;
		}

		return Stat(driver_state);
	}
	else if (stricmp(cmd, "help") == 0) {
		PrintMessage(IDS_HELP_HEADER);
		return PrintHelp(*(argv + 2));
	}
	else {
		PrintMessage(IDS_INVALID_COMMAND, cmd);
		PrintHelp(NULL);
		return -1;
	}
}

//	Install Virtual Floppy Driver Dynamically
//	Command Line Parameters:
//	(optional) driver file path	- default to executive's dir
//	(optional) auto start switch - default to demand start
int	Install(char **args, DWORD driver_state)
{
	static const char *cmd = "install";
	char *install_path = NULL;
	BOOL auto_start = FALSE;

	DWORD ret;

	//	process parameters

	while (*args) {
		if (_stricmp(*args, "/auto") == 0) {
			if (auto_start) {
				PrintMessage(IDS_DUPLICATE_ARGS);
				PrintHelp(cmd);
				return -1;
			}
			else {
				auto_start = TRUE;
			}
		}
		else if (**args == '/') {
			PrintMessage(IDS_INVALID_OPTION, *args);
			PrintHelp(cmd);
			return -1;
		}
		else {
			if (install_path) {
				PrintMessage(IDS_DUPLICATE_ARGS);
				PrintHelp(cmd);
				return -1;
			}
			else {
				install_path = *args;
			}
		}
		args++;
	}

	//	already installed?

	if (driver_state != VFD_NOT_INSTALLED) {
		PrintMessage(IDS_ALREADY_INSTALLED);
		return -1;
	}

	//	install the driver

	if ((ret = VfdInstall(install_path, auto_start)) != ERROR_SUCCESS) {
		PrintMessage(IDS_INSTALL_NG);
		PrintError(ret);
		return -1;
	}

	//	operation successfull
	PrintMessage(IDS_INSTALL_OK);

	return 0;
}

//	Remove Virtual Floppy Driver from system
//	Command Line Parameters: None
int	Remove(DWORD driver_state)
{
	DWORD	ret;
	int		i;

	//	ensure the driver is installed

	if (driver_state == VFD_NOT_INSTALLED) {
		PrintMessage(IDS_NOT_INSTALLED);
		return -1;
	}

	//	ensure the driver is stopped

	if (driver_state == SERVICE_RUNNING) {

		// unmount current image

		ret = EnsureUnmount(Retry_Callback, Continue_Callback, 0);

		if (ret == ERROR_SUCCESS) {
			PrintMessage(IDS_UNMOUNT_OK);
		}
		else if (ret != ERROR_NOT_READY) {
			PrintMessage(IDS_REMOVE_NG);
			return -1;
		}

		// stop the driver

		if ((ret = VfdStop(&driver_state)) != ERROR_SUCCESS) {
			PrintMessage(IDS_STOP_NG);
			PrintError(ret);
			return -1;
		}

		if (driver_state == SERVICE_STOPPED) {
			PrintMessage(IDS_STOP_OK);
		}
		else {
			PrintMessage(IDS_STOP_PENDING);
		}
	}

	//	remove the driver

	if ((ret = VfdRemove()) != ERROR_SUCCESS) {
		PrintMessage(IDS_REMOVE_NG);
		PrintError(ret);
		return -1;
	}

	// Wait for the driver to be actually removed for 3 secs Max.

	for (i = 0; i < 10; i++) {
		ret = VfdGetDriverState(&driver_state);

		if (ret != ERROR_SUCCESS) {
			PrintMessage(IDS_GET_STAT_NG);
			PrintError(ret);
			return -1;
		}

		if (driver_state == VFD_NOT_INSTALLED) {
			break;
		}

		Sleep(300);
	}

	if (driver_state == VFD_NOT_INSTALLED) {
		PrintMessage(IDS_REMOVE_OK);
	}
	else {
		PrintMessage(IDS_REMOVE_PENDING);
	}

	return 0;
}

//	Start the Virtual FD Driver
//	Command Line Parameters: None
int	Start(DWORD driver_state)
{
	DWORD	ret;

	//	ensure that the driver is installed

	if (driver_state == VFD_NOT_INSTALLED) {
		if ((ret = VfdInstall(NULL, FALSE)) == ERROR_SUCCESS) {
			PrintMessage(IDS_INSTALL_OK);
		}
		else {
			PrintMessage(IDS_INSTALL_NG);
			PrintError(ret);
			return -1;
		}
	}
		
	//	ensure that the driver is not started yet 

	if (driver_state == SERVICE_RUNNING) {
		PrintMessage(IDS_ALREADY_RUNNING);
		return -1;
	}

	//	start the driver

	if ((ret = VfdStart(&driver_state)) != ERROR_SUCCESS) {
		PrintMessage(IDS_START_NG);
		PrintError(ret);
		return -1;
	}

	//	operation successfull
	PrintMessage(IDS_START_OK);

	return 0;
}

//	Stop Virtual Floppy Driver
//	Command Line Parameters: None
int	Stop(DWORD driver_state)
{
	DWORD ret;

	//	ensure that the driver is installed

	if (driver_state == VFD_NOT_INSTALLED) {
		PrintMessage(IDS_NOT_INSTALLED);
		return -1;
	}

	//	ensure that the driver is running

	if (driver_state == SERVICE_STOPPED) {
		PrintMessage(IDS_NOT_STARTED);
		return -1;
	}

	//	ensure that the image is unmounted

	if (driver_state == SERVICE_RUNNING) {
		ret = EnsureUnmount(Retry_Callback, Continue_Callback, 0);

		if (ret == ERROR_SUCCESS) {
			PrintMessage(IDS_UNMOUNT_OK);
		}
		else if (ret != ERROR_NOT_READY) {
			PrintMessage(IDS_STOP_NG);
			return -1;
		}
	}

	//	stop the driver

	if ((ret = VfdStop(&driver_state)) != ERROR_SUCCESS) {
		PrintMessage(IDS_STOP_NG);
		PrintError(ret);
		return -1;
	}

	if (driver_state == SERVICE_STOPPED) {
		PrintMessage(IDS_STOP_OK);
	}
	else {
		PrintMessage(IDS_STOP_PENDING);
		PrintMessage(IDS_RESTART_WARN);
	}

	return 0;
}

//	Mount an image file to Virtual Floppy Drive
//	Command Line Parameters:
//	(optional) image file path - default to most recently mounted image
//	(optional) drive letter - default to first available letter
//	(optional) read-only switch - default to image file's attribute
//	(optional) image file size (used only to create a new image)
int	Mount(char **args, DWORD driver_state)
{
	static const char *cmd	= "mount";
	const char *file_name	= NULL;
	BOOL	read_only		= FALSE;
	ULONG	file_size		= 0;
	char	drive_letter	= '\0';
	char	actual_name[MAX_PATH];
	DWORD	ret;

	//	process parameters

	while (*args) {
		if (_stricmp(*args, "/ro") == 0) {
			if (read_only) {
				PrintMessage(IDS_DUPLICATE_ARGS);
				PrintHelp(cmd);
				return -1;
			}
			else {
				read_only = TRUE;
			}
		}
		else if (strcmp(*args, "/720") == 0) {
			if (file_size) {
				PrintMessage(IDS_DUPLICATE_ARGS);
				PrintHelp(cmd);
				return -1;
			}
			else {
				file_size = VFD_FILESIZE_720KB;
			}
		}
		else if (strcmp(*args, "/144") == 0) {
			if (file_size) {
				PrintMessage(IDS_DUPLICATE_ARGS);
				PrintHelp(cmd);
				return -1;
			}
			else {
				file_size = VFD_FILESIZE_1P44MB;
			}
		}
		else if (strcmp(*args, "/288") == 0) {
			if (file_size) {
				PrintMessage(IDS_DUPLICATE_ARGS);
				PrintHelp(cmd);
				return -1;
			}
			else {
				file_size = VFD_FILESIZE_2P88MB;
			}
		}
		else if (strnicmp(*args, "/l:", 3) == 0) {
			if (drive_letter) {
				PrintMessage(IDS_DUPLICATE_ARGS);
				PrintHelp(cmd);
				return -1;
			}
			else if (!isalpha(*(*args + 3))) {
				PrintMessage(IDS_INVALID_LETTER, *(*args + 3));
				PrintHelp(cmd);
				return -1;
			}
			else {
				drive_letter = (char)toupper(*(*args + 3));
			}
		}
		else if (**args != '/') {
			if (file_name) {
				PrintMessage(IDS_DUPLICATE_ARGS);
				PrintHelp(cmd);
				return -1;
			}
			else {
				file_name = *args;
			}
		}
		else {
			PrintMessage(IDS_INVALID_OPTION, *args);
			PrintHelp(cmd);
			return -1;
		}

		if (read_only && file_size) {
			PrintMessage(IDS_OPTION_CONFLICT);
			PrintHelp(cmd);
			return -1;
		}

		args++;
	}

	//	ensure that the driver is installed

	if (driver_state == VFD_NOT_INSTALLED) {
		if ((ret = VfdInstall(NULL, FALSE)) == ERROR_SUCCESS) {
			PrintMessage(IDS_INSTALL_OK);
		}
		else {
			PrintMessage(IDS_INSTALL_NG);
			PrintError(ret);
			return -1;
		}
	}

	//	ensure that the driver is started

	if (driver_state != SERVICE_RUNNING) {
		if ((ret = VfdStart(&driver_state)) == ERROR_SUCCESS) {
			PrintMessage(IDS_START_OK);
		}
		else {
			PrintMessage(IDS_START_NG);
			PrintError(ret);
			return -1;
		}
	}

	//	Ensure that the drive is empty
	
	if ((ret = VfdGetMediaState()) != ERROR_NOT_READY) {
		if (ret == ERROR_SUCCESS || ret == ERROR_WRITE_PROTECT) {
			PrintMessage(IDS_ALREADY_MOUNTED);
		}
		else {
			PrintMessage(IDS_GET_MEDIA_NG);
			PrintError(ret);
		}
		return -1;
	}

	//	assign the drive letter to the drive

	if (drive_letter == '\0') {
		drive_letter = ChooseDriveLetter();

		if (drive_letter == '\0') {
			PrintMessage(IDS_DRIVE_FULL);
			return -1;
		}
	}

	if ((ret = VfdSetDriveLetter(drive_letter)) != ERROR_SUCCESS) {
		PrintMessage(IDS_LINK_NG, drive_letter);
		PrintError(ret);
		return -1;
	}

	//	Mount the image file

	ret = VfdMount(file_name, read_only, file_size);
	
	//	Get the image file actually mounted (or failed).
	
	if (VfdGetFileInfo(actual_name, NULL, NULL) != ERROR_SUCCESS) {
		strcpy(actual_name, "(???)");
	}

	if (ret != ERROR_SUCCESS) {
		VfdDelDriveLetter(drive_letter);
		PrintMessage(IDS_MOUNT_NG, actual_name);
		PrintError(ret);
		return -1;
	}

	//	operation successfull

	PrintMessage(IDS_MOUNT_OK, actual_name, drive_letter);

	return 0;
}

//	Unmount current image file
//	Command Line Parameters: None
int	Unmount(DWORD driver_state)
{
	DWORD	ret;

	//	ensure that the driver is installed

	if (driver_state == VFD_NOT_INSTALLED) {
		PrintMessage(IDS_NOT_INSTALLED);
		return -1;
	}

	//	ensure that the driver is running

	if (driver_state != SERVICE_RUNNING) {
		PrintMessage(IDS_NOT_STARTED);
		return -1;
	}

	//	Unmount the drive

	ret = EnsureUnmount(Retry_Callback, NULL, 0);

	if (ret != ERROR_SUCCESS) {
		PrintMessage(IDS_UNMOUNT_NG);
		PrintError(ret);
		return -1;
	}

	//	operation successfull

	PrintMessage(IDS_UNMOUNT_OK);

	return 0;
}

//	Show current driver state
//	Command Line Parameters: None
int	Stat(DWORD driver_state)
{
	TCHAR	path[MAX_PATH];
	DWORD	start_type;
	char	drive_letter;
	ULONG	file_size;
	BOOL	read_only;
	DWORD	ret;

	//	print current driver state

	PrintMessage(IDS_CURRENT_STATUS);

	switch (driver_state) {
	case VFD_NOT_INSTALLED:
		PrintMessage(IDS_STATUS_NOT_INSTALLED);
		return 0;
		// if the driver is not installed, no more info to show

	case SERVICE_STOPPED:
		PrintMessage(IDS_STATUS_STOPPED);
		break;

	case SERVICE_START_PENDING:
		PrintMessage(IDS_STATUS_START_P);
		break;

	case SERVICE_STOP_PENDING:
		PrintMessage(IDS_STATUS_STOP_P);
		break;

	case SERVICE_RUNNING:
		PrintMessage(IDS_STATUS_RUNNING);
		break;

	case SERVICE_CONTINUE_PENDING:
		PrintMessage(IDS_STATUS_CONT_P);
		break;

	case SERVICE_PAUSE_PENDING:
		PrintMessage(IDS_STATUS_PAUSE_P);
		break;

	case SERVICE_PAUSED:
		PrintMessage(IDS_STATUS_PAUSED);
		break;

	default:
		PrintMessage(IDS_STATUS_UNKNOWN, driver_state);
		break;
	}

	//	get current driver config

	if ((ret = VfdGetDriverConfig(path, &start_type)) != ERROR_SUCCESS) {
		PrintMessage(IDS_GET_CONFIG_NG);
		PrintError(ret);
		return -1;
	}

	//	print driver file path

	PrintMessage(IDS_DRIVER_LOCATION, path);

	//	print driver start type

	PrintMessage(IDS_START_TYPE);

	switch (start_type) {
	case SERVICE_AUTO_START:
		PrintMessage(IDS_START_AUTO);
		break;

	case SERVICE_BOOT_START:
		PrintMessage(IDS_START_BOOT);
		break;

	case SERVICE_DEMAND_START:
		PrintMessage(IDS_START_DEMAND);
		break;

	case SERVICE_DISABLED:
		PrintMessage(IDS_START_DISABLED);
		break;

	case SERVICE_SYSTEM_START :
		PrintMessage(IDS_START_SYSTEM);
		break;

	default:
		PrintMessage(IDS_START_UNKNOWN, start_type);
		break;
	}

	//	if driver is not running, no more info

	if (driver_state != SERVICE_RUNNING) {
		return 0;
	}

	//	image file information

	ret = VfdGetFileInfo(path, &file_size, &read_only);

	if (ret != ERROR_SUCCESS) {
		PrintMessage(IDS_GET_FILE_NG);
		PrintError(ret);
		return -1;
	}

	//	print image file information

	PrintMessage(IDS_IMAGE_FILE);
		
	if (path[0] == '\0') {
		PrintMessage(IDS_IMAGE_NONE);
	}
	else {
		printf("%s\n", path);

		//	print other file info

		if (VfdGetMediaState() == ERROR_NOT_READY) {
			PrintMessage(IDS_IMAGE_UNMOUNTED);
		}
		else {
			PrintMessage(IDS_IMAGE_SIZE, file_size);

			PrintMessage(IDS_ACCESS_TYPE);

			if (read_only) {
				PrintMessage(IDS_ACCESS_RO);
			}
			else {
				PrintMessage(IDS_ACCESS_RW);
			}

			//	get current drive letter

			if ((ret = VfdGetDriveLetter(&drive_letter)) != ERROR_SUCCESS) {
				PrintMessage(IDS_GET_LINK_NG);
				PrintError(ret);
				drive_letter = '?';
			}
			
			PrintMessage(IDS_DRIVE_LETTER, drive_letter ? drive_letter : ' ');
		}
	}

	return 0;
}

//
//	Print usage help
//
int	PrintHelp(const char *topic)
{
	int ret = 0;
	UINT msg;

	if (topic) {
		if (stricmp(topic, "install") == 0) {
			msg = IDS_HELP_INSTALL;
		}
		else if (stricmp(topic, "remove") == 0) {
			msg = IDS_HELP_REMOVE;
		}
		else if (stricmp(topic, "start") == 0) {
			msg = IDS_HELP_START;
		}
		else if (stricmp(topic, "stop") == 0) {
			msg = IDS_HELP_STOP;
		}
		else if (stricmp(topic, "mount") == 0) {
			msg = IDS_HELP_MOUNT;
		}
		else if (stricmp(topic, "umount") == 0) {
			msg = IDS_HELP_UMOUNT;
		}
		else if (stricmp(topic, "stat") == 0) {
			msg = IDS_HELP_STAT;
		}
		else if (stricmp(topic, "help") == 0) {
			msg = IDS_HELP_HELP;
		}
		else {
			PrintMessage(IDS_INVALID_COMMAND, topic);
			msg = IDS_HELP_HELP;
			ret = -1;
		}
	}
	else {
		msg = IDS_HELP_GENERAL;
	}

	PrintMessage(msg);

	return ret;
}

//
//	Confirms unmount retry
//
BOOL Retry_Callback(DWORD param)
{
	UNREFERENCED_PARAMETER(param);

	PrintMessage(IDS_UNMOUNT_NG);
	PrintMessage(IDS_DRIVE_IN_USE);

	for (;;) {
		int ans;

		PrintMessage(IDS_TRY_AGAIN);
		fflush(NULL);

		fflush(stdin);
		ans = getchar();

		if (ans == 'Y' || ans == 'y') {
			return TRUE;
		}
		else if (ans == 'N' || ans == 'n') {
			return FALSE;
		}
	}
}

//
//	Confirms unmount continue
//
BOOL Continue_Callback(DWORD err)
{
	PrintMessage(IDS_UNMOUNT_NG);
	PrintError(err);
	PrintMessage(IDS_UNLOAD_WARN);

	for (;;) {
		int ans;

		PrintMessage(IDS_ASK_CONTINUE);
		fflush(NULL);

		fflush(stdin);
		ans = getchar();
		
		if (ans == 'Y' || ans == 'y') {
			return TRUE;
		}
		else if (ans == 'N' || ans == 'n') {
			return FALSE;
		}
	}
}

//
//	Print error message
//
void PrintError(DWORD err)
{
	char buf[512];

	VfdErrorMessage(err, buf, sizeof(buf));
	printf("%s", buf);
}

//
//	Print message
//
void PrintMessage(UINT msg, ...)
{
	char buf[2048];

	if (LoadString(NULL, msg, buf, sizeof(buf))) {
		va_list list;

		va_start(list, msg);

		vprintf(buf, list);
	}
	else {
		printf("Vfd Message ID %d\n", msg);
	}
}

//	End of file
