#include <QtDebug>
#include <QNetworkReply>
#include <QNetworkAccessManager>
#include <QRegExp>
#include <QStringList>
#include "video.h"

Video::Video(const QUrl& url)
{
  orig_url = url.toString();

  manager = new QNetworkAccessManager;
  connect(manager, SIGNAL(finished(QNetworkReply*)),
          this, SLOT(ReplyFinished(QNetworkReply*)));
}

Video::~Video()
{
  delete manager;
}

QString Video::GetLocation()
{
  return location;
}

QString Video::GetTitle()
{
  return title;
}

QString Video::GetOriginalUrl()
{
  return orig_url;
}

bool Video::HasErrors(QNetworkReply *reply)
{
  if (reply->error() != QNetworkReply::NoError) {
    qDebug() << "error=" << reply->errorString();
    reply->abort();
    emit Error();
    return true;
  }

  if (reply->attribute(QNetworkRequest::HttpStatusCodeAttribute) != 200) {
    qDebug() << "status_code=" << reply->errorString();
    reply->abort();
    emit Error();
    return true;
  }

  return false;
}

Video* VideoFactory::Create(const QUrl& url)
{
  QString tmp = url.toString();

  // TODO: this should be more modularized.
  if (tmp.indexOf("www.youtube.com") != -1) {
    return new Video_YouTube(url);
  } else if (tmp.indexOf("www.dailymotion.com") != -1) {
    return new Video_Dailymotion(url);
  } else if (tmp.indexOf("vimeo.com") != -1) {
    return new Video_Vimeo(url);
  } else if (tmp.indexOf("www.veoh.com") != -1) {
    return new Video_Veoh(url);
  } else if (tmp.indexOf("www.tube8.com") != -1) {
    return new Video_Tube8(url);
  } else if (tmp.indexOf("www.yourfilehost.com") != -1) {
    return new Video_YourFileHost(url);
  } else if (tmp.indexOf("www.bofunk.com") != -1) {
    return new Video_Bofunk(url);
  } else if (tmp.indexOf("video.fc2.com")) {
    return new Video_Fc2(url);
  } else {
    return new Video_Generic(url);
  }
}

void Video_Generic::FindVideo()
{
  // TODO: this does not work currently, so throw an error.
  emit Error();
  return;

  manager->get(QNetworkRequest(QUrl(orig_url)));
}

void Video_Generic::ReplyFinished(QNetworkReply *reply)
{
  if (HasErrors(reply)) {
    return;
  }

  QString content_type = reply->rawHeader("Content-Type");
  qDebug() << "content_type=" << content_type;

  location = orig_url;
  title = orig_url;
  emit Finished();
}

void Video_YouTube::FindVideo()
{
  QRegExp regex("http://www.youtube.com/v/([^&]+)");
  if (regex.indexIn(orig_url) != -1) {
    QString video_id = regex.cap(1);
    QString tmp = "http://www.youtube.com/watch?v=" + video_id;
    manager->get(QNetworkRequest(QUrl(tmp)));
  } else {
    manager->get(QNetworkRequest(QUrl(orig_url)));
  }
}

void Video_YouTube::ReplyFinished(QNetworkReply *reply)
{
  if (HasErrors(reply)) {
    return;
  }

  QString content(reply->readAll());
  QRegExp regex;
  QString video_id;
  QString token;

  regex.setPattern("var pageVideoId = '([^']+)'");
  if (regex.indexIn(content) != -1) {
    video_id = regex.cap(1);
    qDebug() << "video_id=" << video_id;
  } else {
    qDebug() << "Couldn't get video ID.";
    emit Error();
    return;
  }

  regex.setPattern(" \"t\": \"([^\"]+)\"");
  if (regex.indexIn(content) != -1) {
    token = regex.cap(1);
    qDebug() << "token=" << token;
  } else {
    qDebug() << "Couldn't get token ID.";
  }

  // Try to find the video's title.
  regex.setPattern("<meta name=\"title\" content=\"([^\"]+)\">");
  if (regex.indexIn(content) != -1) {
    title = regex.cap(1);
    qDebug() << "title=" << title;
  } else {
    qDebug() << "Couldn't get video title.";
  }

  location = "http://www.youtube.com/get_video?video_id=" + video_id
    + "&t=" + token;

  emit Finished();
}

void Video_Tube8::FindVideo()
{
  manager->get(QNetworkRequest(QUrl(orig_url)));
}

void Video_Tube8::ReplyFinished(QNetworkReply *reply)
{
  if (HasErrors(reply)) {
    return;
  }

  QString content(reply->readAll());
  QRegExp regex;

  regex.setPattern("var videourl=\"([^\"]+)\"");
  if (regex.indexIn(content) != -1) {
    location = regex.cap(1);
    qDebug() << "location=" << location;
  } else {
    qDebug() << "Couldn't get video ID.";
    emit Error();
    return;
  }

  // Try to find the video's title.
  regex.setPattern("<h1 class=\"text\" style=\"margin: 0;\">([^<]+)</h1>");
  if (regex.indexIn(content) != -1) {
    title = regex.cap(1);
    qDebug() << "title=" << title;
  } else {
    qDebug() << "Couldn't get video title.";
  }

  emit Finished();
}

void Video_Dailymotion::FindVideo()
{
  manager->get(QNetworkRequest(QUrl(orig_url)));
}

void Video_Dailymotion::ReplyFinished(QNetworkReply *reply)
{
  if (HasErrors(reply)) {
    return;
  }

  QString content(reply->readAll());
  QRegExp regex;

  regex.setPattern("\"video\", \"([^\"]+)\"");
  if (regex.indexIn(content) != -1) {
    QString tmp = QUrl::fromEncoded(regex.cap(1).toLocal8Bit()).toString();
    location = tmp.split("@@").first();
    qDebug() << "location=" << location;
  } else {
    qDebug() << "Couldn't get video ID.";
    emit Error();
    return;
  }

  // Try to find the video's title.
  regex.setPattern("<h1 class=\"dmco_title\">([^<]+)</h1>");
  if (regex.indexIn(content) != -1) {
    title = regex.cap(1);
    qDebug() << "title=" << title;
  } else {
    qDebug() << "Couldn't get video title.";
  }

  emit Finished();
}

void Video_Vimeo::FindVideo()
{
  QRegExp regex("vimeo.com/([0-9]+)");
  QString tmp = orig_url;

  if (regex.indexIn(orig_url) != -1) {
    QString video_id = regex.cap(1);
    tmp = "http://vimeo.com/moogaloop/load/clip:" + video_id
      + "/embed?param_fullscreen=1&param_clip_id=" + video_id
      + "&param_show_byline=0&param_server=vimeo.com&param_color=cc6600&"
      + "param_show_portrait=0&param_show_title=1";
  }
  manager->get(QNetworkRequest(QUrl(tmp)));
}

void Video_Vimeo::ReplyFinished(QNetworkReply *reply)
{
  if (HasErrors(reply)) {
    return;
  }

  QString content(reply->readAll());
  QRegExp regex;
  QString video_id;
  QString signature;
  QString signature_expires;

  regex.setPattern("<nodeId>([0-9]+)</nodeId>");
  if (regex.indexIn(content) != -1) {
    video_id = regex.cap(1);
    qDebug() << "video_id=" << video_id;
  } else {
    qDebug() << "Couldn't get video ID.";
    emit Error();
    return;
  }

  regex.setPattern("<request_signature>([a-z0-9]+)</");
  if (regex.indexIn(content) != -1) {
    signature = regex.cap(1);
    qDebug() << "signature=" << signature;
  } else {
    qDebug() << "Couldn't get request signature.";
    emit Error();
    return;
  }

  regex.setPattern("<request_signature_expires>([0-9]+)</");
  if (regex.indexIn(content) != -1) {
    signature_expires = regex.cap(1);
    qDebug() << "signature_expires=" << signature_expires;
  } else {
    qDebug() << "Couldn't get request signature expires.";
    emit Error();
    return;
  }

  // Try to find the video's title.
  regex.setPattern("<caption>([^<]+)</caption>");
  if (regex.indexIn(content) != -1) {
    title = regex.cap(1);
    qDebug() << "title=" << title;
  } else {
    qDebug() << "Couldn't get video title.";
  }

  location = "http://vimeo.com/moogaloop/play/clip:" + video_id
    + "/" + signature + "/" + signature_expires + "/?q=sd&type=embed";

  emit Finished();
}

void Video_Veoh::FindVideo()
{
  QRegExp regex("/watch/(v[0-9a-zA-Z]+)");
  QString tmp = orig_url;

  if (regex.indexIn(orig_url) != -1) {
    QString video_id = regex.cap(1);
    tmp = "http://www.veoh.com/rest/video/" + video_id + "/details";
  }
  manager->get(QNetworkRequest(QUrl(tmp)));
}

void Video_Veoh::ReplyFinished(QNetworkReply *reply)
{
  if (HasErrors(reply)) {
    return;
  }

  QString content(reply->readAll());
  QRegExp regex;

  regex.setPattern("fullPreviewHashPath=\"([^\"]+)\"");
  if (regex.indexIn(content) != -1) {
    location = regex.cap(1);
    qDebug() << "location=" << location;
  } else {
    qDebug() << "Couldn't get video URL.";
    emit Error();
    return;
  }

  // Try to find the video's title.
  regex.setPattern("title=\"([^\"]+)\"");
  if (regex.indexIn(content) != -1) {
    title = regex.cap(1);
    qDebug() << "title=" << title;
  } else {
    qDebug() << "Couldn't get video title.";
  }

  emit Finished();
}

void Video_YourFileHost::FindVideo()
{
  QRegExp regex("cat=video&file=([^&]+)");
  if (regex.indexIn(orig_url) != -1) {
    QString tmp = "http://www.yourfilehost.com/media.php?cat=video&file="
      + regex.cap(1);
    manager->get(QNetworkRequest(QUrl(tmp)));
  } else {
    manager->get(QNetworkRequest(QUrl(orig_url)));
  }
}

void Video_YourFileHost::ReplyFinished(QNetworkReply *reply)
{
  if (HasErrors(reply)) {
    return;
  }

  if (reply->url().toString().indexOf("/media.php?cat=video&file=") != -1) {
    ReplyFinished1(reply);
  } else {
    ReplyFinished2(reply);
  }
}

void Video_YourFileHost::ReplyFinished1(QNetworkReply *reply)
{
  QString content(reply->readAll());
  QRegExp regex;
  QString api;

  regex.setPattern("video=([^&]+)");
  if (regex.indexIn(content) != -1) {
    api = QUrl::fromEncoded(regex.cap(1).toLocal8Bit()).toString();
    qDebug() << "api=" << api;
  } else {
    qDebug() << "Couldn't get video API URL.";
    emit Error();
    return;
  }

  // Try to find the video's title.
  title = QUrl(orig_url).queryItemValue("file");
  qDebug() << "title=" << title;

  manager->get(QNetworkRequest(QUrl(api)));
}

void Video_YourFileHost::ReplyFinished2(QNetworkReply *reply)
{
  QString content(reply->readAll());
  QRegExp regex;

  regex.setPattern("video_id=([^&]+)");
  if (regex.indexIn(content) != -1) {
    location = QUrl::fromEncoded(regex.cap(1).toLocal8Bit()).toString();
    qDebug() << "location=" << location;
  } else {
    qDebug() << "Couldn't get video URL.";
    emit Error();
    return;
  }

  emit Finished();
}

void Video_Bofunk::FindVideo()
{
  manager->get(QNetworkRequest(QUrl(orig_url)));
}

void Video_Bofunk::ReplyFinished(QNetworkReply *reply)
{
  if (HasErrors(reply)) {
    return;
  }

  QString content(reply->readAll());
  QRegExp regex;

  regex.setPattern("<embed src=\"/w/([^\"]+)\"");
  if (regex.indexIn(content) != -1) {
    location = "http://media.bofunk.com/media/flvs/" + regex.cap(1) + ".flv";
    qDebug() << "location=" << location;
  } else {
    qDebug() << "Couldn't get video URL.";
    emit Error();
    return;
  }

  // Try to find the video's title.
  regex.setPattern("<meta name=\"title\" content=\"([^\"]+)\">");
  if (regex.indexIn(content) != -1) {
    title = regex.cap(1);
    qDebug() << "title=" << title;
  } else {
    qDebug() << "Couldn't get video title.";
  }

  emit Finished();
}

void Video_Fc2::FindVideo()
{
  manager->get(QNetworkRequest(QUrl(orig_url)));
}

void Video_Fc2::ReplyFinished(QNetworkReply *reply)
{
  if (HasErrors(reply)) {
    return;
  }

  QString content(reply->readAll());
  QRegExp regex;

  regex.setPattern(
    "'(http://video[0-9].fc2.com/up/)thumb(/[0-9]+/[0-9]+/[0-9]/)"
    "FCUT_([^.]+).gif',");
  if (regex.indexIn(content) != -1) {
    location = regex.cap(1) + "flv" + regex.cap(2) + regex.cap(3) + ".flv";
    qDebug() << "location=" << location;
  } else {
    qDebug() << "Couldn't get video URL.";
    emit Error();
    return;
  }

  // Try to find the video's title.
  regex.setPattern("<meta name=\"description\" content=\"([^\"]+)\" />");
  if (regex.indexIn(content) != -1) {
    title = regex.cap(1);
    qDebug() << "title=" << title;
  } else {
    qDebug() << "Couldn't get video title.";
  }

  emit Finished();
}
