/*
 * DateTime class.
 *
 * Copyright (C) 2007 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.util;

import java.io.Serializable;
import java.util.Date;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.TimeZone;
import java.text.SimpleDateFormat;
import java.text.ParseException;
import ts.util.text.StringOperation;

/**
 * \NXB
 * <br>
 * tyюɊւyё܂Ƃ߂NXłB
 * ftHgŎgp̓OSIłÃJ_IuWFNgƂ
 * {@link java.util.GregorianCalendar GregorianCalendar}IuWFNggpB
 * P[^C][́AɎw肵Ȃꍇ̓VXeŃftHĝ
 * gpB
 * <br>
 * gp⃍P[A^C][ύXꍇ́AJ_IuWFNg
 * ɂƂRXgN^gpB
 * <br/>
 * t̉Zs`FbŃAgpĂJ_IuWFNgɔCĂB
 * <br>
 * ̃NX́AJDK̓t֘ÃNXɔׂĈȉ̋@\ǉĂF
 * <ul>
 * <li>𐮐^Őݒ薔͎擾ƂA0`11ł͂Ȃ1`12Ŏw肷悤
 *     Bpӂ萔i<code>JANUARY, FEBRUARY</code>, etc.)1`12̒l
 *     蓖ĂĂB</li>
 * <li>NbȂǂ̒l擾ƂA<code>java.util.Calendar</code>
 *     gċ߂ԂBB</li>
 * <li>t{@link java.sql.Date java.sql.Date}/
 *     {@link java.sql.Time java.sql.Time }/
 *     {@link java.sql.Timestamp java.sql.Timestamp}
 *     NX̃CX^XƂĎ擾郁\bhǉB</li>
 * <li>w肵t̊JnƏIAw肵̊JnƏIݒ
 *     郁\bhpӂB</li>
 * </ul>
 *
 * @author  V. 
 * @version $Revision: 1.1.1.1 $, $Date: 2010-10-16 00:03:43 $
 */
public class DateTime implements Serializable, Cloneable, Comparable<DateTime>
{
  /** ̃NX̃VAo[WIDB */
  static final long serialVersionUID = -2327361093160202340L;

  /** ~bPʂ̓B1970N11000b[ƂB */
  private long dateTimeMillis_ ;

  /** Xbh[JȃJ_IuWFNgi[IuWFNgB */
  private static transient ThreadLocal<Calendar> threadLocalCalendar_ ;

  /** ̃IuWFNgɐݒ肳ꂽJ_IuWFNgB */
  private Calendar calendar_ = null;

  static 
  {
    threadLocalCalendar_ = new ThreadLocal<Calendar>();
  }

  /**
   * Xbh[JȃJ_IuWFNg擾B
   *
   * @return Xbh[JȃJ_IuWFNgB
   */
  private static Calendar getTLCalendar()
  {
    Calendar cal = threadLocalCalendar_.get();
    if (cal == null) {
      cal = new GregorianCalendar();
      cal.setLenient(false);
      threadLocalCalendar_.set(cal);
    }
    return cal;
  }

  /**
   * ŎgpJ_IuWFNg擾B
   *
   * @return J_IuWFNgB
   */
  private Calendar getInnerCalendar()
  {
    return (calendar_ != null) ? calendar_ : getTLCalendar(); 
  }

  /**
   * ftHgRXgN^B
   * <br>
   * ݂̓ێCX^X\zB
   */
  public DateTime()
  {
    this(System.currentTimeMillis());
  }

  /**
   * Rs[RXgN^B
   * <br>
   * ƓyуJ_CX^X\zB
   *
   * @throws AssertionError k̏ꍇifobO[ĥ݁jB
   */
  public DateTime(DateTime datetime)
  {
    assert (datetime != null) : "@param:datetime is null.";

    setDateTime(datetime.getDateTimeMillis());

    if (datetime.calendar_ != null) {
      this.calendar_ = datetime.calendar_ ;
    }
  }

  /**
   * ~bPʂ̓ɂƂRXgN^B
   * <br>
   * 1970N11000b[Ƃ~bPʂ<code>long</code>lݒ
   * B
   *
   * @param datetime ~bPʂ̓B
   */
  public DateTime(long datetime)
  {
    setDateTime(datetime);
  }

  /**
   * JDK̓tIuWFNgɂƂRXgN^B
   *
   * @param  date {@link java.util.Date Date}IuWFNgB
   * @throws AssertionError k̏ꍇifobO[ĥ݁jB
   */
  public DateTime(java.util.Date date)
  {
    setDateTime(date);
  }

  /**
   * NɂƂRXgN^B
   * <br>
   * bɂ́A000bݒ肷B
   * <br>
   * w肳ꂽNsȏꍇ͗OX[B
   *
   * @param  year  NB
   * @param  month (1`12)B
   * @param  day   (1`31)B
   * @throws IllegalArgumentException w肳ꂽNsȏꍇB
   */
  public DateTime(int year, int month, int day) throws IllegalArgumentException
  {
    this(year, month, day, 0, 0, 0);
  }

  /**
   * NbɂƂRXgN^B
   * <br>
   * w肳ꂽNbsȏꍇ͗OX[B
   *
   * @param  year  NB
   * @param  month (1`12)B
   * @param  day   (1`31)B
   * @param  hour  (0`23)B
   * @param  min   (0`59)B
   * @param  sec   b(0`59)B
   * @throws IllegalArgumentException w肳ꂽNsȏꍇB
   */
  public DateTime(int year, int month, int day, int hour, int min, int sec)
    throws IllegalArgumentException
  {
    setDateTime(year, month, day, hour, min, sec);
  }

  /**
   * R[hƔNbɂƂRXgN^B
   * <br>
   * w肳ꂽ͔Nbsȏꍇ͗OX[B
   *
   * @param  era   R[hB
   * @param  year  NB
   * @param  month (1`12)B
   * @param  day   (1`31)B
   * @param  hour  (0`23)B
   * @param  min   (0`59)B
   * @param  sec   b(0`59)B
   * @throws IllegalArgumentException w肳ꂽNsȏꍇB
   */
  public DateTime(
    int era, int year, int month, int day, int hour, int min, int sec
  ) throws IllegalArgumentException
  {
    setDateTime(era, year, month, day, hour, min, sec);
  }

  /**
   * J_IuWFNgɂƂRXgN^B
   * <br>
   * gpɑΉ{@link java.util.Calendar Calendar}IuWFNg
   * ݒ肷B
   *
   * @param  calendar {@link java.util.Calendar Calendar}IuWFNgB
   * @throws AssertionError k̏ꍇifobO[ĥ݁jB
   */
  public DateTime(Calendar calendar)
  {
    assert (calendar != null) : "@param:calendar is null.";
    calendar_ = calendar;
    setDateTime(calendar.getTimeInMillis());
  }

  /**
   * ~bPʂŎ擾B
   *
   * @return ~bPʂ̓B
   */
  public long getDateTimeMillis()
  {
    return dateTimeMillis_ ;
  }

  /**
   * {@link java.util.Date}IuWFNgƂĎ擾B
   *
   * @return {@link java.util.Date}IuWFNgB
   */
  public java.util.Date getDate()
  {
    return new java.util.Date(getDateTimeMillis());
  }

  /**
   * {@link java.sql.Date}IuWFNgƂĎ擾B
   *
   * @return {@link java.sql.Date}IuWFNgB
   */
  public java.sql.Date getSQLDate()
  {
    return new java.sql.Date(getDateTimeMillis());
  }

  /**
   * {@link java.sql.Time}IuWFNgƂĎ擾B
   *
   * @return {@link java.sql.Time}IuWFNgB
   */
  public java.sql.Time getSQLTime()
  {
    return new java.sql.Time(getDateTimeMillis());
  }

  /**
   * {@link java.sql.Timestamp}IuWFNgƂĎ擾B
   *
   * @return {@link java.sql.Timestamp}IuWFNgB
   */
  public java.sql.Timestamp getSQLTimestamp()
  {
    return new java.sql.Timestamp(getDateTimeMillis());
  }

  /**
   * {@link java.util.Calendar}IuWFNgƂĎ擾B
   *
   * @return {@link java.util.Calendar}IuWFNgB
   */
  public Calendar getCalendar()
  {
    Calendar cal = (Calendar) getInnerCalendar().clone();
    cal.setTimeInMillis(getDateTimeMillis());
    return cal;
  }

  /**
   * ^C][擾B
   *
   * @return ^C][擾B
   */
  public TimeZone getTimeZone()
  {
    return getInnerCalendar().getTimeZone();
  }

  /**
   * <code>int</code>l擾B
   *
   * @return <code>int</code>lB
   */
  public int getEra()
  {
    Calendar cal = getInnerCalendar();
    cal.setTimeInMillis(getDateTimeMillis());
    return cal.get(Calendar.ERA);
  }

  /**
   * N擾B
   *
   * @return NB
   */
  public int getYear()
  {
    Calendar cal = getInnerCalendar();
    cal.setTimeInMillis(getDateTimeMillis());
    return cal.get(Calendar.YEAR);
  }

  /**
   * 擾B
   *
   * @return (1`12)B
   */
  public int getMonth()
  {
    Calendar cal = getInnerCalendar();
    cal.setTimeInMillis(getDateTimeMillis());
    return cal.get(Calendar.MONTH) + 1;
  }

  /**
   * 擾B
   *
   * @return (1`31)B
   */
  public int getDay()
  {
    Calendar cal = getInnerCalendar();
    cal.setTimeInMillis(getDateTimeMillis());
    return cal.get(Calendar.DAY_OF_MONTH);
  }

  /**
   * j\R[h擾B
   * <br>
   * R[h́Aj珇1`7̒lƂB
   *
   * @return j\R[hB
   */
  public int getWeek()
  {
    Calendar cal = getInnerCalendar();
    cal.setTimeInMillis(getDateTimeMillis());
    return cal.get(Calendar.DAY_OF_WEEK);
  }

  /**
   * 擾B
   *
   * @return (0`23)B
   */
  public int getHour()
  {
    Calendar cal = getInnerCalendar();
    cal.setTimeInMillis(getDateTimeMillis());
    return cal.get(Calendar.HOUR_OF_DAY);
  }

  /**
   * 擾B
   *
   * @return (0`59)B
   */
  public int getMinute()
  {
    Calendar cal = getInnerCalendar();
    cal.setTimeInMillis(getDateTimeMillis());
    return cal.get(Calendar.MINUTE);
  }

  /**
   * b擾B
   *
   * @return b(0`59)B
   */
  public int getSecond()
  {
    Calendar cal = getInnerCalendar();
    cal.setTimeInMillis(getDateTimeMillis());
    return cal.get(Calendar.SECOND);
  }

  /**
   * ~PʂŐݒ肷B
   *
   * @param  millis ~bPʂ̓B
   */
  public void setDateTime(long millis)
  {
    dateTimeMillis_ = millis;
  }

  /**
   * tIuWFNgŐݒ肷B
   *
   * @param  date {@link java.util.Date}IuWFNgB
   * @throws AssertionError k̏ꍇifobO[ĥ݁jB
   */
  public void setDateTime(java.util.Date date)
  {
    assert (date != null) : "@param:date is null.";

    setDateTime(date.getTime() - 0L);
  }

  /**
   * NŐݒ肷B
   * <br>
   * bɂ͂ꂼ[ݒ肳B
   *
   * @param  year  NB
   * @param  month (1`12)B
   * @param  day   (1`31)B
   * @throws IllegalArgumentException w肳ꂽNsȏꍇB
   */
  public void setDateTime(int year, int month, int day)
    throws IllegalArgumentException
  {
    setDateTime(year, month, day, 0, 0, 0);
  }

  /**
   * NbŐݒ肷B
   *
   * @param  y  NB
   * @param  m  (1`12)B
   * @param  d  (1`31)B
   * @param  hh (0`23)B
   * @param  mm (0`59)B
   * @param  ss b(0`59)B
   * @throws IllegalArgumentException w肳ꂽNbsȏꍇB
   */
  public void setDateTime(int y, int m, int d, int hh, int mm, int ss)
    throws IllegalArgumentException
  {
    Calendar cal = getInnerCalendar();
    cal.clear();

    cal.set(Calendar.ERA, cal.getMaximum(Calendar.ERA));
    cal.set(y, m-1, d, hh, mm, ss);

    setDateTime(cal.getTimeInMillis());
  }

  /**
   * yєNbŐݒ肷B
   *
   * @param  e  \R[hB
   * @param  y  NB
   * @param  m  (1`12)B
   * @param  d  (1`31)B
   * @param  hh (0`23)B
   * @param  mm (0`59)B
   * @param  ss b(0`59)B
   * @throws IllegalArgumentException w肳ꂽNbsȏꍇB
   */
  public void setDateTime(int e, int y, int m, int d, int hh, int mm, int ss)
    throws IllegalArgumentException
  {
    Calendar cal = getInnerCalendar();
    cal.clear();

    cal.set(Calendar.ERA, e);
    cal.set(y, m-1, d, hh, mm, ss);

    setDateTime(cal.getTimeInMillis());
  }

  /**
   * ݒ肳Ăt̎AJn(00:00:00)ɐݒ肷B
   */
  public void setStartTimeOfDay()
  {
    Calendar cal = getInnerCalendar();
    cal.setTimeInMillis(getDateTimeMillis());
    int y = cal.get(Calendar.YEAR);
    int m = cal.get(Calendar.MONTH);
    int d = cal.get(Calendar.DAY_OF_MONTH);
    cal.clear();
    cal.set(y, m, d, 0, 0, 0);

    setDateTime(cal.getTimeInMillis());
  }

  /**
   * ݒ肳Ăt̎AI(23:59:59.999)ɐݒ肷B
   */
  public void setEndTimeOfDay()
  {
    Calendar cal = getInnerCalendar();
    cal.setTimeInMillis(getDateTimeMillis());

    int y = cal.get(Calendar.YEAR);
    int m = cal.get(Calendar.MONTH);
    int d = cal.get(Calendar.DAY_OF_MONTH);
    cal.clear();
    cal.set(y, m, d, 0, 0, 0);
    cal.add(Calendar.DAY_OF_MONTH, 1);

    setDateTime(cal.getTimeInMillis() - 1L);
  }

  /**
   * ݒ肳ĂtǍ̊Jnɐݒ肷B
   */
  public void setStartDayOfMonth()
  {
    Calendar cal = getInnerCalendar();
    cal.setTimeInMillis(getDateTimeMillis());

    int y = cal.get(Calendar.YEAR);
    int m = cal.get(Calendar.MONTH);
    cal.clear();
    cal.set(y, m, 1, 0, 0, 0);

    setDateTime(cal.getTimeInMillis());
  }

  /**
   * ݒ肳ĂtǍ̏Iɐݒ肷B
   */
  public void setEndDayOfMonth()
  {
    Calendar cal = getInnerCalendar();
    cal.setTimeInMillis(getDateTimeMillis());

    int y = cal.get(Calendar.YEAR);
    int m = cal.get(Calendar.MONTH);
    int d = cal.get(Calendar.DAY_OF_MONTH);
    cal.clear();
    cal.set(y, m, 1, 0, 0, 0);
    cal.add(Calendar.MONTH, 1);

    setDateTime(cal.getTimeInMillis() - 1L);
  }

  /**
   * ݒ肳Ăt̏T̊Jnijjɐݒ肷B
   */
  public void setStartDayOfWeek()
  {
    Calendar cal = getInnerCalendar();
    cal.setTimeInMillis(getDateTimeMillis());

    int y = cal.get(Calendar.YEAR);
    int m = cal.get(Calendar.MONTH);
    int d = cal.get(Calendar.DAY_OF_MONTH);
    int w = cal.get(Calendar.DAY_OF_WEEK);
    cal.clear();
    cal.set(y, m, d, 0, 0, 0);
    cal.add(Calendar.DAY_OF_MONTH, -w + cal.SUNDAY);

    setDateTime(cal.getTimeInMillis());
  }

  /**
   * ݒ肳ĂtȀT̏Iiyjjɐݒ肷B
   */
  public void setEndDayOfWeek()
  {
    Calendar cal = getInnerCalendar();
    cal.setTimeInMillis(getDateTimeMillis());
    int y = cal.get(Calendar.YEAR);
    int m = cal.get(Calendar.MONTH);
    int d = cal.get(Calendar.DAY_OF_MONTH);
    int w = cal.get(Calendar.DAY_OF_WEEK);
    cal.clear();
    cal.set(y, m, d, 0, 0, 0);
    cal.add(Calendar.DAY_OF_MONTH, 7 - w + cal.SUNDAY);

    setDateTime(cal.getTimeInMillis() - 1L);
  }

  /**
   * w肳ꂽlŐ؂艺B
   * <br>
   * 0̏ꍇ́A؂艺sȂB
   *
   * @param  minute ؂艺̒PʂƂȂlB
   * @throws AssertionError ̏ꍇifobO[ĥ݁jB
   */
  public void floorMinute(int minute)
  {
    assert (minute >= 0) : "@param:minute is null.";

    if (minute == 0) {
      return;
    }

    Calendar cal = getInnerCalendar();
    cal.setTimeInMillis(getDateTimeMillis());
    int y = cal.get(Calendar.YEAR);
    int m = cal.get(Calendar.MONTH);
    int d = cal.get(Calendar.DAY_OF_MONTH);
    int h = cal.get(Calendar.HOUR_OF_DAY);
    int min = cal.get(Calendar.MINUTE);

    min = (min / minute) * minute;

    cal.clear();
    cal.set(y, m, d, h, min, 0);

    setDateTime(cal.getTimeInMillis());
  }

  /**
   * w肳ꂽlŐ؂グB
   * <br>
   * 0̏ꍇ́A؂グsȂB
   *
   * @param  minute ؂グ̒PʂƂȂlB
   */
  public void ceilMinute(int minute)
  {
    assert (minute >= 0) : "@param:minute is null.";

    if (minute == 0) {
      return;
    }

    Calendar cal = getInnerCalendar();
    cal.setTimeInMillis(getDateTimeMillis());
    int y = cal.get(Calendar.YEAR);
    int m = cal.get(Calendar.MONTH);
    int d = cal.get(Calendar.DAY_OF_MONTH);
    int h = cal.get(Calendar.HOUR_OF_DAY);
    int min = cal.get(Calendar.MINUTE);

    min = 60 - ((60 - min ) / minute) * minute;

    cal.clear();
    cal.set(y, m, d, h, 0, 0);
    cal.add(Calendar.MINUTE, min);

    setDateTime(cal.getTimeInMillis());
  }

  /**
   * ݒ肳ĂtɎw肳ꂽNZB
   *
   * @param  years ZNB
   */
  public void addYear(int years)
  {
    Calendar cal = getInnerCalendar();
    cal.setTimeInMillis(getDateTimeMillis());
    cal.add(Calendar.YEAR, years);

    setDateTime(cal.getTimeInMillis());
  }

  /**
   * w肳ĂtɎw肳ꂽZB
   *
   * @param  months Z錎B
   */
  public void addMonth(int months)
  {
    Calendar cal = getInnerCalendar();
    cal.setTimeInMillis(getDateTimeMillis());
    cal.add(Calendar.MONTH, months);

    setDateTime(cal.getTimeInMillis());
  }

  /**
   * w肳ĂtɎw肳ꂽZB
   *
   * @param  days ZB
   */
  public void addDay(int days)
  {
    Calendar cal = getInnerCalendar();
    cal.setTimeInMillis(getDateTimeMillis());
    cal.add(Calendar.DAY_OF_MONTH, days);

    setDateTime(cal.getTimeInMillis());
  }

  /**
   * w肳Ăt̎Ɏw肳ꂽԂZB
   *
   * @param  hours Z鎞ԁB
   */
  public void addHour(int hours)
  {
    Calendar cal = getInnerCalendar();
    cal.setTimeInMillis(getDateTimeMillis());
    cal.add(Calendar.HOUR_OF_DAY, hours);

    setDateTime(cal.getTimeInMillis());
  }

  /**
   * w肳Ăt̎Ɏw肳ꂽZB
   *
   * @param  minutes Z镪B
   */
  public void addMinute(int minutes)
  {
    Calendar cal = getInnerCalendar();
    cal.setTimeInMillis(getDateTimeMillis());
    cal.add(Calendar.MINUTE, minutes);

    setDateTime(cal.getTimeInMillis());
  }

  /**
   * ݒ肳Ăt̎Ɏw肳ꂽbZB
   *
   * @param  seconds ZbB
   */
  public void addSecond(int seconds)
  {
    Calendar cal = getInnerCalendar();
    cal.setTimeInMillis(getDateTimeMillis());
    cal.add(Calendar.SECOND, seconds);

    setDateTime(cal.getTimeInMillis());
  }

  /**
   * ̃IuWFNg̃nbVR[h擾B
   *
   * @return nbVR[hB
   */
  public int hashCode()
  {
    return (int) getDateTimeMillis();
  }

  /**
   * w肳ꂽIuWFNgƃJ_ǂ𔻒肷B
   *
   * @param  dttm rIuWFNgB
   * @return <tt><tt>true</tt>ԂB
   * @throws AssertionError k̏ꍇifobO[ĥ݁jB
   */
  protected boolean equalsCalendar(DateTime dttm)
  {
    assert (dttm != null) : "@param:dttm is null.";

    Calendar cal1 = getInnerCalendar();
    cal1.clear();
    Calendar cal2 = dttm.getInnerCalendar();
    cal2.clear();
    return cal1.equals(cal2) ? true : false;
  }

  /**
   * w肳ꂽIuWFNgƓeǂrB
   * <br>
   * {@link ts.util.DateTime DateTime}IuWFNg̏ꍇ́A
   * ̃IuWFNg̓Ǝw肳ꂽIuWFNg̓ǂ
   * rB
   * <br>
   * k̏ꍇ܂{@link ts.util.DateTime DateTime}IuWFNg
   * ł͂Ȃꍇ͏<tt>false</tt>ԂB
   *
   * @param  obj rIuWFNgB
   * @return ̃IuWFNgƈ̃IuWFNg̓eꍇ
   *           <tt>true</tt>AłȂ<tt>false</tt>ԂB
   */
  public boolean equals(Object obj)
  {
    if (obj == null) {
      return false;
    }

    if (!(obj instanceof DateTime)) {
      return false;
    }

    DateTime dttm = (DateTime) obj;
    
    if (getDateTimeMillis() != dttm.getDateTimeMillis()) {
      return false;
    }

    if (! equalsCalendar(dttm)) {
      return false;
    }

    return true;
  }

  /**
   * w肳ꂽIuWFNgƂ̑召rB
   * <br>
   * ̃IuWFNg̓Ǝw肳ꂽIuWFNg̓trāA
   * ̃IuWFNg̕ꍇ͕̐lAxꍇ͐̐lA
   * ꍇ̓[ԂB
   *
   * @param  dttm rIuWFNgB
   * @return ̃IuWFNg̓̕ꍇ͕̐lAxꍇ͐
   *           lAꍇ̓[ԂB
   * @throws NullPointerException k̏ꍇB
   * @throws IllegalArgumentException ̃IuWFNgƈ̓IuWFNg
   *           J_قȂꍇB
   */
  public int compareTo(DateTime dttm) throws NullPointerException
  {
    if (! dttm.equalsCalendar(this)) {
      throw new IllegalArgumentException(
        "Calendar of the specified date-time is different from this.");
    }
   
    long v1 = getDateTimeMillis();
    long v2 = dttm.getDateTimeMillis();
    if (v1 > v2) {
      return 1;
    }
    else if (v1 < v2) {
      return -1;
    }
    else {
      return 0;
    }
  }

  /**
   * 𕶎ɕϊďo͂B
   * <br>
   * ́A{@link java.text.SimpleDateFormat SimpleDateFormat}gpāA
   * <tt>yyyy-MM-dd HH:mm:ss</tt>̌`ŏo͂B
   *
   * @return \B
   * @see #toString(DateTime dttm)
   */
  public String toString()
  {
    return toString(this);
  }

  /**
   * Ɏw肳ꂽ𕶎ɕϊB
   * <br>
   * ́A{@link java.text.SimpleDateFormat SimpleDateFormat}gpāA
   * <tt>yyyy-MM-dd HH:mm:ss</tt>̌`ŏo͂B
   * ƈقȂꍇɂ́A<tt>"yyyy-MM-dd HH:mm:ss (G)</tt>̌`ŏo
   * B
   *
   * @param  dttm IuWFNgB
   * @return \B
   * @throws AssertionError k̏ꍇifobO[ĥ݁jB
   * @see java.text.SimpleDateFormat
   */
  public static String toString(DateTime dttm)
  {
    assert (dttm != null) : "@param:dttm is null.";

    if (dttm.getEra() == dttm.getInnerCalendar().getMaximum(Calendar.ERA)) {
      SimpleDateFormat fmt = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
      return fmt.format(dttm.getDate());
    }
    else {
      SimpleDateFormat fmt = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss (G)");
      return fmt.format(dttm.getDate());
    }
  }

  /**
   * Ɏw肳ꂽɕϊB
   * <br>
   * ̉߂ɂ́A{@link java.text.SimpleDateFormat SimpleDateFormat}
   * gpB
   * 񂪓Ƃĉ߂邽߂ɂ́Ȁ<tt>"yyyy-MM-dd"</tt>A
   * <tt>"yyyy-MM-dd HH:mm:ss"</tt>A<tt>"yyyy-MM-dd HH:mm:ss (G)"</tt>
   * ̂ꂩɏ]ĂKvB
   * <br>
   * ̏̕sȏꍇ́AOX[B
   *
   * @param  str \B 
   * @return IuWFNgB
   * @throws ParseException ̏sȏꍇB
   * @throws AssertionError k̏ꍇifobO[ĥ݁jB
   * @see java.text.SimpleDateFormat
   */
  public static DateTime parseDateTime(String str) throws ParseException
  {
    assert (str != null) : "@param:str is null.";

    SimpleDateFormat fmt;

    int len = StringOperation.length(str);
    if (len <= 10) {
      fmt = new SimpleDateFormat("yyyy-MM-dd");
    }
    else if (len <= 19) {
      fmt = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    }
    else {
      fmt = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss (G)");
    }

    fmt.setLenient(false);
    return new DateTime(fmt.parse(str));
  }

  /* -- Constants -- */

  /** P\萔B */
  public final static int JANUARY = 1;

  /** Q\萔B */
  public final static int FEBRUARY = 2;

  /** R\萔B */
  public final static int MARCH = 3;

  /** S\萔B */
  public final static int APRIL = 4;

  /** T\萔B */
  public final static int MAY = 5;

  /** U\萔B */
  public final static int JUNE = 6;

  /** V\萔B */
  public final static int JULY = 7;

  /** W\萔B */
  public final static int AUGUST = 8;

  /** X\萔B */
  public final static int SEPTEMBER = 9;

  /** 10\萔B */
  public final static int OCTOBER = 10;

  /** 11\萔B */
  public final static int NOVEMBER = 11;

  /** 12\萔B */
  public final static int DECEMBER = 12;


  /** j\萔B */
  public final static int SUNDAY = 1;

  /** j\萔B */
  public final static int MONDAY = 2;

  /** Ηj\萔B */
  public final static int TUESDAY = 3;

  /** j\萔B */
  public final static int WEDNESDAY = 4;

  /** ؗj\萔B */
  public final static int THURSDAY = 5;

  /** j\萔B */
  public final static int FRIDAY = 6;

  /** yj\萔B */
  public final static int SATURDAY = 7;
}
