/*
    Text maid for Windows
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "file.h"
#include <shlwapi.h>
#include "abort.h"
#include "common.h"
#include "dialog.h"
#include "fndir.h"
#include "general.h"
#include "history.h"
#include "other.h"
#include "resource.h"


typedef struct _SAMEFILE
{
  TCHAR szFile[MAX_PATH];           /* ja:t@CƃtpX */
  int nCommand, nCount, nSame;
  HWND hWnd;
} SAMEFILE, *LPSAMEFILE;


/******************************************************************************
*                                                                             *
* ja:t@C͊֐Q                                                       *
*                                                                             *
******************************************************************************/
/*  ja:UNICODE(gGfBA)𔻕ʂ
    lpcFile,t@C
     dwSize,oCg
        RET,TRUE:UNICODE,FALSE:̑                                       */
static BOOL
IsUnicodeLittle (LPSTR lpcFile,
                 DWORD dwSize)
{
  return dwSize >= 2 && dwSize % 2 == 0
                            && lpcFile[0] == '\xff' && lpcFile[1] == '\xfe';
}


/*  ja:UNICODE(rbNGfBA)𔻕ʂ
    lpcFile,t@C
     dwSize,oCg
        RET,TRUE:UNICODE,FALSE:̑                                       */
static BOOL
IsUnicodeBig (LPSTR lpcFile,
              DWORD dwSize)
{
  return dwSize >= 2 && dwSize % 2 == 0
                            && lpcFile[0] == '\xfe' && lpcFile[1] == '\xff';
}


/*  ja:UTF-7𔻕ʂ
    lpcFile,t@C
     dwSize,oCg
        RET,TRUE:UTF-7,FALSE:̑                                         */
static BOOL
IsUtf7 (LPSTR lpcFile,
        DWORD dwSize)
{
  int i = 0, nBits;
  BOOL fBase64 = FALSE;
  DWORD dwBits = 0;

  while (i < dwSize)
    {
      if (i % 8192 == 0)
        {
          MSG msg;

          while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
            if (!hDlgCancel || !IsDialogMessage (hDlgCancel, &msg))
              {
                TranslateMessage (&msg);
                DispatchMessage (&msg);
              }
          if (!fUserBreak)
            break;
        }
      if (fBase64)
        {
          if (lpcFile[i] == '+' || lpcFile[i] == '/'
                                    || '0' <= lpcFile[i] && lpcFile[i] <= '9'
                                    || 'A' <= lpcFile[i] && lpcFile[i] <= 'Z'
                                    || 'a' <= lpcFile[i] && lpcFile[i] <= 'z')
            {
              dwBits <<= 6;
              nBits += 6;
              if ('A' <= lpcFile[i] && lpcFile[i] <= 'Z')
                dwBits |= lpcFile[i] - 'A';
              else if ('a' <= lpcFile[i] && lpcFile[i] <= 'z')
                dwBits |= lpcFile[i] - 'a' + 26;
              else if ('0' <= lpcFile[i] && lpcFile[i] <= '9')
                dwBits |= lpcFile[i] - '0' + 52;
              else if (lpcFile[i] == '+')
                dwBits |= 62;
              else
                dwBits |= 63;
              i++;
              if (nBits >= 16)
                nBits -= 16;
            }
          else
            {
              if (lpcFile[i] == '-')
                i++;
              if (dwBits & (1 << nBits) - 1)
                return FALSE;
              fBase64 = FALSE;
            }
        }
      else
        {
          if (i + 1 < dwSize && lpcFile[i] == '+' && lpcFile[i + 1] == '-')
            {
              i += 2;
            }
          else if (lpcFile[i] == '+')
            {
              i++;
              nBits = 0;
              fBase64 = TRUE;
            }
          else if (lpcFile[i] == '\t'
                            || lpcFile[i] == cCRLF[0] || lpcFile[i] == cCRLF[1]
                            || ' ' <= lpcFile[i] && lpcFile[i] <= '*'
                            || ',' <= lpcFile[i] && lpcFile[i] <= '['
                            || ']' <= lpcFile[i] && lpcFile[i] <= '}')
            {
              i++;
            }
          else
            {
              return FALSE;
            }
        }
    }
  return TRUE;
}


/*  ja:UTF-8𔻕ʂ
    lpcFile,t@C
     dwSize,oCg
        RET,TRUE:UTF-8,FALSE:̑                                         */
static BOOL
IsUtf8 (LPSTR lpcFile,
        DWORD dwSize)
{
  int i = 0;

  while (i < dwSize)
    {
      if (i % 8192 == 0)
        {
          MSG msg;

          while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
            if (!hDlgCancel || !IsDialogMessage (hDlgCancel, &msg))
              {
                TranslateMessage (&msg);
                DispatchMessage (&msg);
              }
          if (!fUserBreak)
            break;
        }
      if (i + 2 < dwSize && lpcFile[i] == '\x1b'
            && (lpcFile[i + 1] == '('
                            && (lpcFile[i + 2] == 'B' || lpcFile[i + 2] == 'J')
            || lpcFile[i + 1] == '$'
                        && (lpcFile[i + 2] == '@' || lpcFile[i + 2] == 'B')))
        return FALSE;
      if ((BYTE)lpcFile[i] <= 0x7f)
        i++;
      else if (i + 1 < dwSize
                && (0xc2 <= (BYTE)lpcFile[i] && (BYTE)lpcFile[i] <= 0xdf
                && 0x80 <= (BYTE)lpcFile[i + 1] && (BYTE)lpcFile[i + 1] <= 0xbf
                || (BYTE)lpcFile[i] == 0xc0 && (BYTE)lpcFile[i + 1] == 0x80))
        i += 2;
      else if (i + 2 < dwSize
                && 0xe0 <= (BYTE)lpcFile[i] && (BYTE)lpcFile[i] <= 0xef
                && 0x80 <= (BYTE)lpcFile[i + 1] && (BYTE)lpcFile[i + 1] <= 0xbf
                && 0x80 <= (BYTE)lpcFile[i + 2] && (BYTE)lpcFile[i + 2] <= 0xbf
                                && lpcFile[i] & 0x0f | lpcFile[i + 1] & 0x20)
        i += 3;
      else
        return FALSE;
    }
  return TRUE;
}


/*  ja:ISO-2022-JP𔻕ʂ
    lpcFile,t@C
     dwSize,oCg
        RET,TRUE:ISO-2022-JP,FALSE:̑                                   */
static BOOL
IsIso2022Jp (LPSTR lpcFile,
             DWORD dwSize)
{
  int i = 0;
  BOOL fKanji = FALSE;

  while (i < dwSize)
    {
      if (i % 8192 == 0)
        {
          MSG msg;

          while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
            if (!hDlgCancel || !IsDialogMessage (hDlgCancel, &msg))
              {
                TranslateMessage (&msg);
                DispatchMessage (&msg);
              }
          if (!fUserBreak)
            break;
        }
      if (i + 2 < dwSize && lpcFile[i] == '\x1b' && lpcFile[i + 1] == '('
                        && (lpcFile[i + 2] == 'B' || lpcFile[i + 2] == 'J'))
        {
          /* ja:OUT */
          if (!fKanji)
            return FALSE;
          fKanji = FALSE;
          i += 3;
        }
      else if (i + 2 < dwSize && lpcFile[i] == '\x1b' && lpcFile[i + 1] == '$'
                        && (lpcFile[i + 2] == '@' || lpcFile[i + 2] == 'B'))
        {
          /* ja:IN */
          if (fKanji)
            return FALSE;
          fKanji = TRUE;
          i += 3;
        }
      else
        {
          if (fKanji)
            {
              int j;

              if (i + 1 >= dwSize)
                return FALSE;
              for (j = 0; j < 2; j++)
                {
                  if ((BYTE)lpcFile[i] < 0x21 || 0x7e < (BYTE)lpcFile[i])
                    return FALSE;
                  i++;
                }
            }
          else
            {
              if (0x80 <= (BYTE)lpcFile[i])
                return FALSE;
              i++;
            }
        }
    }
  return !fKanji;
}


/*  ja:EUC-JP𔻕ʂ
    lpcFile,t@C
     dwSize,oCg
        RET,TRUE:EUC-JP,FALSE:̑                                        */
static BOOL
IsEucJp (LPSTR lpcFile,
         DWORD dwSize)
{
  int i = 0;

  while (i < dwSize)
    {
      if (i % 8192 == 0)
        {
          MSG msg;

          while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
            if (!hDlgCancel || !IsDialogMessage (hDlgCancel, &msg))
              {
                TranslateMessage (&msg);
                DispatchMessage (&msg);
              }
          if (!fUserBreak)
            break;
        }
      if (i + 2 < dwSize && lpcFile[i] == '\x1b'
            && (lpcFile[i + 1] == '('
                        && (lpcFile[i + 2] == 'B' || lpcFile[i + 2] == 'J')
            || lpcFile[i + 1] == '$'
                        && (lpcFile[i + 2] == '@' || lpcFile[i + 2] == 'B')))
        return FALSE;
      if (0xa1 <= (BYTE)lpcFile[i] && (BYTE)lpcFile[i] <= 0xfe)
        {
          i++;
          if (i >= dwSize
                        || (BYTE)lpcFile[i] < 0xa1 || 0xfe < (BYTE)lpcFile[i])
            return FALSE;
          i++;
        }
      else if ((BYTE)lpcFile[i] == 0x8f)
        {
          int j;

          i++;
          for (j = 0; j < 2; j++)
            {
              if (i >= dwSize
                        || (BYTE)lpcFile[i] < 0xa1 || 0xfe < (BYTE)lpcFile[i])
                return FALSE;
              i++;
            }
        }
      else
        {
          if ((BYTE)lpcFile[i] == 0x8e)
            {
              i++;
              if (i >= dwSize
                        || (BYTE)lpcFile[i] < 0xa1 || 0xdf < (BYTE)lpcFile[i])
                return FALSE;
            }
          i++;
        }
    }
  return TRUE;
}


/*  ja:SHIFT_JIS𔻕ʂ
    lpcFile,t@C
     dwSize,oCg
        RET,TRUE:SHIFT_JIS,FALSE:̑                                     */
static BOOL
IsShiftJis (LPSTR lpcFile,
            DWORD dwSize)
{
  int i = 0;

  while (i < dwSize)
    {
      if (i % 8192 == 0)
        {
          MSG msg;

          while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
            if (!hDlgCancel || !IsDialogMessage (hDlgCancel, &msg))
              {
                TranslateMessage (&msg);
                DispatchMessage (&msg);
              }
          if (!fUserBreak)
            break;
        }
      if (i + 2 < dwSize && lpcFile[i] == '\x1b'
            && (lpcFile[i + 1] == '('
                        && (lpcFile[i + 2] == 'B' || lpcFile[i + 2] == 'J')
            || lpcFile[i + 1] == '$'
                        && (lpcFile[i + 2] == '@' || lpcFile[i + 2] == 'B')))
        return FALSE;
      if (0x81 <= (BYTE)lpcFile[i] && (BYTE)lpcFile[i] <= 0x9f
                    || 0xe0 <= (BYTE)lpcFile[i] && (BYTE)lpcFile[i] <= 0xfc)
        {
          i++;
          if (i >= dwSize || (BYTE)lpcFile[i] < 0x40
                        || (BYTE)lpcFile[i] == 0x7f || 0xfc < (BYTE)lpcFile[i])
            return FALSE;
          i++;
        }
      else
        {
          i++;
        }
    }
  return TRUE;
}


/*  ja:UNICODE(gGfBA)CRLF
    lpcFile,t@C
     dwSize,oCg
        RET,                                                          */
static int
EnterUnicodeLittleCrlf (LPSTR lpcFile,
                        DWORD dwSize)
{
  int i;

  for (i = 0; i + 3 < dwSize
            && (lpcFile[i] != cCRLF[0] || lpcFile[i + 1] != '\0'
            || lpcFile[i + 2] != cCRLF[1] || lpcFile[i + 3] != '\0'); i += 2);
  if (i + 3 >= dwSize)
    i += 2;
  return i;
}


/*  ja:UNICODE(gGfBA)CR
    lpcFile,t@C
     dwSize,oCg
        RET,                                                          */
static int
EnterUnicodeLittleCr (LPSTR lpcFile,
                      DWORD dwSize)
{
  int i;

  for (i = 0; i + 1 < dwSize
                && (lpcFile[i] != cCRLF[0] || lpcFile[i + 1] != '\0'); i += 2);
  return i;
}


/*  ja:UNICODE(gGfBA)LF
    lpcFile,t@C
     dwSize,oCg
        RET,                                                          */
static int
EnterUnicodeLittleLf (LPSTR lpcFile,
                      DWORD dwSize)
{
  int i;

  for (i = 0; i + 1 < dwSize
                && (lpcFile[i] != cCRLF[1] || lpcFile[i + 1] != '\0'); i += 2);
  return i;
}


/*  ja:UNICODE(rbNGfBA)CRLF
    lpcFile,t@C
     dwSize,oCg
        RET,                                                          */
static int
EnterUnicodeBigCrlf (LPSTR lpcFile,
                     DWORD dwSize)
{
  int i;

  for (i = 0; i + 3 < dwSize
            && (lpcFile[i] != '\0' || lpcFile[i + 1] != cCRLF[0]
            || lpcFile[i + 2] !='\0' || lpcFile[i + 3] != cCRLF[1]); i += 2);
  if (i + 3 >= dwSize)
    i += 2;
  return i;
}


/*  ja:UNICODE(rbNGfBA)CR
    lpcFile,t@C
     dwSize,oCg
        RET,                                                          */
static int
EnterUnicodeBigCr (LPSTR lpcFile,
                   DWORD dwSize)
{
  int i;

  for (i = 0; i + 1 < dwSize
                && (lpcFile[i] != '\0' || lpcFile[i + 1] != cCRLF[0]); i += 2);
  return i;
}


/*  ja:UNICODE(rbNGfBA)LF
    lpcFile,t@C
     dwSize,oCg
        RET,                                                          */
static int
EnterUnicodeBigLf (LPSTR lpcFile,
                   DWORD dwSize)
{
  int i;

  for (i = 0; i + 1 < dwSize
                && (lpcFile[i] != '\0' || lpcFile[i + 1] != cCRLF[1]); i += 2);
  return i;
}


/*  ja:}`oCgCRLF
    lpcFile,t@C
     dwSize,oCg
        RET,                                                          */
static int
EnterMultiByteCrlf (LPSTR lpcFile,
                    DWORD dwSize)
{
  int i;

  for (i = 0; i + 1 < dwSize
            && (lpcFile[i] != cCRLF[0] || lpcFile[i + 1] != cCRLF[1]); i++);
  if (i + 1 >= dwSize)
    i++;
  return i;
}


/*  ja:}`oCgCR
    lpcFile,t@C
     dwSize,oCg
        RET,                                                          */
static int
EnterMultiByteCr (LPSTR lpcFile,
                  DWORD dwSize)
{
  int i;

  for (i = 0; i < dwSize && lpcFile[i] != cCRLF[0]; i++);
  return i;
}


/*  ja:}`oCgLF
    lpcFile,t@C
     dwSize,oCg
        RET,                                                          */
static int
EnterMultiByteLf (LPSTR lpcFile,
                  DWORD dwSize)
{
  int i;

  for (i = 0; i < dwSize && lpcFile[i] != cCRLF[1]; i++);
  return i;
}


/*  ja:UNICODE(gGfBA)ϊ
          p,Cobt@
    lpcFile,t@C
     dwSize,oCg
        RET,TRUE:I,FALSE:G[                                      */
static BOOL
TranslateUnicodeLittle (LPLINEBUF p,
                        LPSTR     lpcFile,
                        DWORD     dwSize)
{
  p->nLength = WideCharToMultiByte (CP_SJIS, 0,
                (LPWSTR)lpcFile, dwSize / sizeof (WCHAR), NULL, 0, NULL, NULL);
  p->lpszText = MemoryAlloc (p->nLength * sizeof (CHAR));
  p->nLength = WideCharToMultiByte (CP_SJIS,0,
                                    (LPWSTR)lpcFile, dwSize / sizeof (WCHAR),
                                    p->lpszText, p->nLength, NULL, NULL);
  return TRUE;
}


/*  ja:UNICODE(rbNGfBA)ϊ
          p,Cobt@
    lpcFile,t@C
     dwSize,oCg
        RET,TRUE:I,FALSE:G[                                      */
static BOOL
TranslateUnicodeBig (LPLINEBUF p,
                     LPSTR     lpcFile,
                     DWORD     dwSize)
{
  int i;
  BOOL fResult;
  LPSTR lpcText;

  lpcText = MemoryAlloc (dwSize * sizeof (CHAR));
  for (i = 0; i < dwSize; i += 2)
    {
      lpcText[i] = lpcFile[i + 1];
      lpcText[i + 1] = lpcFile[i];
    }
  fResult = TranslateUnicodeLittle (p, lpcText, dwSize);
  MemoryFree (lpcText);
  return fResult;
}


/*  ja:UTF-7ϊ
          p,Cobt@
    lpcFile,t@C
     dwSize,oCg
        RET,TRUE:I,FALSE:G[                                      */
static BOOL
TranslateUtf7 (LPLINEBUF p,
               LPSTR     lpcFile,
               DWORD     dwSize)
{
  int i = 0, j = 0, nBits;
  BOOL fBase64 = FALSE, fResult;
  DWORD dwBits = 0;
  LPSTR lpcText;

  while (i < dwSize)
    if (fBase64)
      {
        if (lpcFile[i] == '+' || lpcFile[i] == '/'
                                    || '0' <= lpcFile[i] && lpcFile[i] <= '9'
                                    || 'A' <= lpcFile[i] && lpcFile[i] <= 'Z'
                                    || 'a' <= lpcFile[i] && lpcFile[i] <= 'z')
          {
            nBits += 6;
            i++;
            if (nBits >= 16)
              {
                j++;
                nBits -= 16;
              }
          }
        else
          {
            if (lpcFile[i] == '-')
              i++;
            fBase64 = FALSE;
          }
      }
    else
      {
        if (i + 1 < dwSize && lpcFile[i] == '+' && lpcFile[i + 1] == '-')
          {
            i += 2;
            j++;
          }
        else if (lpcFile[i] == '+')
          {
            i++;
            nBits = 0;
            fBase64 = TRUE;
          }
        else
          {
            i++;
            j++;
          }
      }
  lpcText = MemoryAlloc (j * sizeof (WCHAR));
  i = j = 0;
  fBase64 = FALSE;
  while (i < dwSize)
    if (fBase64)
      {
        if (lpcFile[i] == '+' || lpcFile[i] == '/'
                                    || '0' <= lpcFile[i] && lpcFile[i] <= '9'
                                    || 'A' <= lpcFile[i] && lpcFile[i] <= 'Z'
                                    || 'a' <= lpcFile[i] && lpcFile[i] <= 'z')
          {
            dwBits <<= 6;
            nBits += 6;
            if ('A' <= lpcFile[i] && lpcFile[i] <= 'Z')
              dwBits |= lpcFile[i] - 'A';
            else if ('a' <= lpcFile[i] && lpcFile[i] <= 'z')
              dwBits |= lpcFile[i] - 'a' + 26;
            else if ('0' <= lpcFile[i] && lpcFile[i] <= '9')
              dwBits |= lpcFile[i] - '0' + 52;
            else if (lpcFile[i] == '+')
              dwBits |= 62;
            else
              dwBits |= 63;
            i++;
            if (nBits >= 16)
              {
                lpcText[j++] = dwBits >> nBits - 16 & 0xff;
                lpcText[j++] = dwBits >> nBits - 8 & 0xff;
                nBits -= 16;
              }
          }
        else
          {
            if (lpcFile[i] == '-')
              i++;
            fBase64 = FALSE;
          }
      }
    else
      {
        if (i + 1 < dwSize && lpcFile[i] == '+' && lpcFile[i + 1] == '-')
          {
            lpcText[j++] = '+';
            lpcText[j++] = '\0';
            i += 2;
          }
        else if (lpcFile[i] == '+')
          {
            i++;
            nBits = 0;
            fBase64 = TRUE;
          }
        else
          {
            lpcText[j++] = lpcFile[i++];
            lpcText[j++] = '\0';
          }
      }
  fResult = TranslateUnicodeLittle (p, lpcText, j);
  MemoryFree (lpcText);
  return fResult;
}


/*  ja:UTF-8ϊ
          p,Cobt@
    lpcFile,t@C
     dwSize,oCg
        RET,TRUE:I,FALSE:G[                                      */
static BOOL
TranslateUtf8 (LPLINEBUF p,
               LPSTR     lpcFile,
               DWORD     dwSize)
{
  int i = 0, j = 0;
  BOOL fResult;
  LPSTR lpcText;

  while (i < dwSize)
    {
      if (i + 1 < dwSize
                    && 0xc0 <= (BYTE)lpcFile[i] && (BYTE)lpcFile[i] <= 0xdf)
        i += 2;
      else if (i + 2 < dwSize && 0xe0 <= (BYTE)lpcFile[i])
        i += 3;
      else
        i++;
      j++;
    }
  lpcText = MemoryAlloc (j * sizeof (WCHAR));
  i = j = 0;
  while (i < dwSize)
    {
      if (i + 1 < dwSize
                    && 0xc0 <= (BYTE)lpcFile[i] && (BYTE)lpcFile[i] <= 0xdf)
        {
          lpcText[j] = lpcFile[i] << 6 | lpcFile[i + 1] & 0x3f;
          lpcText[j + 1] = lpcFile[i] >> 2 & 0x07;
          i += 2;
        }
      else if (i + 2 < dwSize && 0xe0 <= (BYTE)lpcFile[i])
        {
          lpcText[j] = lpcFile[i + 1] << 6 | lpcFile[i + 2] & 0x3f;
          lpcText[j + 1] = lpcFile[i] << 4 | (lpcFile[i + 1] & 0x3f) >> 2;
          i += 3;
        }
      else
        {
          lpcText[j] = lpcFile[i++];
          lpcText[j + 1] = '\0';
        }
      j += 2;
    }
  fResult = TranslateUnicodeLittle (p, lpcText, j);
  MemoryFree (lpcText);
  return fResult;
}


/*  ja:ISO-2022-JPϊ
          p,Cobt@
    lpcFile,t@C
     dwSize,oCg
        RET,TRUE:I,FALSE:G[                                      */
static BOOL
TranslateIso2022Jp (LPLINEBUF p,
                    LPSTR     lpcFile,
                    DWORD     dwSize)
{
  int i = 0, j = 0;
  BOOL fKanji = FALSE;

  p->lpszText = MemoryAlloc (dwSize * sizeof (CHAR));
  while (i < dwSize)
    {
      if (i + 2 < dwSize && lpcFile[i] == '\x1b' && lpcFile[i + 1] == '('
                        && (lpcFile[i + 2] == 'B' || lpcFile[i + 2] == 'J'))
        {
          /* ja:OUT */
          fKanji = FALSE;
          i += 3;
        }
      else if (i + 2 < dwSize && lpcFile[i] == '\x1b' && lpcFile[i + 1] == '$'
                        && (lpcFile[i + 2] == '@' || lpcFile[i + 2] == 'B'))
        {
          /* ja:IN */
          fKanji = TRUE;
          i += 3;
        }
      else if (i + 1 < dwSize && fKanji)
        {
          p->lpszText[j] = lpcFile[i];
          p->lpszText[j + 1] = lpcFile[i + 1];
          p->lpszText[j + 1] += p->lpszText[j] & 1 ? 0x1f : 0x7d;
          if ((BYTE)p->lpszText[j + 1] >= 0x7f)
            p->lpszText[j + 1]++;
          p->lpszText[j] = ((BYTE)p->lpszText[j] - 0x21 >> 1) + 0x81;
          if ((BYTE)p->lpszText[j] > 0x9f)
            p->lpszText[j] += 0x40;
          i += 2;
          j += 2;
        }
      else
        {
          p->lpszText[j++] = lpcFile[i++];
        }
    }
  p->nLength = j;
  return TRUE;
}


/*  ja:EUC-JPϊ
          p,Cobt@
    lpcFile,t@C
     dwSize,oCg
        RET,TRUE:I,FALSE:G[                                      */
static BOOL
TranslateEucJp (LPLINEBUF p,
                LPSTR     lpcFile,
                DWORD     dwSize)
{
  int i = 0, j = 0;

  p->lpszText = MemoryAlloc (dwSize * sizeof (CHAR));
  while (i < dwSize)
    {
      if (i + 1 < dwSize
                    && 0xa1 <= (BYTE)lpcFile[i] && (BYTE)lpcFile[i] <= 0xfe)
        {
          p->lpszText[j] = lpcFile[i] & ~0x80;
          p->lpszText[j + 1] = lpcFile[i + 1] & ~0x80;
          p->lpszText[j + 1] += p->lpszText[j] & 1 ? 0x1f : 0x7d;
          if ((BYTE)p->lpszText[j + 1] >= 0x7f)
            p->lpszText[j + 1]++;
          p->lpszText[j] = ((BYTE)p->lpszText[j] - 0x21 >> 1) + 0x81;
          if ((BYTE)p->lpszText[j] > 0x9f)
            p->lpszText[j] += 0x40;
          i += 2;
          j += 2;
        }
      else if ((BYTE)lpcFile[i] == 0x8f)
        {
          i += 3;
          p->lpszText[j++] = '?';
        }
      else
        {
          if (i + 1 < dwSize && (BYTE)lpcFile[i] == 0x8e)
            i++;
          p->lpszText[j++] = lpcFile[i++];
        }
    }
  p->nLength = j;
  return TRUE;
}


/*  ja:SHIFT_JISϊ
          p,Cobt@
    lpcFile,t@C
     dwSize,oCg
        RET,TRUE:I,FALSE:G[                                      */
static BOOL
TranslateShiftJis (LPLINEBUF p,
                   LPSTR     lpcFile,
                   DWORD     dwSize)
{
  p->lpszText = MemoryAlloc (dwSize * sizeof (CHAR));
  p->nLength = dwSize;
  MemoryCopy (p->lpszText, lpcFile, dwSize * sizeof (CHAR));
  return TRUE;
}


/*  ja:TXTt@CJ
           ptw,TXTEChE
     dwCharSet,LN^[Zbg̎ʌ
    fNegotiate,TRUE:s,FALSE:ftHg
      uRetCode,sR[h
       RET,TRUE:I,FALSE:G[                                       */
BOOL
OpenTextFile (LPTEXTWND ptw,
              DWORD     dwCharSet,
              BOOL      fNegotiate,
              UINT      uRetCode)
{
  int i = 0, nLength;
  DWORD dwSize;
  HANDLE hFile, hMap;
  LPTSTR lpszText;
  LPLINEBUF p, q;
  LPSTR lpcFile;
  BOOL (*IsCharSet[7])(LPSTR, DWORD) = {IsUnicodeLittle, IsUnicodeBig,
                            IsUtf7, IsUtf8, IsIso2022Jp, IsEucJp, IsShiftJis};
  int (*EnterCharSet[7][3])(LPSTR, DWORD) = {
        {EnterUnicodeLittleCrlf, EnterUnicodeLittleCr, EnterUnicodeLittleLf},
        {EnterUnicodeBigCrlf,    EnterUnicodeBigCr,    EnterUnicodeBigLf},
        {EnterMultiByteCrlf,     EnterMultiByteCr,     EnterMultiByteLf},
        {EnterMultiByteCrlf,     EnterMultiByteCr,     EnterMultiByteLf},
        {EnterMultiByteCrlf,     EnterMultiByteCr,     EnterMultiByteLf},
        {EnterMultiByteCrlf,     EnterMultiByteCr,     EnterMultiByteLf},
        {EnterMultiByteCrlf,     EnterMultiByteCr,     EnterMultiByteLf}};
  int (*EnterText)(LPSTR, DWORD);
  BOOL (*TranslateCharSet[7])(LPLINEBUF, LPSTR, DWORD) = {
        TranslateUnicodeLittle, TranslateUnicodeBig, TranslateUtf7,
        TranslateUtf8, TranslateIso2022Jp, TranslateEucJp, TranslateShiftJis};
  BOOL (*TranslateText)(LPLINEBUF, LPSTR, DWORD);

  hFile = CreateFile (ptw->szFile, GENERIC_READ, FILE_SHARE_READ, NULL,
                                OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  if (hFile == INVALID_HANDLE_VALUE)
    {
      lpszText = LoadText (hInst, IDS_FILE_OPEN);
      MessageBox (hWndClient, lpszText,
                                APPLICATION, MB_OK | MB_ICONEXCLAMATION);
      MemoryFree (lpszText);
      return FALSE;
    }
  dwSize = GetFileSize (hFile, NULL);
  if (dwSize == -1)
    {
      MessageBox (hWndClient, _T("GetFileSize"),
                                APPLICATION, MB_OK | MB_ICONEXCLAMATION);
      CloseHandle (hFile);
      return FALSE;
    }
  if (dwCharSet == 0)
    dwCharSet = 0x2135467;
  ptw->uCharSet = (dwCharSet & 15) - 1;
  ptw->uRetCode = uRetCode;
  p = MemoryAlloc (sizeof (LINEBUF));
  ptw->nMax = 1;
  ptw->nOff = 0;
  ptw->lpStart = p;
  if (dwSize == 0)
    {
      if (!CloseHandle (hFile))
        {
          MessageBox (hWndClient, _T("CloseHandle"),
                                    APPLICATION, MB_OK | MB_ICONEXCLAMATION);
          MemoryFree (p->lpszText);
          MemoryFree (p);
          return FALSE;
        }
      return TRUE;
    }
  hMap = CreateFileMapping (hFile, NULL, PAGE_READONLY, 0, 0, NULL);
  if (!hMap)
    {
      MessageBox (hWndClient, _T("CreateFileMapping"),
                                APPLICATION, MB_OK | MB_ICONEXCLAMATION);
      MemoryFree (p->lpszText);
      MemoryFree (p);
      CloseHandle (hFile);
      return FALSE;
    }
  lpcFile = MapViewOfFile (hMap, FILE_MAP_READ, 0, 0, 0);
  if (!lpcFile)
    {
      MessageBox (hWndClient, _T("MapViewOfFile"),
                                APPLICATION, MB_OK | MB_ICONEXCLAMATION);
      MemoryFree (p->lpszText);
      MemoryFree (p);
      CloseHandle (hMap);
      CloseHandle (hFile);
      return FALSE;
    }
  fUserBreak = TRUE;
  EnableWindow (hWndMain, FALSE);
  lpszText = LoadText (hInst, IDS_JOB_READ);
  hDlgCancel = CreateDialogParamGUI (hInst, MAKEINTRESOURCE (DIALOG_G),
                                hWndClient, AbortDlgProc, (LPARAM)lpszText);
  MemoryFree (lpszText);
  if (!hDlgCancel)
    {
      MessageBox (hWndClient, _T("CreateDialogGUIParam"),
                                APPLICATION, MB_OK | MB_ICONEXCLAMATION);
      MemoryFree (p->lpszText);
      MemoryFree (p);
      EnableWindow (hWndMain, TRUE);
      UnmapViewOfFile (lpcFile);
      CloseHandle (hMap);
      CloseHandle (hFile);
      return FALSE;
    }

  /* ja:LN^[Zbg */
  if ((dwCharSet & ~15) != 0)
    {
      DWORD dwAuto;

      for (dwAuto = dwCharSet; fUserBreak && dwAuto & 15; dwAuto >>= 4)
        if (IsCharSet[(dwAuto & 15) - 1](lpcFile, dwSize) || !fUserBreak)
          {
            ptw->uCharSet = (dwAuto & 15) - 1;
            break;
          }
    }

  /* ja:sR[h */
  if (fNegotiate)
    {
      int nCRLF[3] = {0, 0, 0};

      switch (ptw->uCharSet)
        {
          case 0:/* ja:UNICODE(gGfBA) */
            while (i < dwSize)
              if (i <= dwSize - 4
                    && lpcFile[i] == cCRLF[0] && lpcFile[i + 1] == '\0'
                    && lpcFile[i + 2] == cCRLF[1] && lpcFile[i + 3] == '\0')
                {
                  i += 4;
                  nCRLF[0]++;
                }
              else if (i <= dwSize - 2
                        && lpcFile[i] == cCRLF[0] && lpcFile[i + 1] == '\0')
                {
                  i += 2;
                  nCRLF[1]++;
                }
              else if (i <= dwSize - 2
                        && lpcFile[i] == cCRLF[1] && lpcFile[i + 1] == '\0')
                {
                  i += 2;
                  nCRLF[2]++;
                }
              else
                {
                  i += 2;
                }
            break;
          case 1:/* ja:UNICODE(rbNGfBA) */
            while (i < dwSize)
              if (i <= dwSize - 4
                    && lpcFile[i] == '\0' && lpcFile[i + 1] == cCRLF[0]
                    && lpcFile[i + 2] == '\0' && lpcFile[i + 3] == cCRLF[1])
                {
                  i += 4;
                  nCRLF[0]++;
                }
              else if (i <= dwSize - 2
                        && lpcFile[i] == '\0' && lpcFile[i + 1] == cCRLF[0])
                {
                  i += 2;
                  nCRLF[1]++;
                }
              else if (i <= dwSize - 2
                        && lpcFile[i] == '\0' && lpcFile[i + 1] == cCRLF[1])
                {
                  i += 2;
                  nCRLF[2]++;
                }
              else
                {
                  i += 2;
                }
            break;
          default:/* ja:ISO-2022-JP,UTF-7,UTF-8,EUC-JP,SHIFT_JIS */
            while (i < dwSize)
              if (i <= dwSize - 2
                    && lpcFile[i] == cCRLF[0] && lpcFile[i + 1] == cCRLF[1])
                {
                  i += 2;
                  nCRLF[0]++;
                }
              else if (lpcFile[i] == cCRLF[0])
                {
                  i++;
                  nCRLF[1]++;
                }
              else if (lpcFile[i] == cCRLF[1])
                {
                  i++;
                  nCRLF[2]++;
                }
              else
                {
                  i++;
                }
        }
      if (nCRLF[0] >= nCRLF[1] && nCRLF[0] >= nCRLF[2])
        ptw->uRetCode = 0;
      else if (nCRLF[1] >= nCRLF[2])
        ptw->uRetCode = 1;
      else
        ptw->uRetCode = 2;
    }

  /* ja:ǂݍ */
  i = 0;
  switch (ptw->uCharSet)
    {
      case 0:/* ja:UNICODE(gGfBA) */
        if (dwSize >= 2 && lpcFile[0] == '\xff' && lpcFile[1] == '\xfe')
          i = 2;
        dwSize &= ~1;
        break;
      case 1:/* ja:UNICODE(rbNGfBA) */
        if (dwSize >= 2 && lpcFile[0] == '\xfe' && lpcFile[1] == '\xff')
          i = 2;
        dwSize &= ~1;
    }
  EnterText = EnterCharSet[ptw->uCharSet][ptw->uRetCode];
  TranslateText = TranslateCharSet[ptw->uCharSet];
  while (i < dwSize)
    {
      MSG msg;

      while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
        if (!hDlgCancel || !IsDialogMessage (hDlgCancel, &msg))
          {
            TranslateMessage (&msg);
            DispatchMessage (&msg);
          }
      if (!fUserBreak)
        break;
      nLength = EnterText (lpcFile + i, dwSize - i);
      if (nLength > 0 && !TranslateText (p, lpcFile + i, nLength))
        {
          MessageBox (hWndClient, _T("TranslateText"),
                                    APPLICATION, MB_OK | MB_ICONEXCLAMATION);
          for (p = ptw->lpStart; p; p = q)
            {
              q = p->next;
              MemoryFree (p->lpszText);
              MemoryFree (p);
            }
          EnableWindow (hWndMain, TRUE);
          DestroyWindow (hDlgCancel);
          UnmapViewOfFile (lpcFile);
          CloseHandle (hMap);
          CloseHandle (hFile);
          return FALSE;
        }
      i += nLength;
      if (i + (ptw->uRetCode == 0 ? (ptw->uCharSet <= 1 ? 3 : 1) : 0) < dwSize)
        {
          q = MemoryAlloc (sizeof (LINEBUF));
          q->lpszText = NULL;
          q->nLength = 0;
          q->fMargin = FALSE;
          q->prev = p;
          q->next = p->next;
          p->next = q;
          p = q;
          ptw->nMax++;
          i += (ptw->uRetCode == 0 ? 2 : 1) * (ptw->uCharSet <= 1 ? 2 : 1);
        }
    }

  if (!UnmapViewOfFile (lpcFile))
    {
      MessageBox (hWndClient, _T("UnmapViewOfFile"),
                                APPLICATION, MB_OK | MB_ICONEXCLAMATION);
      for (p = ptw->lpStart; p; p = q)
        {
          q = p->next;
          MemoryFree (p->lpszText);
          MemoryFree (p);
        }
      EnableWindow (hWndMain, TRUE);
      DestroyWindow (hDlgCancel);
      CloseHandle (hMap);
      CloseHandle (hFile);
      return FALSE;
    }
  if (!CloseHandle (hMap) | !CloseHandle (hFile))
    {
      MessageBox (hWndClient, _T("CloseHandle"),
                                APPLICATION, MB_OK | MB_ICONEXCLAMATION);
      for (p = ptw->lpStart; p; p = q)
        {
          q = p->next;
          MemoryFree (p->lpszText);
          MemoryFree (p);
        }
      EnableWindow (hWndMain, TRUE);
      DestroyWindow (hDlgCancel);
      return FALSE;
    }
  /* ja:E}[WŐ܂Ԃ */
  if (!ModifyMargin (ptw))
    {
      for (p = ptw->lpStart; p; p = q)
        {
          q = p->next;
          MemoryFree (p->lpszText);
          MemoryFree (p);
        }
      EnableWindow (hWndMain, TRUE);
      DestroyWindow (hDlgCancel);
      return FALSE;
    }
  EnableWindow (hWndMain, TRUE);
  if (!DestroyWindow (hDlgCancel))
    {
      MessageBox (hWndClient, _T("DestroyWindow"),
                                APPLICATION, MB_OK | MB_ICONEXCLAMATION);
      for (p = ptw->lpStart; p; p = q)
        {
          q = p->next;
          MemoryFree (p->lpszText);
          MemoryFree (p);
        }
      return FALSE;
    }
  return TRUE;
}


/******************************************************************************
*                                                                             *
* ja:t@Co͊֐Q                                                       *
*                                                                             *
******************************************************************************/
/*  ja:UNICODE(gGfBA)ɕϊ
    hFile,t@Cnh
        p,Cobt@
      RET,TRUE:I,FALSE:G[                                        */
static BOOL
ConvertUnicodeLittle (HANDLE    hFile,
                      LPLINEBUF p)
{
  int nLength;
  BOOL fResult;
  DWORD dwWrite;
  LPWSTR lpwText;

  nLength = MultiByteToWideChar (CP_SJIS, 0, p->lpszText, p->nLength, NULL, 0);
  lpwText = MemoryAlloc (nLength * sizeof (WCHAR));
  nLength = MultiByteToWideChar (CP_SJIS, 0,
                                    p->lpszText, p->nLength, lpwText, nLength);
  fResult = WriteFile (hFile, lpwText,
                                    nLength * sizeof (WCHAR), &dwWrite, NULL)
                                        && nLength * sizeof (WCHAR) == dwWrite;
  MemoryFree (lpwText);
  return fResult;
}


/*  ja:UNICODE(rbNGfBA)ɕϊ
    hFile,t@Cnh
        p,Cobt@
      RET,TRUE:I,FALSE:G[                                        */
static BOOL
ConvertUnicodeBig (HANDLE    hFile,
                   LPLINEBUF p)
{
  int i, nLength;
  BOOL fResult;
  DWORD dwWrite;
  LPWSTR lpwText;

  nLength = MultiByteToWideChar (CP_SJIS, 0, p->lpszText, p->nLength, NULL, 0);
  lpwText = MemoryAlloc (nLength * sizeof (WCHAR));
  nLength = MultiByteToWideChar (CP_SJIS, 0,
                                    p->lpszText, p->nLength, lpwText, nLength);
  for (i = 0; i < nLength; i++)
    lpwText[i] = lpwText[i] >> 8 | lpwText[i] << 8;
  fResult = WriteFile (hFile, lpwText,
                                    nLength * sizeof (WCHAR), &dwWrite, NULL)
                                        && nLength * sizeof (WCHAR) == dwWrite;
  MemoryFree (lpwText);
  return fResult;
}


/*  ja:UTF-7ɕϊ
    hFile,t@Cnh
        p,Cobt@
      RET,TRUE:I,FALSE:G[                                        */
static BOOL
ConvertUtf7 (HANDLE    hFile,
             LPLINEBUF p)
{
  int i, j = 0, n, nLength;
  BOOL fMode, fResult;
  CHAR cBase64[64] = {'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H',
                      'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P',
                      'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X',
                      'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f',
                      'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
                      'o', 'p', 'q', 'r', 's', 't', 'u', 'v',
                      'w', 'x', 'y', 'z', '0', '1', '2', '3',
                      '4', '5', '6', '7', '8', '9', '+', '/'};
  DWORD dwWrite;
  LPSTR lpcText;
  LPWSTR lpwText;
  static int nBits;
  static BOOL fBase64;
  static DWORD dwBits;

  if (!p->prev || !p->prev->fMargin)
    fBase64 = FALSE;
  nLength = MultiByteToWideChar (CP_SJIS, 0, p->lpszText, p->nLength, NULL, 0);
  lpwText = MemoryAlloc (nLength * sizeof (WCHAR));
  nLength = MultiByteToWideChar (CP_SJIS, 0,
                                    p->lpszText, p->nLength, lpwText, nLength);
  fMode = fBase64;
  n = nBits;
  for (i = 0; i < nLength; i++)
    if (lpwText[i] == '\t' || lpwText[i] == cCRLF[0] || lpwText[i] == cCRLF[1]
                || lpwText[i] == ' ' || '\'' <= lpwText[i] && lpwText[i] <= ')'
                || '+' <= lpwText[i] && lpwText[i] <=':' || lpwText[i] == '?'
                || 'A' <= lpwText[i] && lpwText[i] <='Z'
                || 'a' <= lpwText[i] && lpwText[i] <='z')
      {
        if (fMode)
          {
            if (n > 0)
              j++;
            if (lpwText[i] == '+' || lpwText[i] == '-' || lpwText[i] == '/'
                                    || '0' <= lpwText[i] && lpwText[i] <= '9'
                                    || 'A' <= lpwText[i] && lpwText[i] <= 'Z'
                                    || 'a' <= lpwText[i] && lpwText[i] <= 'z')
              j++;
            fMode = FALSE;
          }
        j++;
        if (lpwText[i] == '+')
          j++;
      }
    else
      {
        if (!fMode)
          {
            j++;
            n = 0;
            fMode = TRUE;
          }
        n += 16;
        while (n >= 6)
          {
            j++;
            n -= 6;
          }
      }
  if (fMode && (!p->fMargin || !p->next))
    {
      if (n > 0)
        j++;
      if (!p->next)
        j++;
    }
  lpcText = MemoryAlloc (j * sizeof (CHAR));
  j = 0;
  for (i = 0; i < nLength; i++)
    if (lpwText[i] == '\t' || lpwText[i] == cCRLF[0] || lpwText[i] == cCRLF[1]
                || lpwText[i] == ' ' || '\'' <= lpwText[i] && lpwText[i] <= ')'
                || '+' <= lpwText[i] && lpwText[i] <= ':' || lpwText[i] == '?'
                || 'A' <= lpwText[i] && lpwText[i] <= 'Z'
                || 'a' <= lpwText[i] && lpwText[i] <= 'z')
      {
        if (fBase64)
          {
            if (nBits > 0)
              lpcText[j++] = cBase64[dwBits << 6 - nBits & 0x3f];
            if (lpwText[i] == '+' || lpwText[i] == '-' || lpwText[i] == '/'
                                    || '0' <= lpwText[i] && lpwText[i] <= '9'
                                    || 'A' <= lpwText[i] && lpwText[i] <= 'Z'
                                    || 'a' <= lpwText[i] && lpwText[i] <= 'z')
              lpcText[j++] = '-';
            fBase64 = FALSE;
          }
        lpcText[j++] = lpwText[i];
        if (lpwText[i] == '+')
          lpcText[j++] = '-';
      }
    else
      {
        if (!fBase64)
          {
            lpcText[j++] = '+';
            nBits = 0;
            fBase64 = TRUE;
          }
        dwBits = dwBits << 16 | lpwText[i];
        nBits += 16;
        while (nBits >= 6)
          {
            nBits -= 6;
            lpcText[j++] = cBase64[dwBits >> nBits & 0x3f];
          }
      }
  if (fBase64 && (!p->fMargin || !p->next))
    {
      if (nBits > 0)
        lpcText[j++] = cBase64[dwBits << 6 - nBits & 0x3f];
      if (!p->next)
        lpcText[j++] = '-';
    }
  MemoryFree (lpwText);
  fResult = WriteFile (hFile, lpcText, j * sizeof (CHAR), &dwWrite, NULL)
                                            && j * sizeof (CHAR) == dwWrite;
  MemoryFree (lpcText);
  return fResult;
}


/*  ja:UTF-8ɕϊ
    hFile,t@Cnh
        p,Cobt@
      RET,TRUE:I,FALSE:G[                                        */
static BOOL
ConvertUtf8 (HANDLE    hFile,
             LPLINEBUF p)
{
  int i, j = 0, nLength;
  BOOL fResult;
  DWORD dwWrite;
  LPSTR lpcText;
  LPWSTR lpwText;

  nLength = MultiByteToWideChar (CP_SJIS, 0, p->lpszText, p->nLength, NULL, 0);
  lpwText = MemoryAlloc (nLength * sizeof (WCHAR));
  nLength = MultiByteToWideChar (CP_SJIS, 0,
                                    p->lpszText, p->nLength, lpwText, nLength);
  for (i = 0; i < nLength; i++)
    if (lpwText[i] <= 0x007f)
      j++;
    else if (lpwText[i] <= 0x07ff)
      j += 2;
    else
      j += 3;
  lpcText = MemoryAlloc (j * sizeof (CHAR));
  j = 0;
  for (i = 0; i < nLength; i++)
    if (lpwText[i] <= 0x007f)
      {
        lpcText[j++] = lpwText[i];
      }
    else if (lpwText[i] <= 0x07ff)
      {
        lpcText[j++] = 0xc0 | lpwText[i] >> 6;
        lpcText[j++] = 0x80 | lpwText[i] & 0x3f;
      }
    else
      {
        lpcText[j++] = 0xe0 | lpwText[i] >> 12;
        lpcText[j++] = 0x80 | lpwText[i] >> 6 & 0x3f;
        lpcText[j++] = 0x80 | lpwText[i] & 0x3f;
      }
  MemoryFree (lpwText);
  fResult = WriteFile (hFile, lpcText, j * sizeof (CHAR), &dwWrite, NULL)
                                            && j * sizeof (CHAR) == dwWrite;
  MemoryFree (lpcText);
  return fResult;
}


/*  ja:ISO-2022-JPɕϊ
    hFile,t@Cnh
        p,Cobt@
      RET,TRUE:I,FALSE:G[                                        */
static BOOL
ConvertIso2022Jp (HANDLE    hFile,
                  LPLINEBUF p)
{
  int i = 0, j = 0;
  BOOL fMode, fResult;
  CHAR cKI[3] = {'\x1b', '$', 'B'}, cKO[3] = {'\x1b', '(', 'B'};
  DWORD dwWrite;
  LPSTR lpcText;
  static BOOL fKanji;

  if (!p->prev || !p->prev->fMargin)
    fKanji = FALSE;
  fMode = fKanji;
  while (i < p->nLength)
    if (i + 1 < p->nLength && (0x81 <= (BYTE)p->lpszText[i]
                                                && (BYTE)p->lpszText[i] <= 0x9f
                            || 0xe0 <= (BYTE)p->lpszText[i]
                                            && (BYTE)p->lpszText[i] <= 0xfc))
      {
        if (!fMode)
          {
            j += 3;
            fMode = TRUE;
          }
        i += 2;
        j += 2;
      }
    else
      {
        if (fMode)
          {
            j += 3;
            fMode = FALSE;
          }
        i++;
        j++;
      }
  if (fMode && (!p->fMargin || !p->next))
    j += 3;
  lpcText = MemoryAlloc (j * sizeof (CHAR) + 10000);
  i = j = 0;
  while (i < p->nLength)
    if (i + 1 < p->nLength && (0x81 <= (BYTE)p->lpszText[i]
                                                && (BYTE)p->lpszText[i] <= 0x9f
                            || 0xe0 <= (BYTE)p->lpszText[i]
                                            && (BYTE)p->lpszText[i] <= 0xfc))
      {
        if (!fKanji)
          {
            MemoryCopy (lpcText + j, cKI, sizeof (CHAR) * 3);
            j += 3;
            fKanji = TRUE;
          }
        lpcText[j] = p->lpszText[i++];
        lpcText[j + 1] = p->lpszText[i++];
        lpcText[j] -= (BYTE)lpcText[j] >= 0xe0 ? 0xc1 : 0x81;
        lpcText[j] <<= 1;
        if ((BYTE)lpcText[j + 1] >= 0x9f)
          {
            lpcText[j] += 0x22;
            lpcText[j + 1] -= 0x7e;
          }
        else
          {
            lpcText[j] += 0x21;
            lpcText[j + 1] -= (BYTE)lpcText[j + 1] >= 0x80 ? 0x20 : 0x1f;
          }
        j += 2;
      }
    else
      {
        if (fKanji)
          {
            MemoryCopy (lpcText + j, cKO, sizeof (CHAR) * 3);
            j += 3;
            fKanji = FALSE;
          }
        lpcText[j++] = p->lpszText[i++];
      }
  if (fKanji && (!p->fMargin || !p->next))
    {
      MemoryCopy (lpcText + j, cKO, sizeof (CHAR) * 3);
      j += 3;
    }
  fResult = WriteFile (hFile, lpcText, j * sizeof (CHAR), &dwWrite, NULL)
                                            && j * sizeof (CHAR) == dwWrite;
  MemoryFree (lpcText);
  return fResult;
}


/*  ja:EUC-JPɕϊ
    hFile,t@Cnh
        p,Cobt@
      RET,TRUE:I,FALSE:G[                                        */
static BOOL
ConvertEucJp (HANDLE    hFile,
              LPLINEBUF p)
{
  int i = 0, j = 0;
  BOOL fResult;
  DWORD dwWrite;
  LPSTR lpcText;

  while (i < p->nLength)
    if (i + 1 < p->nLength && (0x81 <= (BYTE)p->lpszText[i]
                                            && (BYTE)p->lpszText[i] <= 0x9f
                            || 0xe0 <= (BYTE)p->lpszText[i]
                                            && (BYTE)p->lpszText[i] <= 0xfc))
      {
        i += 2;
        j += 2;
      }
    else if (0xa1 <= (BYTE)p->lpszText[i] && (BYTE)p->lpszText[i] <= 0xfe)
      {
        i++;
        j += 2;
      }
    else
      {
        i++;
        j++;
      }
  lpcText = MemoryAlloc (j * sizeof (CHAR));
  i = j = 0;
  while (i < p->nLength)
    if (i + 1 < p->nLength && (0x81 <= (BYTE)p->lpszText[i]
                                            && (BYTE)p->lpszText[i] <= 0x9f
                            || 0xe0 <= (BYTE)p->lpszText[i]
                                            && (BYTE)p->lpszText[i] <= 0xfc))
      {
        lpcText[j] = p->lpszText[i++];
        lpcText[j + 1] = p->lpszText[i++];
        lpcText[j] -= (BYTE)lpcText[j] >= 0xe0 ? 0xc1 : 0x81;
        lpcText[j] <<= 1;
        if ((BYTE)lpcText[j + 1] >= 0x9f)
          {
            lpcText[j] += 0x22;
            lpcText[j + 1] -= 0x7e;
          }
        else
          {
            lpcText[j] += 0x21;
            lpcText[j + 1] -= (BYTE)lpcText[j + 1] >= 0x80 ? 0x20 : 0x1f;
          }
        lpcText[j] |= 0x80;
        lpcText[j + 1] |= 0x80;
        j += 2;
      }
    else
      {
        if (0xa1 <= (BYTE)p->lpszText[i] && (BYTE)p->lpszText[i] <= 0xfe)
          lpcText[j++] = 0x8e;
          lpcText[j++] = p->lpszText[i++];
      }
  fResult = WriteFile (hFile, lpcText, j * sizeof (CHAR), &dwWrite, NULL)
                                            && j * sizeof (CHAR) == dwWrite;
  MemoryFree (lpcText);
  return fResult;
}


/*  ja:SHIFT_JISɕϊ
    hFile,t@Cnh
        p,Cobt@
      RET,TRUE:I,FALSE:G[                                        */
static BOOL
ConvertShiftJis (HANDLE    hFile,
                 LPLINEBUF p)
{
  DWORD dwWrite;

  return WriteFile (hFile, p->lpszText,
                                    p->nLength * sizeof (CHAR), &dwWrite, NULL)
                                    && p->nLength * sizeof (CHAR) == dwWrite;
}


/*  ja:TXTt@Cۑ
    szFile,t@C
       ptw,TXTEChE
       RET,TRUE:I,FALSE:G[                                       */
BOOL
SaveTextFile (LPCTSTR   lpszFile,
              LPTEXTWND ptw)
{
  BOOL fEOF = FALSE;
  CHAR cText[4];
  DWORD dwSize, dwWrite;
  HANDLE hFile;
  LPTSTR lpszText;
  LPLINEBUF p;
  BOOL (*ConvertCharSet[7])(HANDLE, LPLINEBUF) = {
                ConvertUnicodeLittle, ConvertUnicodeBig, ConvertUtf7,
                ConvertUtf8, ConvertIso2022Jp, ConvertEucJp, ConvertShiftJis};
  BOOL (*ConvertText)(HANDLE, LPLINEBUF);

  if (ptw->fRecycle && PathFileExists (lpszFile))
    {
      /* ja:ɂt@CݔɈړ */
      TCHAR szDelete[MAX_PATH + 1];
      SHFILEOPSTRUCT shfo;

      lstrcpy (szDelete, lpszFile);
      szDelete[lstrlen (szDelete) + 1] = '\0';
      MemorySet (&shfo, 0, sizeof (SHFILEOPSTRUCT));
      shfo.hwnd = hWndClient;
      shfo.wFunc = FO_DELETE;
      shfo.pFrom = szDelete;
      shfo.fFlags = FOF_ALLOWUNDO | FOF_NOCONFIRMATION;
      SHFileOperation (&shfo);
    }
  p = ptw->lpStart;
  while (p->prev)
    p = p->prev;
  /* ja:J */
  hFile = CreateFile (lpszFile, GENERIC_WRITE, FILE_SHARE_READ, NULL,
                                CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
  if (hFile == INVALID_HANDLE_VALUE)
    {
      lpszText = LoadText (hInst, IDS_FILE_OPEN);
      MessageBox (hWndClient, lpszText,
                                APPLICATION, MB_OK | MB_ICONEXCLAMATION);
      MemoryFree (lpszText);
      return FALSE;
    }
  /* ja:_CAO{bNX */
  fUserBreak = TRUE;
  EnableWindow (hWndMain, FALSE);
  lpszText = LoadText (hInst, IDS_JOB_WRITE);
  hDlgCancel = CreateDialogParamGUI (hInst, MAKEINTRESOURCE (DIALOG_G),
                                hWndClient, AbortDlgProc, (LPARAM)lpszText);
  MemoryFree (lpszText);
  if (!hDlgCancel)
    {
      MessageBox (hWndClient, _T("CreateDialogGUIParam"),
                                APPLICATION, MB_OK | MB_ICONEXCLAMATION);
      EnableWindow (hWndMain, TRUE);
      CloseHandle (hFile);
      return FALSE;
    }
  /* ja: */
  ConvertText = ConvertCharSet[ptw->uCharSet];
  switch (ptw->uCharSet)
    {
      case 0:/* ja:UNICODE(gGfBA) */
        cText[0] = '\xff';
        cText[1] = '\xfe';
        if (!WriteFile (hFile, cText, sizeof (CHAR) * 2, &dwWrite, NULL)
                                            || dwWrite != sizeof (CHAR) * 2)
          {
            lpszText = LoadText (hInst, IDS_FILE_WRITE);
            MessageBox (hWndClient, lpszText,
                                    APPLICATION, MB_OK | MB_ICONEXCLAMATION);
            MemoryFree (lpszText);
            EnableWindow (hWndMain, TRUE);
            DestroyWindow (hDlgCancel);
            CloseHandle (hFile);
            return FALSE;
          }
        switch (ptw->uRetCode)
          {
            case 1: dwSize = 2; cText[0] = cCRLF[0]; cText[1] = '\0'; break;
            case 2: dwSize = 2; cText[0] = cCRLF[1]; cText[1] = '\0'; break;
            default:
              dwSize = 4;
              cText[0] = cCRLF[0];
              cText[1] = '\0';
              cText[2] = cCRLF[1];
              cText[3] = '\0';
          }
        break;
      case 1:/* ja:UNICODE(rbNGfBA) */
        cText[0] = '\xfe';
        cText[1] = '\xff';
        if (!WriteFile (hFile, cText, sizeof (CHAR) * 2, &dwWrite, NULL)
                                            || dwWrite != sizeof (CHAR) * 2)
          {
            lpszText = LoadText (hInst, IDS_FILE_WRITE);
            MessageBox (hWndClient, lpszText,
                                    APPLICATION, MB_OK | MB_ICONEXCLAMATION);
            MemoryFree (lpszText);
            EnableWindow (hWndMain, TRUE);
            DestroyWindow (hDlgCancel);
            CloseHandle (hFile);
            return FALSE;
          }
        switch (ptw->uRetCode)
          {
            case 1: dwSize = 2; cText[0] = '\0'; cText[1] = cCRLF[0]; break;
            case 2: dwSize = 2; cText[0] = '\0'; cText[1] = cCRLF[1]; break;
            default:
              dwSize = 4;
              cText[0] = '\0';
              cText[1] = cCRLF[0];
              cText[2] = '\0';
              cText[3] = cCRLF[1];
          }
        break;
      default:/* en:ISO-2022-JP,UTF-7,UTF-8,EUC-JP,SHIFT_JIS */
        switch (ptw->uRetCode)
          {
            case 1:  dwSize = 1; cText[0] = cCRLF[0]; break;
            case 2:  dwSize = 1; cText[0] = cCRLF[1]; break;
            default: dwSize = 2; cText[0] = cCRLF[0]; cText[1] = cCRLF[1];
          }
    }
  while (p)
    {
      MSG msg;

      while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
        if (!hDlgCancel || !IsDialogMessage (hDlgCancel, &msg))
          {
            TranslateMessage (&msg);
            DispatchMessage (&msg);
          }
      if (!fUserBreak)
        break;
      if (p->nLength > 0 && !ConvertText (hFile, p))
        {
          lpszText = LoadText (hInst, IDS_FILE_WRITE);
          MessageBox (hWndClient, lpszText,
                                    APPLICATION, MB_OK | MB_ICONEXCLAMATION);
          MemoryFree (lpszText);
          EnableWindow (hWndMain, TRUE);
          DestroyWindow (hDlgCancel);
          CloseHandle (hFile);
          return FALSE;
        }
      if (!p->fMargin && p->next)
        {
          if (!WriteFile (hFile, cText, dwSize, &dwWrite, NULL)
                                                        || dwSize != dwWrite)
            {
              lpszText = LoadText (hInst, IDS_FILE_WRITE);
              MessageBox (hWndClient, lpszText,
                                    APPLICATION, MB_OK | MB_ICONEXCLAMATION);
              MemoryFree (lpszText);
              EnableWindow (hWndMain, TRUE);
              DestroyWindow (hDlgCancel);
              CloseHandle (hFile);
              return FALSE;
            }
          fEOF = FALSE;
        }
      else if (p->nLength > 0)
        {
          fEOF = p->lpszText[p->nLength - 1] == '\x1a';
        }
      p = p->next;
    }
  /* ja:Ō^Zt */
  if (fUserBreak && !fEOF && ptw->fEOF)
    {
      switch (ptw->uCharSet)
        {
          case 0:  dwSize = 2; cText[0] = '\x1a'; cText[1] = '\0';   break;
          case 1:  dwSize = 2; cText[0] = '\0';   cText[1] = '\x1a'; break;
          default: dwSize = 1; cText[0] = '\x1a';
        }
      if (!WriteFile (hFile, cText, dwSize, &dwWrite, NULL)
                                                        || dwSize != dwWrite)
        {
          lpszText = LoadText (hInst, IDS_FILE_WRITE);
          MessageBox (hWndClient, lpszText,
                                    APPLICATION, MB_OK | MB_ICONEXCLAMATION);
          MemoryFree (lpszText);
          EnableWindow (hWndMain, TRUE);
          DestroyWindow (hDlgCancel);
          CloseHandle (hFile);
          return FALSE;
        }
    }
  /* ja:I */
  if (!CloseHandle (hFile))
    {
      MessageBox (hWndClient, _T("CloseHandle"),
                        APPLICATION, MB_OK | MB_ICONEXCLAMATION);
      EnableWindow (hWndMain, TRUE);
      DestroyWindow (hDlgCancel);
      return FALSE;
    }
  EnableWindow (hWndMain, TRUE);
  if (!DestroyWindow (hDlgCancel))
    {
      MessageBox (hWndClient, _T("DestroyWindow"),
                                APPLICATION, MB_OK | MB_ICONEXCLAMATION);
      return FALSE;
    }
  return fUserBreak;
}


/******************************************************************************
*                                                                             *
* ja:t@C֐Q                                                           *
*                                                                             *
******************************************************************************/
/*  ja:
    0:pX̃t@C̐𐔂
    1:̃t@C̐𐔂
    2:pX̃t@CtpX+\ɕύX
    3:̃t@CtpX\ɕύX
    4:pX̃t@CtpX\ɕύX
    5:̃t@Cʏ\ɕύX                                    */
static BOOL
CALLBACK SameFileEnumProc (HWND   hWnd,
                           LPARAM lParam)
{
  TCHAR szText[MAX_PATH + 2];

  GetClassName (hWnd, szText, MAX_PATH - 1);
  if (((LPSAMEFILE)lParam)->hWnd != hWnd && lstrcmp (szText, TEXTCLASS) == 0)
    {
      LPTEXTWND ptw;

      ptw = (LPTEXTWND)GetWindowLong (hWnd, 0);
      switch (((LPSAMEFILE)lParam)->nCommand)
        {
          case 0:/* ja:pX̃t@C̐𐔂 */
            if (lstrcmpi (((LPSAMEFILE)lParam)->szFile, ptw->szFile) == 0)
              {
                if (((LPSAMEFILE)lParam)->nSame < ptw->nSame)
                  ((LPSAMEFILE)lParam)->nSame = ptw->nSame;
                ((LPSAMEFILE)lParam)->nCount++;
              }
            break;
          case 1:/* ja:̃t@C̐𐔂 */
            if (lstrcmpi (PathFindFileName (((LPSAMEFILE)lParam)->szFile),
                          PathFindFileName (ptw->szFile)) == 0)
              ((LPSAMEFILE)lParam)->nCount++;
            break;
          case 2:/* ja:pX̃t@CtpX+\ɕύX */
            if (lstrcmpi (((LPSAMEFILE)lParam)->szFile, ptw->szFile) == 0)
              {
                if (ptw->nSame < 0)
                  {
                    ptw->nSame = 0;
                    lstrcpy (szText, ptw->szFile);
                    lstrcat (szText, _T(":0"));
                    if (!SetWindowText (hWnd, szText))
                      MessageBox (hWnd, _T("SetWindowText"),
                                    APPLICATION, MB_OK | MB_ICONEXCLAMATION);
                  }
                if (((LPSAMEFILE)lParam)->nSame < ptw->nSame)
                  ((LPSAMEFILE)lParam)->nSame = ptw->nSame;
                ((LPSAMEFILE)lParam)->nCount++;
              }
            break;
          case 3:/* ja:̃t@CtpX\ɕύX */
            if (lstrcmpi (PathFindFileName (((LPSAMEFILE)lParam)->szFile),
                          PathFindFileName (ptw->szFile)) == 0)
              {
                ((LPSAMEFILE)lParam)->nCount++;
                if (ptw->nSame < 0 && !SetWindowText (hWnd, ptw->szFile))
                  MessageBox (hWnd, _T("SetWindowText"),
                                    APPLICATION, MB_OK | MB_ICONEXCLAMATION);
              }
            break;
          case 4:/* ja:pX̃t@CtpX\ɕύX */
            if (lstrcmpi (((LPSAMEFILE)lParam)->szFile, ptw->szFile) == 0)
              {
                ptw->nSame = -1;
                ((LPSAMEFILE)lParam)->nCount++;
                if (!SetWindowText (hWnd, ptw->szFile))
                  MessageBox (hWnd, _T("SetWindowText"),
                                    APPLICATION, MB_OK | MB_ICONEXCLAMATION);
              }
            break;
          case 5:/* ja:̃t@Cʏ\ɕύX */
            {
              LPTSTR lpszName;

              lpszName = PathFindFileName (ptw->szFile);
              if (lstrcmpi (PathFindFileName (((LPSAMEFILE)lParam)->szFile),
                                                                lpszName) == 0)
                {
                  ptw->nSame = -1;
                  ((LPSAMEFILE)lParam)->nCount++;
                  if (!SetWindowText (hWnd, lpszName))
                    MessageBox (hWnd, _T("SetWindowText"),
                                    APPLICATION, MB_OK | MB_ICONEXCLAMATION);
                }
            }
        }
    }
  return TRUE;
}


/*  ja:t@C
         hWnd,悤ƂEChE
     lpszFile,悤Ƃt@C̃tpX
      lpnSame,tpXɕt鐔l
    lpszTitle,悤Ƃt@C̃x                                */
VOID
AddEditFile (HWND     hWnd,
             LPCTSTR  lpszFile,
             int     *lpnSame,
             LPTSTR   lpszTitle)
{
  SAMEFILE samefile;

  *lpnSame = -1;
  lstrcpy (samefile.szFile, lpszFile);
  samefile.nCommand = 2;
  samefile.nCount = 0;
  samefile.nSame = -1;
  samefile.hWnd = hWnd;
  EnumChildWindows (hWndClient, SameFileEnumProc, (LPARAM)&samefile);
  if (samefile.nCount > 0)
    {
      /* ja:pXƂ */
      *lpnSame = samefile.nSame + 1;
      wsprintf (lpszTitle, _T("%s:%d"), lpszFile, *lpnSame);
    }
  else
    {
      /* ja:pX͂ȂƂ */
      samefile.nCommand = 3;
      EnumChildWindows (hWndClient, SameFileEnumProc, (LPARAM)&samefile);
      /* ja:كpXƂ or ʏ */
      lstrcpy (lpszTitle, samefile.nCount > 0 ? lpszFile
                                              : PathFindFileName (lpszFile));
    }
}


/*  ja:t@C폜
        hWnd,폜悤ƂEChE
    lpszFile,폜悤Ƃt@C̃tpX                             */
VOID
DeleteEditFile (HWND    hWnd,
                LPCTSTR lpszFile)
{
  SAMEFILE samefile;

  /* ja:كpX𒲂ׂ */
  lstrcpy (samefile.szFile, lpszFile);
  samefile.nCommand = 1;
  samefile.nCount = 0;
  samefile.hWnd = hWnd;
  EnumChildWindows (hWndClient, SameFileEnumProc, (LPARAM)&samefile);
  if (samefile.nCount == 1)
    {
      /* ja:كpX̃t@Cʏ\ɕύX */
      samefile.nCommand = 5;
      EnumChildWindows (hWndClient, SameFileEnumProc, (LPARAM)&samefile);
    }
  else
    {
      /* ja:pX𒲂ׂ */
      samefile.nCommand = samefile.nCount = 0;
      EnumChildWindows (hWndClient, SameFileEnumProc, (LPARAM)&samefile);
      if (samefile.nCount == 1)
        {
          /* ja:pX̃t@CtpX\ɕύX */
          samefile.nCommand = 4;
          EnumChildWindows (hWndClient, SameFileEnumProc, (LPARAM)&samefile);
        }
    }
}


/*  ja:t@CJ
    lpszFile,t@C
         RET,EChE,NULL:G[                                         */
HWND
OpenEditFile (LPCTSTR lpszFile)
{
  HWND hWnd;
  MDICREATESTRUCT mcs;
  LPTSTR lpszFormat;
  TCHAR szTitle[MAX_PATH + 12];
  LPTEXTWND ptw;

  ptw = MemoryAlloc (sizeof (TEXTWND));
  ptw->ptSelect.x = -1;
  if (!lpszFile)
    {
      /* ja:VK */
      int i;
      TCHAR **plpszArray;
      LPTSTR lpszExt;

      plpszArray = StringSplitDelimiter (lpFileType[0].szExt, _T(";"));
      lpszExt = plpszArray ? PathFindExtension (plpszArray[0]) : _T("");
      lpszFormat = LoadText (hInst, IDS_OTHER_CREATE);
      wsprintf (ptw->szFile, lpszFormat, nFileCount++ % 10000, lpszExt);
      MemoryFree (plpszArray);
      MemoryFree (lpszFormat);
      ptw->fCreate = TRUE;
      ptw->nID = lpFileType[0].fAssociate ? lpFileType[0].nID : -1;
      ptw->nMargin = lpFileType[0].nMargin;
      ptw->nTab = lpFileType[0].nTab;
      ptw->fAutoIndent = lpFileType[0].fAutoIndent;
      ptw->fCode = lpFileType[0].fCode;
      ptw->fCRLF = lpFileType[0].fCRLF;
      ptw->fEOF = lpFileType[0].fEOF;
      ptw->fLimit = lpFileType[0].fLimit;
      ptw->fOverWrite = lpFileType[0].fOverWrite;
      ptw->fRecycle = lpFileType[0].fRecycle;
      ptw->fSpace = lpFileType[0].fSpace;
      ptw->fSysColor = lpFileType[0].fSysColor;
      ptw->fTabConv = lpFileType[0].fTabConv;
      ptw->fGline = lpFileType[0].fGline;
      ptw->fMline = lpFileType[0].fMline;
      ptw->fUline = lpFileType[0].fUline;
      ptw->fVline = lpFileType[0].fVline;
      ptw->uRetCode = lpFileType[0].uRetCode;
      ptw->uCharSet = (lpFileType[0].dwCharSet & 15) - 1;
      MemoryCopy (ptw->crColor, lpFileType[0].crColor, sizeof (COLORREF) * 12);
      ptw->lf = lpFileType[0].lf;
    }
  else
    {
      int i;

      GetLongFile (ptw->szFile, lpszFile);
      for (i = 0; i < nFileType; i++)
        {
          TCHAR **plpszArray;

          plpszArray = StringSplitDelimiter (lpFileType[i].szExt, _T(";"));
          if (plpszArray)
            {
              int j;
              BOOL fMatch = FALSE;

              for (j = 0; plpszArray[j]; j++)
                if (PathMatchSpec (ptw->szFile, plpszArray[j]))
                  {
                    fMatch = TRUE;
                    break;
                  }
              MemoryFree (plpszArray);
              if (fMatch)
                break;
            }
        }
      if (i >= nFileType)
        for (i = 0; i < nFileType; i++)
          if (lstrcmp (lpFileType[i].szExt, _T("*")) == 0)
            break;
      if (i >= nFileType)
        i = 0;
      ptw->nID = lpFileType[i].fAssociate ? lpFileType[i].nID : -1;
      ptw->nMargin = lpFileType[i].nMargin;
      ptw->nTab = lpFileType[i].nTab;
      ptw->fAutoIndent = lpFileType[i].fAutoIndent;
      ptw->fCode = lpFileType[i].fCode;
      ptw->fCRLF = lpFileType[i].fCRLF;
      ptw->fEOF = lpFileType[i].fEOF;
      ptw->fLimit = lpFileType[i].fLimit;
      ptw->fOverWrite = lpFileType[i].fOverWrite;
      ptw->fRecycle = lpFileType[i].fRecycle;
      ptw->fSpace = lpFileType[i].fSpace;
      ptw->fSysColor = lpFileType[i].fSysColor;
      ptw->fTabConv = lpFileType[i].fTabConv;
      ptw->fGline = lpFileType[i].fGline;
      ptw->fMline = lpFileType[i].fMline;
      ptw->fUline = lpFileType[i].fUline;
      ptw->fVline = lpFileType[i].fVline;
      MemoryCopy (ptw->crColor, lpFileType[i].crColor, sizeof (COLORREF) * 12);
      ptw->lf = lpFileType[i].lf;
      if (PathFileExists (ptw->szFile) && !OpenTextFile (ptw,
                                                    lpFileType[i].dwCharSet,
                                                    lpFileType[i].fNegotiate,
                                                    lpFileType[i].uRetCode))
        {
          MemoryFree (ptw);
          return NULL;
        }
    }
  if (!ptw->lpStart)
    {
      /* ja:t@C0oCĝƂ */
      ptw->nOff = 0;
      ptw->nMax = 1;
      ptw->lpStart = MemoryAlloc (sizeof (LINEBUF));
    }
  AddEditFile (NULL, ptw->szFile, &ptw->nSame, szTitle);
  mcs.szClass = TEXTCLASS;
  mcs.szTitle = szTitle;
  mcs.hOwner = hInst;
  mcs.x = mcs.y = mcs.cx = mcs.cy = CW_USEDEFAULT;
  mcs.style = WS_HSCROLL | WS_VSCROLL;
  mcs.lParam = (LPARAM)ptw;
  hWnd = (HWND)SendMessage (hWndClient, WM_MDIGETACTIVE, 0, 0);
  if (IsWindow (hWnd))
    {
      if (IsZoomed (hWnd))
        mcs.style |= WS_MAXIMIZE;
    }
  else if (fZoomed)
    {
      mcs.style |= WS_MAXIMIZE;
    }
  hWnd = (HWND)SendMessage (hWndClient, WM_MDICREATE, 0, (LPARAM)&mcs);
  /* ja:VK쐬ł͂ȂAt@C݂鎞ɂ͗ɉ */
  if (lpszFile && PathFileExists (lpszFile)
        && !SetHistory (ptw->szFile, hWndMain, nHistory, MENUTOP, MENUFILE))
    {
      MessageBox (hWndClient, _T("SetHistory"),
                                APPLICATION, MB_OK | MB_ICONSTOP);
      DestroyWindow (hWndMain);
      return NULL;
    }
  return hWnd;
}
