package pantheon.web;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.util.Assert;
import org.springframework.validation.BindException;
import org.springframework.validation.FieldError;
import org.springframework.validation.ObjectError;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.mvc.SimpleFormController;
import org.springframework.web.servlet.view.RedirectView;

import pantheon.ItemService;
import pantheon.PeopleService;
import pantheon.model.Book;
import pantheon.model.Item;
import pantheon.model.ItemProperty;
import pantheon.model.ItemType;
import pantheon.model.Label;
import pantheon.user.LoginUser;
import pantheon.user.UserUtility;

public class UpdateItemController extends SimpleFormController {
	/** Logger for this class and subclasses */
	protected final Log logger = LogFactory.getLog(getClass());
	private ItemType itemType;
	private ItemService itemService;
	private PeopleService peopleService;

	public UpdateItemController() {
		super();
		setCommandName("ItemCommand");
		setCommandClass(ItemCommand.class);
	}

	@Override
	protected ModelAndView showForm(HttpServletRequest request, HttpServletResponse response, BindException errors) throws Exception {
		if (errors.hasErrors()) {
			return super.showForm(request, response, errors);
		}

		String sItemId = request.getParameter("itemId");
		if (!StringUtils.isNumeric(sItemId)) {
			return new ModelAndView("/403.html");
		}
		long itemId = Long.valueOf(sItemId);

		Item item = itemService.findByIdWithProperties(itemId);
		Assert.notNull(item);

		Map<String, Object> myModel = referenceData(request);

		ItemCommand itemCommand = new ItemCommand();
		itemCommand.setId(item.getId());
		itemCommand.setItemType(item.getType());
		itemCommand.setName(item.getName());
		itemCommand.setKanaName(item.getKanaName());
		itemCommand.setShortName(item.getShortName());
		itemCommand.setWikiName(item.getWikiName());
		itemCommand.setEnglishName(item.getEnglishName());
		if (item.getType().isHasFromYear()) {
			itemCommand.setFromYear(String.valueOf(item.getFromYear()));
		}
		if (item.getType().isHasToYear()) {
			itemCommand.setToYear(String.valueOf(item.getToYear()));
		}
		itemCommand.setVersion(item.getVersion());

		//人物に関連する書籍情報
		if (item.getType() == ItemType.PEOPLE) {
			List<Item> bookList = itemService.findAllTargetItem(item.getId(), Label.WRITTEN);
			List<Long> bookIds = new ArrayList<Long>();
			for (Item targetItem : bookList) {
				bookIds.add(targetItem.getId());
			}
			itemCommand.setPropertyBookIds(bookIds.toArray(new Long[bookIds.size()]));
		}

		//時点プロパティ情報
		List<String> fromYearList = new ArrayList<String>();
		List<String> textList = new ArrayList<String>();
		for (ItemProperty property : item.getPointPropertyList()) {
			fromYearList.add(String.valueOf(property.getFromYear()));
			textList.add(property.getText());
		}
		itemCommand.setPropertyFromYear(fromYearList.toArray(new String[CreateItemController.PORPERTY_SIZE]));
		itemCommand.setPropertyText(textList.toArray(new String[CreateItemController.PORPERTY_SIZE]));

		myModel.put("ItemCommand", itemCommand);
		myModel.put("itemType", itemType);

		return new ModelAndView(getFormView(), myModel);
	}

	@Override
	protected boolean suppressValidation(HttpServletRequest request, Object command) {
		ItemCommand itemCommand = (ItemCommand) command;
		itemCommand.setIsInsert(Boolean.FALSE);
		itemCommand.setItemType(itemType);

		return super.suppressValidation(request, command);
	}

	@Override
	protected ModelAndView onSubmit(HttpServletRequest request, HttpServletResponse response, Object command, BindException errors) throws Exception {

		ItemCommand itemCommand = (ItemCommand) command;
		long itemId = itemCommand.getId();

		// 名前重複チェック
		Item sameNameItem = itemService.findByName(itemCommand.getName());
		if (sameNameItem != null && sameNameItem.getId() != itemCommand.getId()) {
			ObjectError error = new FieldError("ItemCommand", "name", ValidatorError.editSameNameErrMessage(itemCommand.getName()));
			errors.addError(error);
			return super.showForm(request, response, errors);
		}

		LoginUser user = UserUtility.getLoginUser(request);

		Item item = itemService.findById(itemId);
		if (item.getVersion() != itemCommand.getVersion()) {
			ObjectError error = new FieldError("ItemCommand", "name", ValidatorError.editUnmuchVersionErrMessage(item.getVersion(), itemCommand
					.getVersion()));
			errors.addError(error);
			return super.showForm(request, response, errors);
		}
		item.setName(itemCommand.getName());
		item.setKanaName(itemCommand.getKanaName());
		item.setShortName(itemCommand.getShortName());
		item.setWikiName(itemCommand.getWikiName());
		item.setEnglishName(itemCommand.getEnglishName());
		if (item.getType().isHasFromYear()) {
			item.setFromYear(Integer.parseInt(itemCommand.getFromYear()));
		}
		if (item.getType().isHasToYear()) {
			item.setToYear(Integer.parseInt(itemCommand.getToYear()));
		}
		item.setPropertyList(itemCommand.getPointPropertyList());

		if (item.getType() == ItemType.PEOPLE) {
			peopleService.update(item, itemCommand.getPropertyBookIds(), user.getId());
		} else {
			itemService.update(item, user.getId());
		}

		return new ModelAndView(new RedirectView(getSuccessView()));
	}

	@Override
	protected Map<String, Object> referenceData(HttpServletRequest request) throws Exception {
		Map<String, Object> myModel = new HashMap<String, Object>();
		myModel.put("itemType", this.itemType);

		if (this.itemType == ItemType.PEOPLE) {
			List<Book> bookList = itemService.findAllBook();
			myModel.put("bookList", bookList);
		}

		return myModel;
	}

	@Required
	public void setItemType(String itemType) {
		this.itemType = ItemType.valueOf(itemType);
	}

	@Required
	public void setItemService(ItemService itemService) {
		this.itemService = itemService;
	}

	@Required
	public void setPeopleService(PeopleService peopleService) {
		this.peopleService = peopleService;
	}
}
