package pantheon.web;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.springframework.util.Assert;
import org.springframework.validation.Errors;

import pantheon.model.ItemProperty;
import pantheon.model.ItemType;
import pantheon.model.People;
import pantheon.model.meta.PantheonStringUtil;

public class ItemValidator extends AbstractPantheonValidator {

	@SuppressWarnings("unchecked")
	public boolean supports(Class clazz) {
		return ItemCommand.class.equals(clazz);
	}

	public void validate(Object obj, Errors errors) {
		Assert.notNull(obj);

		ItemCommand command = (ItemCommand) obj;
		Assert.notNull(command.getIsInsert());
		Assert.notNull(command.getItemType());

		// **************************
		// バージョン
		// **************************
		if (!command.getIsInsert() && command.getVersion() < 1) {
			rejectVersionError(errors, "version");
		}

		// **************************
		// 名前
		// **************************
		String currentField = "name";
		String currentValue = command.getName();
		String currentFieldName = "名前";

		if (StringUtils.isBlank(currentValue)) {
			rejectBlankError(errors, currentField, currentFieldName);
		}
		if (!errors.hasFieldErrors(currentField) && People.NAME_LENGTH < currentValue.length()) {
			rejectOverLengthError(errors, currentField, currentFieldName, People.NAME_LENGTH);
		}

		// **************************
		// 名前（カナ）
		// **************************
		currentField = "kanaName";
		currentValue = command.getKanaName();
		currentFieldName = "名前（カナ）";

		if (StringUtils.isNotBlank(currentValue)) {

			if (People.NAME_LENGTH < currentValue.length()) {
				rejectOverLengthError(errors, currentField, currentFieldName, People.NAME_LENGTH);
			}

			if (!errors.hasFieldErrors(currentField) && !PantheonStringUtil.isZenkakuKanaSpace(currentValue)) {
				rejectZenkakuKataKanaSpaceError(errors, currentField, currentFieldName);
			}
		}

		// **************************
		// 略称
		// **************************
		currentField = "shortName";
		currentValue = command.getShortName();
		currentFieldName = "略称";

		if (StringUtils.isNotBlank(currentValue)) {
			if (!errors.hasFieldErrors(currentField) && People.SHORT_NAME_LENGTH < currentValue.length()) {
				rejectOverLengthError(errors, currentField, currentFieldName, People.SHORT_NAME_LENGTH);
			}
		}

		// **************************
		// Wikipediaのページ
		// **************************
		currentField = "wikiName";
		currentValue = command.getWikiName();
		currentFieldName = "Wikipediaのページ";

		if (StringUtils.isNotBlank(currentValue)) {
			if (!errors.hasFieldErrors(currentField) && People.NAME_LENGTH < currentValue.length()) {
				rejectOverLengthError(errors, currentField, currentFieldName, People.NAME_LENGTH);
			}
		}

		// **************************
		// 名前（英語）
		// **************************
		currentField = "englishName";
		currentValue = command.getEnglishName();
		currentFieldName = "名前（英語）";

		if (StringUtils.isNotBlank(currentValue)) {
			if (!errors.hasFieldErrors(currentField) && People.ENGLISH_NAME_LENGTH < currentValue.length()) {
				rejectOverLengthError(errors, currentField, "currentFieldName", People.ENGLISH_NAME_LENGTH);
			}
		}

		// **************************
		// fromYear
		// **************************
		if (command.getItemType() != ItemType.NOT_CLASSIFICATION) {
			currentField = "fromYear";
			currentValue = command.getFromYear();
			switch (command.getItemType()) {
			case PEOPLE: {
				currentFieldName = "生誕年";
				break;
			}
			case BOOK: {
				currentFieldName = "出版年";
				break;
			}
			default:
				throw new IllegalArgumentException();
			}

			if (StringUtils.isBlank(currentValue)) {
				rejectBlankError(errors, currentField, currentFieldName);
			}
			if (!errors.hasFieldErrors(currentField) && People.YEAR_LENGTH < currentValue.length()) {
				rejectOverLengthError(errors, currentField, currentFieldName, People.YEAR_LENGTH);
			}
			if (!errors.hasFieldErrors("fromYear") && !NumberUtils.isNumber(command.getFromYear())) {
				rejectNumericError(errors, currentField, currentFieldName);
			}
		}

		// **************************
		// toYear
		// **************************
		if (command.getItemType() == ItemType.PEOPLE) {
			currentField = "toYear";
			currentValue = command.getToYear();
			currentFieldName = "死没年";

			if (StringUtils.isBlank(currentValue)) {
				rejectBlankError(errors, currentField, currentFieldName);
			}
			if (!errors.hasFieldErrors(currentField) && People.YEAR_LENGTH < currentValue.length()) {
				rejectOverLengthError(errors, currentField, currentFieldName, People.YEAR_LENGTH);
			}
			if (!errors.hasFieldErrors("toYear") && !NumberUtils.isNumber(command.getToYear())) {
				rejectNumericError(errors, currentField, currentFieldName);
			}
		}

		if (command.getItemType() == ItemType.PEOPLE) {
			// **************************
			// 生誕年,死没年
			// **************************
			if (!errors.hasFieldErrors("fromYear") && !errors.hasFieldErrors("toYear")) {
				int fromYear = Integer.parseInt(command.getFromYear());
				int toYear = Integer.parseInt(command.getToYear());

				if (toYear < fromYear) {
					errors.rejectValue("fromYear", ValidatorError.ERROR_PEOPLE_FROM_TO.name());
				}
			}
		}

		// **************************
		// プロパティ
		// **************************
		List<String> fromYearErrors = new ArrayList<String>();
		List<String> noteErrors = new ArrayList<String>();
		boolean propertyErrorFlag = false;
		for (int i = 0; i < command.getPropertyCount(); i++) {
			String fromYear = command.getPropertyFromYear()[i];
			String text = command.getPropertyText()[i];

			// どちらも入力されていなかったら無視
			if (StringUtils.isBlank(fromYear) && StringUtils.isBlank(text)) {
				fromYearErrors.add(null);
				noteErrors.add(null);
				continue;
			}

			if (StringUtils.isBlank(fromYear)) {
				fromYearErrors.add("年 を入力してください。");
				propertyErrorFlag = true;
			} else if (People.YEAR_LENGTH < fromYear.length()) {
				fromYearErrors.add("年 は半角数値5桁以内で入力してください。");
				propertyErrorFlag = true;
			} else if (!NumberUtils.isNumber(fromYear)) {
				fromYearErrors.add("年 は半角数値5桁以内で入力してください。");
				propertyErrorFlag = true;
			} else {
				fromYearErrors.add(null);
			}

			if (StringUtils.isBlank(text)) {
				noteErrors.add("できごと  を入力してください。");
				propertyErrorFlag = true;
			} else if (ItemProperty.TEXT_MAX_LENGTH < text.length()) {
				noteErrors.add("できごと  は" + ItemProperty.TEXT_MAX_LENGTH + "桁以内で入力してください。");
				propertyErrorFlag = true;
			} else {
				noteErrors.add(null);
			}
		}
		command.setPropertyFromYearErrMsg(fromYearErrors.toArray(new String[fromYearErrors.size()]));
		command.setPropertyTextErrMsg(noteErrors.toArray(new String[noteErrors.size()]));
		if(propertyErrorFlag){
			errors.rejectValue("propertyErr", "これはSpringフレームワークにエラーが発生したことを知らせるためのフラグです");
		}
	}
}