/*
 * Copyright (c) 2007 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.file.dao.standard;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Queue;
import java.util.concurrent.ArrayBlockingQueue;

import jp.terasoluna.fw.file.annotation.FileFormat;
import jp.terasoluna.fw.file.annotation.InputFileColumn;
import jp.terasoluna.fw.file.annotation.StringConverter;
import jp.terasoluna.fw.file.dao.FileException;
import jp.terasoluna.fw.file.dao.FileLineException;
import jp.terasoluna.fw.file.dao.FileLineIterator;

import org.apache.commons.lang.StringUtils;

/**
 * t@CANZX(f[^擾)p̋ʃNXB
 * 
 * <p>
 * t@CANZX(f[^擾)s3̃NX(CSVAŒ蒷Aϒ) ɋʂ鏈܂Ƃ߂ۃNXB
 * t@C̎ނɑΉTuNXsB<br>
 * gp{@link jp.terasoluna.fw.file.dao.FileLineIterator}QƂ̂ƁB
 * </p>
 * 
 * t@C擾͉L̎菇ŌĂяo悤Ɏ邱ƁB<br>
 * <ul>
 * <li>wb_擾(getHeader())</li>
 * <li>XLbv(skip())</li>
 * <li>f[^擾(readLine())</li>
 * <li>gC擾(getTrailer())</li>
 * </ul>
 * L̏Ԃł̂ݐmɃf[^擾łBL̏ʊOŏs<code>IllegalStateException<code>B<br>
 * WuĎs܂łɃt@CXVƃX^[g@\ɓ삵ȂB
 * 
 * @see jp.terasoluna.fw.file.dao.FileLineIterator
 * @see jp.terasoluna.fw.file.dao.standard.CSVFileLineIterator
 * @see jp.terasoluna.fw.file.dao.standard.FixedFileLineIterator
 * @see jp.terasoluna.fw.file.dao.standard.VariableFileLineIterator
 * @see jp.terasoluna.fw.file.dao.standard.PlainFileLineIterator
 * 
 * @param <T> t@CsIuWFNgB
 */
public abstract class AbstractFileLineIterator<T> implements
        FileLineIterator<T> {

    /**
     * t@CB
     */
    private String fileName = null;

    /**
     * ʃNXB
     */
    private Class<T> clazz = null;

    /**
     * s؂蕶B
     */
    private String lineFeedChar = System.getProperty("line.separator");

    /**
     * t@CGR[fBOB
     */
    private String fileEncoding = System.getProperty("file.encoding");

    /**
     * wb_sB
     */
    private int headerLineCount = 0;

    /**
     * gCsB
     */
    private int trailerLineCount = 0;

    /**
     * ݂̃f[^1s̕B
     */
    private String currentLineString = null;

    /**
     * t@C͏ς݂̃f[^̍sB
     */
    private int currentLineCount = 0;

    /**
     * t@CANZXp̕Xg[B
     */
    private Reader reader = null;

    /**
     * t@CsIuWFNgFieldiAnnotationji[ϐB
     */
    private Field[] fields = null;

    /**
     * t@CsIuWFNg̃XgORo[^i[ϐB
     */    
    private StringConverter[] stringConverter = null;
    
    /**
     * t@CsIuWFNg̃XgORo[^i[}bvB
     */
    private static Map<Class, StringConverter> stringConverterCacheMap = 
        new HashMap<Class, StringConverter>();
 
    /**
     * t@CsIuWFNgFieldɑΉsetter\bhi[B
     */
    private Method[] methods = null;

    /**
     * Jp[U[i[}bvB
     */
    private Map<String, ColumnParser> columnParserMap = null;

    /**
     * wb_̕񃊃XgB
     */
    private List<String> header = new ArrayList<String>();

    /**
     * gC̕񃊃XgB
     */
    private List<String> trailer = new ArrayList<String>();

    /**
     * wb_mFptOB
     */
    private boolean readHeader = false;

    /**
     * f[^mFptOB
     */
    private boolean readData = false;

    /**
     * gCmFptOB
     */
    private boolean readTrailer = false;

    /**
     * gC̈ꎞi[p̃L[B
     */
    private Queue<String> trailerQueue = null;

    /**
     * 1s̕ǂݍރIuWFNg
     */
    private LineReader lineReader = null;

    /**
     * t@CsIuWFNgŒ`ĂJi@InputFileColumntĂ鑮̐ji[B
     */
    private int columnCount = 0;

    /**
     * RXgN^B
     * 
     * @param fileName t@C
     * @param clazz ʃNX
     * @param columnParserMap tH[}bgXg
     */
    public AbstractFileLineIterator(String fileName, Class<T> clazz,
            Map<String, ColumnParser> columnParserMap) {
        if (fileName == null || "".equals(fileName)) {
            throw new FileException("File is not found.", 
                    new IllegalStateException(), fileName);
        }
        FileFormat fileFormat = clazz.getAnnotation(FileFormat.class);
        if (fileFormat == null) {
            throw new FileException("FileFormat annotation is not found.", 
                    new IllegalStateException(), fileName);
        }
        // ؂蕶ƈ͂ݕꍇAOX[B
        if (fileFormat.delimiter() == fileFormat.encloseChar()) {
            throw new FileException("Delimiter is the same as EncloseChar and is no use.",
                    new IllegalStateException(), fileName);
        }  
        this.fileName = fileName;
        this.clazz = clazz;
        if (fileFormat.lineFeedChar() != null
                && !"".equals(fileFormat.lineFeedChar())) {
            this.lineFeedChar = fileFormat.lineFeedChar();
        }
        if (fileFormat.fileEncoding() != null
                && !"".equals(fileFormat.fileEncoding())) {
            this.fileEncoding = fileFormat.fileEncoding();
        }
        this.headerLineCount = fileFormat.headerLineCount();
        this.trailerLineCount = fileFormat.trailerLineCount();
        this.columnParserMap = columnParserMap;
    }

    /**
     * ̍s̃R[h邩ǂmF邽߂̃\bhB<br>
     * JԂłɗvfꍇ true Ԃ܂B
     * 
     * @return JԂłɗvfꍇ <code>true</code>
     */
    public boolean hasNext() {
        try {
            if (reader.ready()) {
                return true;
            }
        } catch (IOException e) {
            throw new FileException(e, fileName);
        }
        return false;
    }

    /**
     * JԂŃt@CsIuWFNgԋpB
     * 
     * <p>
     * ̍s̃R[h̏t@CsIuWFNgɊi[ĕԋp܂B<br>
     * JԂŎ̗vfԂ܂B
     * </p>
     * 
     * @return t@CsIuWFNg
     */
    public T next() {
        if (readTrailer) {
            throw new FileLineException(
                    "Data part should be called before trailer part.", 
                    new IllegalStateException(),
                    fileName, currentLineCount, null, -1);
        }
        if (!hasNext()) {
            throw new FileLineException(new NoSuchElementException(), 
                fileName, currentLineCount, null, -1);
        }

        T fileLineObject = null;

        String currentString = readLine();

        // t@CsIuWFNgVɐ鏈B
        try {
            fileLineObject = clazz.newInstance();
        } catch (InstantiationException e) {
            throw new FileLineException(e, fileName, currentLineCount + 1);
        } catch (IllegalAccessException e) {
            throw new FileLineException(e, fileName, currentLineCount + 1);
        }

        // CSV̋؂蕶ɂē̓f[^𕪉B
        // ؂蕶̓Ame[V擾B
        String[] columns = separateColumns(currentString);

        // t@CǂݎJƃt@CsIuWFNg̃JrB
        if (columnCount != columns.length) {
            throw new FileLineException("Column Count is different from "
                    + "FileLineObject's column counts",
                    new IllegalStateException(),
                    fileName,
                    currentLineCount,
                    null,
                    -1);
        }

        for (int i = 0; i < fields.length; i++) {
            // JavaBean̓͗p̃Ame[Vݒ肷B
            InputFileColumn inputFileColumn = null;

            if (fields[i] != null) {
                inputFileColumn = fields[i]
                        .getAnnotation(InputFileColumn.class);
            }

            // t@CsIuWFNg̃Ame[VnullłȂΏpB
            if (inputFileColumn != null) {

                // 1J̕ZbgB
                String columnString = columns[inputFileColumn.columnIndex()];

                // TCY̊mFB
                if (0 < inputFileColumn.bytes()) {
                    try {
                        if (columnString.getBytes(fileEncoding).length
                                != inputFileColumn.bytes()) {
                            throw new FileLineException(
                                    "Data size is different from a set point "
                                    + "of a column.", 
                                    new IllegalStateException(),
                                    fileName, currentLineCount + 1,
                                    fields[i].getName(),
                                    inputFileColumn.columnIndex());
                        }
                    } catch (UnsupportedEncodingException e) {
                        throw new FileException(e, fileName);
                    }
                }

                // g
                columnString = FileDAOUtility.trim(columnString, fileEncoding,
                        inputFileColumn.trimChar(), inputFileColumn.trimType());

                // pfBO
                columnString = FileDAOUtility.padding(columnString,
                        fileEncoding, inputFileColumn.bytes(), inputFileColumn
                                .paddingChar(), inputFileColumn.paddingType());

                // ϊ̏B
                columnString = stringConverter[i].convert(columnString);

                // li[鏈B
                // JavaBean̑̌^̖OɂďU蕪B
                ColumnParser textSetter = columnParserMap.get(fields[i]
                        .getType().getName());
                try {
                    textSetter.parse(columnString, fileLineObject, methods[i],
                            inputFileColumn.columnFormat());
                } catch (IllegalArgumentException e) {
                    throw new FileLineException(e, fileName,  
                            currentLineCount + 1, fields[i].getName(),
                            inputFileColumn.columnIndex());
                } catch (IllegalAccessException e) {
                    throw new FileLineException(e, fileName,  
                            currentLineCount + 1, fields[i].getName(),
                            inputFileColumn.columnIndex());
                } catch (InvocationTargetException e) {
                    throw new FileLineException(e, fileName,  
                            currentLineCount + 1, fields[i].getName(),
                            inputFileColumn.columnIndex());
                } catch (ParseException e) {
                    throw new FileLineException(e, fileName,  
                            currentLineCount + 1, fields[i].getName(),
                            inputFileColumn.columnIndex());
                }

            }
        }
        readData = true;
        currentLineCount++;
        return fileLineObject;
    }

    /**
     * IteratorŒ`Ă郁\bhB
     * FileQueryDAOł͎Ȃ̂ŁÃNXĂяoꍇA
     * UnsupportedOperationExceptionX[B
     */
    public void remove() {
        throw new UnsupportedOperationException();
    }

    /**
     * B <br>
     * ōŝ͈ȉ̂RB
     * <ul>
     * <li>t@CsIuWFNg̑(Field)̎擾</li>
     * <li>gCL[̏</li>
     * <li>t@CI[v</li>
     * </ul>
     */
    protected void init() {
        try {
            this.reader = new BufferedReader(new InputStreamReader(
                    (new FileInputStream(fileName)), fileEncoding));
        } catch (UnsupportedEncodingException e) {
            throw new FileException(e, fileName);
        } catch (FileNotFoundException e) {
            throw new FileException(e, fileName);
        }

        if (1 <= trailerLineCount) {
            trailerQueue = new ArrayBlockingQueue<String>(trailerLineCount);
        }

        buildFields();
        buildStringConverter();
        buildMethods();

        // t@C1ǂŁAs؂蕶ƈv1sƃJEgB
        if (getEncloseChar() == Character.MIN_VALUE) {
            // ͂ݕAs؂蕶2
            if (lineFeedChar.length() == 2) {
                lineReader = new LineFeed2LineReader(reader, lineFeedChar);
            } else if (lineFeedChar.length() == 1) {
                // ͂ݕAs؂蕶1
                lineReader = new LineFeed1LineReader(reader, lineFeedChar);
            } else {
                throw new FileException(
                        "lineFeedChar length must be 1 or 2. but: "
                        + lineFeedChar.length(), new IllegalStateException(),
                        fileName);
            }
        } else {
            // ͂ݕAs؂蕶1
            if (lineFeedChar.length() == 1) {
                lineReader = new EncloseCharLineFeed1LineReader(getDelimiter(),
                       getEncloseChar(), reader, lineFeedChar);
                    // ͂ݕAs؂蕶2
            } else if (lineFeedChar.length() == 2) {
                lineReader = new EncloseCharLineFeed2LineReader(getDelimiter(),
                        getEncloseChar(), reader, lineFeedChar);
            } else {
                throw new FileException(
                        "lineFeedChar length must be 1 or 2. but: "
                        + lineFeedChar.length(), new IllegalStateException(),
                        fileName);
            }
        }
    }

    /**
     * t@CsIuWFNg̑̃tB[hIuWFNg̔z𐶐B<br>
     */
    private void buildFields() {
        // tB[hIuWFNg𐶐
        fields = clazz.getDeclaredFields();

        Field[] fieldArray = new Field[fields.length];

        for (Field field : getFields()) {
            InputFileColumn inputFileColumn = field
                    .getAnnotation(InputFileColumn.class);
            if (inputFileColumn != null) {
                if (fieldArray[inputFileColumn.columnIndex()] == null) {
                    fieldArray[inputFileColumn.columnIndex()] = field;
                } else {
                    throw new FileException("Column Index: "
                            + inputFileColumn.columnIndex()
                            + " is duplicated.", fileName);
                }
                columnCount++;
            }
        }
    }

    /**
     * t@CsIuWFNg̑̕ϊʃIuWFNg̔z𐶐B<br>
     */
    private void buildStringConverter() {
 
        // ϊʂ̔z𐶐
        stringConverter = new StringConverter[fields.length];
        
        for (int i = 0; i < fields.length; i++) {
            // JavaBean̓͗p̃Ame[V擾B
            InputFileColumn inputFileColumn = null;

            if (fields[i] != null) {
                inputFileColumn = fields[i]
                        .getAnnotation(InputFileColumn.class);
            }   
            
            // t@CsIuWFNg̃Ame[VnullłȂΏpB
            if (inputFileColumn != null) {
                
                // inputFileColumn.stringConverter()̓eɂ菈U蕪B
                try {
                    // ϊʂ̃Ame[V擾B
                    Class convertKind = inputFileColumn.stringConverter();
                    
                    // }bvɎ擾ϊʂƈvL[݂邩肷B
                    if (stringConverterCacheMap.containsKey(convertKind)) {
                        // }bvIuWFNg擾Aϊʂ̔zɃZbgB
                        stringConverter[i] = stringConverterCacheMap.
                            get(convertKind);  
                    
                    } else {
                        // CX^X𐶐Aϊʂ̔zɃZbgB
                        stringConverter[i] = inputFileColumn.
                            stringConverter().newInstance();
                        stringConverterCacheMap.put(convertKind,
                                stringConverter[i]);
                    }           
                    
                } catch (InstantiationException e) {
                    throw new FileLineException(e, fileName,
                            currentLineCount + 1, fields[i].getName(),
                            inputFileColumn.columnIndex());
                    
                } catch (IllegalAccessException e) {
                    throw new FileLineException(e, fileName,
                            currentLineCount + 1, fields[i].getName(),
                            inputFileColumn.columnIndex());
                }
            }
        }
    }

    /**
     * t@CsIuWFNg̑setter\bh̃\bhIuWFNg̔z𐶐B
     */
    private void buildMethods() {
        List<Method> methodList = new ArrayList<Method>();
        StringBuilder setterName = new StringBuilder();

        for (Field field : fields) {
            if (field.getAnnotation(InputFileColumn.class) != null) {
                // JavaBean珈̑ΏۂƂȂ鑮̑擾B
                String fieldName = field.getName();

                // ɁAsetter\bh̖O𐶐B
                setterName.setLength(0);
                setterName.append("set");
                setterName.append(StringUtils.upperCase(fieldName.substring(0,
                        1)));
                setterName.append(fieldName.substring(1, fieldName.length()));

                // setter̃tNVIuWFNg擾B
                // fields[i].getType()ň̌^w肵ĂB
                try {
                    methodList.add(clazz.getMethod(setterName.toString(),
                            new Class[] { field.getType() }));
                } catch (NoSuchMethodException e) {
                    throw new FileException(e, fileName);
                }
            } else {
                methodList.add(null);
            }
        }
        methods = methodList.toArray(new Method[methodList.size()]);
    }

    /**
     * t@CǏB
     */
    public void closeFile() {
        try {
            reader.close();
        } catch (IOException e) {
            throw new FileException(e, fileName);
        }
    }

    /**
     * wb_̃f[^擾郁\bhB
     * 
     * @return header wb_̕񃊃Xg
     */
    public List<String> getHeader() {
        if (readTrailer || readData) {
            throw new FileException(new IllegalStateException(), fileName);
        }
        if (!readHeader) {
            readHeader = true;
            if (0 < headerLineCount) {
                for (int i = 0; i < headerLineCount; i++) {
                    if (!hasNext()) {
                        throw new FileException(new NoSuchElementException(),
                                fileName);
                    }
                    try {
                        header.add(lineReader.readLine());
                    } catch (FileException e) {
                        throw new FileException(e, fileName);
                    }
                }
            }
        }
        return header;
    }

    /**
     * gC̃f[^擾郁\bh.
     * 
     * @return gC̕񃊃Xg
     */
    public List<String> getTrailer() {
        if (0 < trailerLineCount) {
            while (hasNext()) {
                if (!readHeader) {
                    getHeader();
                }
                if (trailerLineCount <= trailerQueue.size()) {
                    currentLineString = trailerQueue.poll();
                }
                try {
                    trailerQueue.add(lineReader.readLine());
                } catch (FileException e) {
                    throw new FileException(e, fileName);
                }
            }
            for (String fileLineBuilder : trailerQueue) {
                trailer.add(fileLineBuilder);
            }
        }
        readTrailer = true;
        return trailer;
    }

    /**
     * t@Cf[^̃f[^1sǂݎAƂČďoɕԋp.
     * 
     * @return f[^̂Ps̕
     */
    protected String readLine() {
        if (!hasNext()) {
            throw new FileException(new NoSuchElementException(), fileName);
        }

        if (!readHeader) {
            getHeader();
        }

        if (1 <= trailerLineCount) {
            if (trailerQueue.size() < trailerLineCount) {
                int loopCount = trailerLineCount - trailerQueue.size();
                for (int i = 0; i < loopCount; i++) {
                    if (!hasNext()) {
                        throw new FileException(new NoSuchElementException(),
                                fileName);
                    }
                    try {
                        trailerQueue.add(lineReader.readLine());
                    } catch (FileException e) {
                        throw new FileException(e, fileName);
                    }
                }
                if (!hasNext()) {
                    return null;
                }
            }

            currentLineString = trailerQueue.poll();
            try {
                trailerQueue.add(lineReader.readLine());
            } catch (FileException e) {
                throw new FileException(e, fileName);
            }
        } else {
            try {
                currentLineString = lineReader.readLine();
            } catch (FileException e) {
                throw new FileException(e, fileName);
            }
        }

        return currentLineString;
    }

    /**
     * X^[gɏς̃f[^̃f[^ǂݔ΂sB
     * 
     * @param skipLines ǂݔ΂sB
     */
    public void skip(int skipLines) {
        for (int i = 0; i < skipLines; i++) {
            readLine();
        }
    }

    /**
     * ؂蕶擾B
     * 
     * @return s؂蕶B
     */
    protected abstract char getDelimiter();

    /**
     * ͂ݕ擾B
     * 
     * @return ͂ݕB
     */
    protected abstract char getEncloseChar();

    /**
     * 񕪊.
     * <p>
     * f[^̃f[^Pst@CsIuWFNg̃Ame[V̋Lq
     * ]JɕB
     * </p>
     * 
     * @param fileLineString f[^̃f[^Ps
     * @return f[^Ps̕𕪉z
     */
    protected abstract String[] separateColumns(String fileLineString);

    /**
     * s؂蕶擾B
     * 
     * @return s؂蕶
     */
    protected String getLineFeedChar() {
        return lineFeedChar;
    }

    /**
     * t@CGR[fBO擾B
     * 
     * @return t@CGR[fBO
     */
    protected String getFileEncoding() {
        return fileEncoding;
    }

    /**
     * wb_s擾B
     * 
     * @return wb_s
     */
    protected int getHeaderLineCount() {
        return headerLineCount;
    }

    /**
     * gCs擾B
     * 
     * @return gCs
     */
    protected int getTrailerLineCount() {
        return trailerLineCount;
    }

    /**
     * t@C͏ς݂̃f[^̍s擾B
     * 
     * @return t@C͏ς݂̃f[^̍sB
     */
    public int getCurrentLineCount() {
        return currentLineCount;
    }

    /**
     * t@CsIuWFNgFieldiAnnotationji[ϐ擾B
     * 
     * @return t@CsIuWFNgFieldiAnnotationji[ϐ
     */
    protected Field[] getFields() {
        return fields;
    }

    /**
     * t@C擾B
     * @return fileName t@C
     */
    protected String getFileName() {
        return fileName;
    }
}
