/*
 * Copyright (c) 2007 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.batch.springsupport.init;

import java.util.HashMap;
import java.util.Map;

import jp.terasoluna.fw.batch.core.JobException;
import jp.terasoluna.fw.batch.core.JobStatus;
import jp.terasoluna.fw.batch.core.JobStatusSummarizer;
import jp.terasoluna.fw.batch.core.SupportProcessor;
import jp.terasoluna.fw.batch.core.WorkUnit;
import jp.terasoluna.fw.batch.core.Workable;
import jp.terasoluna.fw.batch.init.AbstractJobControlInfo;
import jp.terasoluna.fw.batch.init.EndFileChecker;
import jp.terasoluna.fw.batch.init.JobInfo;
import jp.terasoluna.fw.batch.openapi.JobContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;

/**
 * WuNpCNXB<BR>
 * WuNO<code>JobBeanFactory</code>̐jsB<BR>
 * ꂽ<code>JobBeanFactory</code>̓LbVꓯWusꂽ
 * ꍇ LbV<code>JobBeanFactory</code>ԂB
 * 
 */
public class JobExecutor implements DisposableBean,
        Workable<AbstractJobControlInfo> {

    /**
     * OCX^XB
     */
    private static Log log = LogFactory.getLog(JobExecutor.class);

    /**
     * t[[NBean`t@CB
     */
    private static final String FRAMEWORK_BEAN_DEFINITION_NAME = 
        "/common/FrameworkBean.xml";
    
    /**
     * f[^ANZXpBean`t@CB
     */
    protected static final String DATA_ACCESS_CONTEXT_DEFINITION_NAME = 
        "/common/dataAccessContext-batch.xml";

    /**
     * Xbhv[pBean`t@CB
     */
    private static final String THREAD_POOL_DEFINITION_NAME = 
        "/common/ThreadPoolContext-batch.xml";

    /**
     * Xbhv[pBean`t@CB
     */
    protected static final String VALIDATE_DEFINITION_NAME = 
        "/common/ValidationContext-batch.xml";

    /**
     * LbVLpBeanB
     */
    protected static final String USECACHE_NAME = "useCache";
    
    /**
     * WuReLXgBeanB
     */
    private static final String JOBCONTEXT_NAME = "jobContext";
    
    /**
     * Wu}l[WBeanB
     */
    private static final String JOBMANAGER_NAME = "jobManager";
    
    /**
     * Wu󋵗pBeanB
     */
    protected static final String JOBSTATUS_NAME = "JobStatus";

    /**
     * j^[pWu󋵗pBeanB
     */
    protected static final String MONITORABLE_JOBSTATUS_NAME = 
        "MonitorableJobStatus";

    /**
     * WuIĎpBeanB
     */
    private static final String ENDFILECHECKER_NAME = "endFileChecker";
    
    /**
     * WuIĎpBeanB
     */
    protected static final String USE_MONITORABLE = "useMonitorable";
    
    /**
     * Ԃ̃T}CUB
     */
    protected static final String JOBSTATUS_SUMMARIZER = "JobStatusSummarizer";

    /**
     * ftHgIR[hB
     */
    private static final String EXITCODE_MAP = "exitCodeMap";

    /**
     * t[[NBeanFactoryB
     */
    private ConfigurableApplicationContext frameworkFactory = null;

    /**
     * WuBeanFactorỹLbVpMapB
     */
    private Map<String, ConfigurableApplicationContext> jobFactoryMap = 
        new HashMap<String, ConfigurableApplicationContext>();

    /**
     * 񓯊NLtOB
     */
    private JobContext.START_TYPE async = JobContext.START_TYPE.SYNC;

    /**
     * WuOpCX^XB
     */
    private SupportProcessor preJobProcessor = null;

    /**
     * Wu㏈pCX^XB
     */
    private SupportProcessor postJobProcessor = null;

    /**
     * ftHg̃WuIR[h̃}bvB
     */
    private Map<String, String> jobExitCodeMap = null;
    
    /**
     * Wup[^̋؂蕶B
     */
    private String jobParametersSplitStr = null;
    
    /**
     * RXgN^B
     * 
     * <p><code>FrameworkFactory</code> sB</p>
     * 
     */
    protected JobExecutor() {
        frameworkFactory = new ClassPathXmlApplicationContext(
                FRAMEWORK_BEAN_DEFINITION_NAME);
        initDefaultJobExitCodeMap();
    }

    /**
     * RXgN^B(񓯊WuNp)
     * 
     * <p><code>FrameworkFactory</code> sB</p>
     * 
     * @param beanFileName Bean`t@C
     */
    protected JobExecutor(String[] beanFileName) {
        frameworkFactory = new ClassPathXmlApplicationContext(beanFileName);
        initDefaultJobExitCodeMap();
    }
    
    /**
     * Wu<code>BeanFactory</code>擾郁\bhB<BR>
     * ꂽ<code>BeanFactory</code>̓LbVėpB
     * 
     * @param jobInfo Wu˗
     * @return Wu<code>BeanFactory</code>
     */
    protected synchronized ConfigurableApplicationContext getJobBeanFactory(
            JobInfo jobInfo) {
        ConfigurableApplicationContext jobFactory = null;
        if (isAsync()) {
            // LbVꂽ񂪂邩mF
            if (jobFactoryMap.containsKey(jobInfo.getJobDiscriptorPath())) {
                if (log.isDebugEnabled()) {
                    log.debug("Get the cached JobBeanFactory");
                }
                jobFactory = jobFactoryMap.get(jobInfo.getJobDiscriptorPath());
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("Get the newly created JobBeanFactory");
                }
                jobFactory = new ClassPathXmlApplicationContext(
                        new String[] { jobInfo.getJobDiscriptorPath() },
                        frameworkFactory);
                if ((Boolean) jobFactory.getBean(USECACHE_NAME)) {
                    // WuBeanFactorỹLbV
                    jobFactoryMap.put(jobInfo.getJobDiscriptorPath(),
                            jobFactory);
                }
            }
        } else {
            jobFactory = new ClassPathXmlApplicationContext(new String[] {
                    jobInfo.getJobDiscriptorPath(),
                    DATA_ACCESS_CONTEXT_DEFINITION_NAME,
                    THREAD_POOL_DEFINITION_NAME, VALIDATE_DEFINITION_NAME },
                    frameworkFactory);
        }
        return jobFactory;
    }

    /**
     * 񓯊WuNł邩]B
     * 
     * @return 񓯊WűNł<code>true</code>ԂB
     */
    protected boolean isAsync() {
        return async == JobContext.START_TYPE.ASYNC;
    }

    /**
     * Wu}l[W擾B
     * 
     * @param jobFactory Wu<code>BeanFactory</code>
     * @return Wu}l[W
     */
    @SuppressWarnings("unchecked")
    protected Workable<WorkUnit> getJobManager(
            ConfigurableApplicationContext jobFactory) {
        Workable<WorkUnit> jobManager = (Workable<WorkUnit>) jobFactory
                .getBean(JOBMANAGER_NAME);
        return jobManager;
    }

    /**
     * WuReLXg̎擾B<BR>
     * WuReLXg擾Alݒ肷B
     * 
     * @param jobFactory Wu<code>BeanFactory</code>
     * @param jobInfo Wu˗
     * @return WuReLXg
     */
    protected JobContext getJobContext(
            ConfigurableApplicationContext jobFactory, JobInfo jobInfo) {
        JobContext jobContext = (JobContext) jobFactory
                .getBean(JOBCONTEXT_NAME);
        jobContext.setStartType(async);
        jobContext.setPartitionNo(-1);
        jobContext.setJobId(jobInfo.getJobId());
        jobContext.setJobRequestNo(jobInfo.getJobRequestNo());
        jobContext.setParameter(jobInfo.getJobParameters());
        return jobContext;
    }

    /**
     * WuԊi[p<code>JobStatus</code>擾B<BR>
     * <code>JobStatus</code>擾Alݒ肷B
     * 
     * @param jobFactory Wu<code>BeanFactory</code>
     * @param jobInfo Wu˗
     * @return WuԊi[p<code>JobStatus</code>
     */
    protected JobStatus getJobStatus(ConfigurableApplicationContext jobFactory,
            JobInfo jobInfo) {
        JobStatus jobStatus = null;
        if ((Boolean) jobFactory.getBean(USE_MONITORABLE)) {
            jobStatus = (JobStatus) jobFactory.getBean(
                    MONITORABLE_JOBSTATUS_NAME);
        } else {
            jobStatus = (JobStatus) jobFactory.getBean(JOBSTATUS_NAME);
        }

        jobStatus.setJobState(JobStatus.STATE.STARTED);
        jobStatus.setJobId(jobInfo.getJobId());
        jobStatus.setJobRequestNo(jobInfo.getJobRequestNo());
        return jobStatus;
    }

    /**
     * <code>WorkUnit</code>𐶐AWuReLXgݒ肷B<BR>
     * 
     * @param jobContext WuReLXg
     * @return <code>WorkUnit</code>
     */
    protected WorkUnit getWorkUnit(JobContext jobContext) {
        WorkUnit rootWorkQueueElement = new WorkUnit() {

            private JobContext workUnitJobContext;

            public boolean isEndMark() {
                return false;
            }

            public JobContext getJobContext() {
                return workUnitJobContext;
            }

            public void setJobContext(JobContext jobContext) {
                this.workUnitJobContext = jobContext;
            }
        };
        rootWorkQueueElement.setJobContext(jobContext);
        return rootWorkQueueElement;
    }

    /**
     * It@CĎXbh̏sB<br>
     * WȕԂo^B
     * 
     * @param jobStatus Wȕ
     * @param jobContext WuReLXg
     */
    protected void initEndFileChecker(JobStatus jobStatus,
            JobContext jobContext) {
        EndFileChecker endFileChecker
            = (EndFileChecker) frameworkFactory.getBean(ENDFILECHECKER_NAME);
        if (endFileChecker != null) {
            endFileChecker.addParentJobStatus(jobStatus, jobContext.getJobId(),
                    jobContext.getJobRequestNo());
        }
    }

    /**
     * It@CĎXbhɓo^ꂽWȕԂ폜B
     * 
     * @param jobStatus Wȕ
     * @param jobContext WuReLXg
     */
    protected void removeJobStatusFromEndFileChecker(JobStatus jobStatus,
            JobContext jobContext) {
        EndFileChecker endFileChecker
            = (EndFileChecker) frameworkFactory.getBean(ENDFILECHECKER_NAME);
        if (endFileChecker != null) {
            endFileChecker.removeParentJobStatus(jobStatus,
                    jobContext.getJobId(), jobContext.getJobRequestNo());
        }
    }
    
    /**
     * WusB<BR>
     * Wu˗pĎw肳ꂽWuNB<BR>
     * 
     * @param jobInfo Wu˗
     * @return Wus
     */
    protected JobStatus execute(JobInfo jobInfo) {
        //JnOo
        printStartLog(jobInfo);
        
        //WuÑ`FbN
        checkJobInfo(jobInfo);
        
        // WuBeanFactory
        ConfigurableApplicationContext jobFactory = getJobBeanFactory(jobInfo);

        // }lW[擾
        Workable<WorkUnit> jobManager = getJobManager(jobFactory);

        // WuReLXg
        JobContext jobContext = getJobContext(jobFactory, jobInfo);
        
        // WuԂ̏
        JobStatus jobStatus = getJobStatus(jobFactory, jobInfo);

        // It@CĎXbhɃWuԂo^
        initEndFileChecker(jobStatus, jobContext);
        
        WorkUnit rootWorkQueueElement = getWorkUnit(jobContext);
        
        jobManager.work(rootWorkQueueElement, jobStatus);

        ((JobStatusSummarizer) jobFactory.getBean(JOBSTATUS_SUMMARIZER))
                .summarize(jobStatus);

        setDefaultJobExitCode(jobStatus);
        
        if (!(Boolean) jobFactory.getBean(USECACHE_NAME)) {
            jobFactory.close();
        }

        // It@CĎXbhWuԂ폜
        removeJobStatusFromEndFileChecker(jobStatus, jobContext);
        
        //WȕIOo
        printEndLog(jobStatus);
        
        return jobStatus;
    }

    /**
     * WůJnOo͂B
     * @param jobInfo Wu˗
     */
    protected void printStartLog(JobInfo jobInfo) {
        if (log.isInfoEnabled()) {
            StringBuilder logStr = new StringBuilder();
            logStr.append("Job processing START : [jobId=");
            logStr.append(jobInfo.getJobId());
            logStr.append("] [jobRequestNo=");
            logStr.append(jobInfo.getJobRequestNo());
            logStr.append("] [StartType=");
            logStr.append(async);
            logStr.append("]");
            log.info(logStr.toString());
        }
    }

    /**
     * WůJnOo͂B
     * @param jobStatus Wȕ
     */
    protected void printEndLog(JobStatus jobStatus) {
        if (log.isInfoEnabled()) {
            StringBuilder logStr = new StringBuilder();
            logStr.append("Job processing END : [jobId=");
            logStr.append(jobStatus.getJobId());
            logStr.append("] [jobRequestNo=");
            logStr.append(jobStatus.getJobRequestNo());
            logStr.append("] [StartType=");
            logStr.append(async);
            logStr.append("] [jobExitCode=");
            logStr.append(jobStatus.getJobExitCode());
            logStr.append("]");
            log.info(logStr.toString());
        }
    }

    /**
     * pł邩`FbNB
     * 
     * @param jobInfo `FbNΏۂ̕
     */
    private void checkJobInfo(JobInfo jobInfo) {
        for (char c : jobInfo.getJobId().toCharArray()) {
            if (!(('0' <= c && c <= '9')
                    || ('a' <= c && c <= 'z')
                    || ('A' <= c && c <= 'Z')
                    || c == '-'
                    || c == '_')
                    ) {
                throw new JobException("JobID should use only an "
                        + "alphanumeric character");
            } 
        }
    }


    /**
     * WuIR[hݒ肷B
     * IR[h̐ݒ肪Null̏ꍇABean`̏ԕʏIR[h
     * ftHgIR[hƂĐݒ肷B
     * 
     * @param jobStatus
     *            Ώۂ̏
     */
    protected void setDefaultJobExitCode(JobStatus jobStatus) {
        if (jobStatus.getJobExitCode() != null) {
            return;
        }
        String code = jobExitCodeMap.get(jobStatus.getJobStateStr());
        jobStatus.setJobExitCode(Integer.valueOf(code));
    }

    /**
     * WuIR[h̃}bvB
     *
     */
    @SuppressWarnings("unchecked")
    private void initDefaultJobExitCodeMap() {
        jobExitCodeMap = 
            (Map<String, String>) frameworkFactory.getBean(EXITCODE_MAP);

        // IR[hݒ肳Ėꍇ̓G[ƂB
        if (jobExitCodeMap == null || jobExitCodeMap.size() < 0) {
            throw new JobException("There is no setup of Default exitCodeMap");
        }
    }
    
    /**
     * 񓯊^WuÑ[J\bhB
     * 
     * @param jobInfo Wu˗
     * @param jobStatus WuXe[^X
     */
    public void work(AbstractJobControlInfo jobInfo, JobStatus jobStatus) {
        jobInfo.setJobParametersSplitStr(jobParametersSplitStr);
        JobStatus workJobStatus = jobStatus.getChild(new JobContext());
        workJobStatus.setJobState(JobStatus.STATE.STARTED);

        AsyncJobContext asyncJobContext = new AsyncJobContext(jobInfo);

        // WuJnXV
        preJobProcessor.process(asyncJobContext, workJobStatus);

        if (workJobStatus.getJobState() != JobStatus.STATE.STARTED) {
            return;
        }

        JobStatus resultJobStatus = null;
        try {
            resultJobStatus = execute(jobInfo);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            resultJobStatus = new JobStatus();
            resultJobStatus.setJobState(JobStatus.STATE.ENDING_ABNORMALLY);
            
            String exitCode = 
                jobExitCodeMap.get(resultJobStatus.getJobStateStr());
            resultJobStatus.setJobExitCode(Integer.valueOf(exitCode));
            
        }
        
        // Wuʂ̐ݒ
        jobInfo.setJobExitCode(
                String.valueOf(resultJobStatus.getJobExitCode()));
        // WuIԐݒ
        jobInfo.setJobState(String.valueOf(
                resultJobStatus.getJobState().ordinal()));
        
        // WusDBf
        postJobProcessor.process(asyncJobContext, workJobStatus);
    }

    /**
     * 쐬<code>BeanFactory</code>̔jsB<BR>
     * LbV<code>JobBeanFactory</code>y<code>FrameworkBeanFactory
     * </code>jB
     * 
     */
    public void destroy() {
        // WuBeanFactroy̏I
        for (ConfigurableApplicationContext context : jobFactoryMap.values()) {
            context.close();
        }
        // t[[NBeanFactroy̏I
        if (frameworkFactory != null) {
            frameworkFactory.close();
        }
    }

    /**
     * 񓯊NLtOݒ肷B<BR>
     * Spring̒lCWFNVB
     * 
     * @param async 񓯊NLtO
     */
    public void setAsync(boolean async) {
        if (async) {
            this.async = JobContext.START_TYPE.ASYNC;
        } else {
            this.async = JobContext.START_TYPE.SYNC;
        }
    }

    /**
     * WuN㏈ݒ肷B
     * 
     * @param preJobProcessor WuNO
     */
    public void setPreJobProcessor(SupportProcessor preJobProcessor) {
        this.preJobProcessor = preJobProcessor;
    }

    /**
     * WuNOݒ肷B
     * 
     * @param postJobProcessor WuNO
     */
    public void setPostJobProcessor(SupportProcessor postJobProcessor) {
        this.postJobProcessor = postJobProcessor;
    }

    /**
     * Wup[^̋؂蕶ݒ肷B
     * 
     * @param jobParametersSplitStr Wup[^̋؂蕶
     */
    public void setJobParametersSplitStr(String jobParametersSplitStr) {
        this.jobParametersSplitStr = jobParametersSplitStr;
    }
    
    /**
     * t[[NBeanFactory̎擾
     * 
     * @return frameworkFactory t[[NBeanFactory
     */
    protected ConfigurableApplicationContext getFrameworkFactory() {
        return frameworkFactory;
    }
}
