package jp.sourceforge.talisman.i18n;

/*
 * $Id: Messages.java 140 2008-07-15 04:01:32Z tama3 $
 */

import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import javax.swing.Icon;
import javax.swing.ImageIcon;

/**
 * This class manages multiple {@link ResourceBundle
 * <code>ResourceBundle</code>} and current locale.
 * 
 * @author Haruaki Tamada
 * @version $Revision: 140 $
 */
public class Messages{
    private Locale currentLocale;
    private String resourcePath;
    private Map<Locale, ResourceBundle> resources = new HashMap<Locale, ResourceBundle>();

    /**
     * basic constructor.
     *
     * @param defaultLocale default locale of this instance.
     * @param resourcePath resource path
     * @exception ResourceNotFoundException resource file is not found.
     */
    public Messages(Locale defaultLocale, String resourcePath) throws ResourceNotFoundException{
        this.resourcePath = resourcePath;
        this.currentLocale = defaultLocale;

        ResourceBundle rb = getBundle(defaultLocale);
        if(rb == null){
            throw new ResourceNotFoundException(resourcePath + " is not found");
        }
    }

    /**
     * <code>this(Locale.getDefault(), resourcePath);</code>
     */
    public Messages(String resourcePath) throws ResourceNotFoundException{
        this(Locale.getDefault(), resourcePath);
    }

    /**
     * <code>MessageFormat.format(get(locale, key), args);</code>
     */
    public String format(Locale locale, String key, Object... args){
        String value = get(locale, key);

        if(value != null){
            return MessageFormat.format(value, args);
        }
        return value;
    }

    /**
     * <code>format(getLocale(), key, args);</code>
     */
    public String format(String key, Object... args){
        return format(getLocale(), key, args);
    }

    /**
     * returns a resource value related the given key.
     */
    public String get(Locale locale, String key){
        return get(locale, key, null);
    }

    /**
     * returns a resource value related the given key.
     * If the resource value is null, then returns given defaultValue.
     */
    public String get(Locale locale, String key, String defaultValue){
        ResourceBundle rb = getBundle(locale);
        int currentIndex = 0;
        String value;
        try{
            value = rb.getString(key);
        } catch(MissingResourceException e){
            value = defaultValue;
            if(value == null){
                return null;
            }
        }

        // replace "hoge${fugakey}" to "hogefuga"
        // when fugakey=fuga was defined.
        while((value.indexOf('$', currentIndex) >= 0)){
            int index = value.indexOf('$', currentIndex);
            if(value.charAt(index + 1) == '$'){
                currentIndex = index + 2;
            }
            else if(value.charAt(index + 1) == '{'){
                currentIndex = index;
                int last = value.indexOf('}', currentIndex + 1);
                if(last > 0){
                    String subkey = value.substring(index + 2, last);
                    if(hasValue(locale, subkey)){
                        String subvalue = get(locale, subkey);
                        StringBuilder builder = new StringBuilder();
                        builder.append(value.substring(0, currentIndex));
                        builder.append(subvalue);
                        builder.append(value.substring(last + 1));
                        currentIndex += subvalue.length() + 1;
                        value = new String(builder);
                    }
                }
                else{
                    currentIndex++;
                }
            }
            else{
                currentIndex = index + 1;
            }
        }

        return value;
    }

    public String get(String key){
        return get(getLocale(), key);
    }

    public String get(String key, String defaultValue){
        return get(getLocale(), key, defaultValue);
    }

    /**
     * <code>get(locale, key).split(", *");</code>
     */
    public String[] getArray(Locale locale, String key){
        String value = get(locale, key);
        if(value != null){
            return value.split(", *");
        }
        return new String[0];
    }

    /**
     * <code>getArray(getLocale(), key);</code>
     */
    public String[] getArray(String key){
        return getArray(getLocale(), key);
    }

    /**
     * <p>
     * finds and returns a icon.
     * The icon is located at the value of key.
     * The main routine is as follows.
     * </p>
     * <pre>new ImageIcon(getClass().getResource(get(key)));</pre>
     * <p>
     * Not that icon may not be found because class loader might be different this and icon resource.
     * </p> 
     */
    public Icon getIcon(Locale locale, String key){
        String value = get(locale, key);
        if(value != null){
            if(value.endsWith(".png") || value.endsWith(".jpg") || value.endsWith(".gif")){
                return new ImageIcon(getClass().getResource(value));
            }
        }
        return null;
    }

    /**
     * finds and returns a icon.
     * @see #getIcon(Locale, String)
     */
    public Icon getIcon(String key){
        return getIcon(getLocale(), key);
    }

    /**
     * returns current locale.
     */
    public Locale getLocale(){
        return currentLocale;
    }

    public boolean hasValue(Locale locale, String key){
        try{
            ResourceBundle rb = getBundle(locale);
            rb.getString(key);
            return true;
        }catch(MissingResourceException e){
        }
        return false;
    }

    public boolean hasValue(String key){
        return hasValue(getLocale(), key);
    }

    /**
     * sets current locale.
     * @param locale new locale.
     */
    public void setLocale(Locale locale){
        if(locale == null){
            throw new NullPointerException();
        }
        this.currentLocale = locale;
    }

    private ResourceBundle getBundle(Locale locale){
        ResourceBundle rb = resources.get(locale);
        if(rb == null){
            try{
                rb = ResourceBundle.getBundle(resourcePath, locale);
                resources.put(locale, rb);
            } catch(MissingResourceException e){
            }
        }
        return rb;
    }
}
