/*
 *  Copyright 2004 Clinton Begin
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *
 * "Portions Copyrighted [2007] [MASAHITO HENMI]"
 *
 */
package dwarf.lru;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.logging.*;
import java.text.SimpleDateFormat;
/**
 * LRU (least recently used) cache controller implementation
 */
public class LruCacheController /*implements CacheController*/ {

  // jdk1.4 logging
  protected final static Logger log = Logger.getLogger("strawberry.lru.LruCacheController");

  private int cacheSize;
  private Map cache;
  private List keyList;

  /**
   * Default constructor
   */
  public LruCacheController() {
    this.cacheSize = 10;
    this.cache = Collections.synchronizedMap(new HashMap());
    this.keyList = Collections.synchronizedList(new LinkedList());
  }

  public int getMapSize() {
    return cache.size();
  }

  public String getHtml() {
    StringWriter sw = new StringWriter();
    PrintWriter pw = new PrintWriter(sw);
    pw.println("<table border='1'>");
    pw.println("<tr>");
    pw.println("<td>key</td>");
    pw.println("<td>creation</td>");
    pw.println("<td># used</td>");
    pw.println("<td>execution time(ms)</td>");
    pw.println("</tr>");
    
    int n = keyList.size();
    for (int i = 0; i < n; i++) {
      Object key = keyList.get(i);
      Object result = cache.get(key);
      SqlResultBean bean = (SqlResultBean) result;

      String wk = simpleDateFormat.format( bean.getCreatedTime().getTime() );

        
      pw.println("<tr>");
      pw.println("<td>" + key + "</td>");
      pw.println("<td>" + wk + "</td>");
      pw.println("<td>" + bean.getCount() + "</td>");
      pw.println("<td>" + bean.getExecutionTime() + "</td>");
      pw.println("</tr>");
    }

    pw.println("</table>");
    pw.close();
    return sw.toString();
  }

  static  SimpleDateFormat simpleDateFormat = new SimpleDateFormat("MM/dd HH:mm:ss");
  
  /**
   * Configures the cache
   *
   * @param props Optionally can contain properties [reference-type=WEAK|SOFT|STRONG]
   */
  public void configure(Properties props) {
    String size = props.getProperty("cache-size");
    if (size == null) {
      size = props.getProperty("size");
    }
    if (size != null) {
      cacheSize = Integer.parseInt(size);
    }
  }

  /**
   * Add an object to the cache
   *
   * @param cacheModel The cacheModel
   * @param key        The key of the object to be cached
   * @param value      The object to be cached
   */
  public void putObject(Object cacheModel, Object key, Object value) {
    cache.put(key, value);
    keyList.add(key);
    if (keyList.size() > cacheSize) {
      try {
        Object oldestKey = keyList.remove(0);
        log.info("TCYI[ôŁA폜܂ " + oldestKey);
        cache.remove(oldestKey);
      } catch (IndexOutOfBoundsException e) {
        //ignore
      }
    }
  }

  /**
   * Get an object out of the cache.
   *
   * @param cacheModel The cache model
   * @param key        The key of the object to be returned
   * @return The cached object (or null)
   */
  public Object getObject(Object cacheModel, Object key) {
    Object result = cache.get(key);
    keyList.remove(key);
    if (result != null) {
      keyList.add(key);
    }
    return result;
  }

  public Object removeObject(Object cacheModel, Object key) {
    keyList.remove(key);
    return cache.remove(key);
  }

  /**
   * Flushes the cache.
   *
   * @param cacheModel The cache model
   */
  public void flush(Object cacheModel) {
    cache.clear();
    keyList.clear();
  }

}
