/*
 
 TopMind jLynx "jLynx JDBC Framework"
 Copyright (c) 2004-2006. TopMind Systems Inc.
 All rights reserved.
 
 This file is part of TopMind jLynx.
 
 TopMind jLynx is free software; you can redistribute it and/or modify
 it under the terms of the License. See website for License.
 
 */
package net.sf.jlynx;

import java.rmi.server.UID;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.DriverManager;
import java.sql.SQLException;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.sql.DataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * Class used to manage database connections and local transactions
 *
 * $Revision: 73 $
 * $Date: 2006-11-13 09:58:31 -0500 (Mon, 13 Nov 2006) $
 *
 */
final class TransactionManager {
    
    static final int MYSQL = 1;
    
    static final int MSSQL = 2;
    
    static final int ORACLE = 3;
    
    static final int DB2 = 4;
    
    static final int HSQL = 5;
    
    public static int dbVendor;
    
    private static String driver;
    
    protected static boolean isLookup;
    
    private static String jndiDS;
    
    static Logger logger = LoggerFactory.getLogger(TransactionManager.class);
    
    private static String password;
    
    private static String url;
    
    private static String user;
    
    /**
     * Commits local database transaction identified by txID parameter and
     * closes connection
     *
     * @param txID
     * @throws SQLException
     */
    public synchronized static void commit(Connection cn) throws SQLException {
        // Connection cn = (Connection) sessionConnectionMap.get(txID);
        cn.commit();
        cn.close();
        cn = null;
    }
    
    /**
     *
     * This method will return a unique String that can be used as a transaction
     * identifier. Uses java.rmi.server.UID.
     *
     * @return a unique String
     */
    public static synchronized String getUID() {
        
        return new UID().toString();
        // return new Integer(++connCount).toString();
    }
    
    /**
     * Initializes default jLynx connection
     *
     * Only used by UI (1/22/06)
     *
     * @return void
     */
    static void guiInit() {
        
        // java.io.InputStream is = null;
        InitialContext ic = null;
        Connection conn1 = null;
        
        try {
            
            jndiDS = ConfigParser.getJndiDs();
            
            if (jndiDS != null) {
                jndiDS = jndiDS.trim();
                isLookup = true;
                logger.info("Using JNDI DataSource " + jndiDS);
                
            } else {
                
                isLookup = false;
                
                url = ConfigParser.getUrl();
                driver = ConfigParser.getDriver();
                user = ConfigParser.getUsername();
                password = ConfigParser.getPassword();
                
            }
            
            if (isLookup) {
                
                ic = new InitialContext();
                DataSource ds = (DataSource) ic.lookup("java:comp/env/"
                        + jndiDS);
                conn1 = ds.getConnection();
                
            } else {
                
                Class.forName(driver).newInstance();
                conn1 = DriverManager.getConnection(url, user, password);
                conn1.setReadOnly(false);
                
            }
            
            DatabaseMetaData dmd = conn1.getMetaData();
            String dbName = dmd.getDatabaseProductName();
            logger.info("DatabaseProductName: " + dbName);
            
            if ("MySQL".equalsIgnoreCase(dbName)) {
                dbVendor = MYSQL;
                // escapeChar = "`";
            } else if ("Microsoft SQL Server".equalsIgnoreCase(dbName))
                dbVendor = MSSQL;
            else if ("Oracle".equalsIgnoreCase(dbName))
                dbVendor = ORACLE;
            else if ("DB2".equalsIgnoreCase(dbName.substring(0, 3)))
                dbVendor = DB2;
            else
                dbVendor = 0;
            
        } catch (Exception e) {
            dbVendor = 0;
        }
        conn1 = null;
        
    }
    
    /**
     * Rollback local database transaction identified by txID parameter and
     * closes connection
     *
     * @param txID
     * @throws SQLException
     */
    public synchronized static void rollback(Connection cn) throws SQLException {
        // Connection cn = (Connection) sessionConnectionMap.get(txID);
        cn.rollback();
        cn.close();
        cn = null;
    }
    
    ConnectionBean cb = (ConnectionBean) ConfigParser.getConnectionDefs()
    .get(ConfigParser.DEFAULT);
    
    /**
     * Open and return a new database connection using jLynx default
     * configuration
     *
     * @throws java.sql.SQLException
     * @return java.sql.Connection
     */
    Connection open() throws SQLException, NamingException {
        
        if (cb == null) {
            
            throw new SQLException("Connection named <<"
                    + ConfigParser.DEFAULT
                    + ">> does not exist, i.e. <connection name='default'>");
            
        }
        String dbUrl = cb.getUrl();
        String dbUser = cb.getUsername();
        String dbPwd = cb.getPassword();
        String jdbcDriver = cb.getDriver();
        String jndiDs = cb.getJndiDataSource();
        
        // logger.info("opening connection: " + dbUrl);
        
        Connection conn1 = null;
        
        if (jndiDs != null) {
            
            InitialContext ic = new InitialContext();
            DataSource ds = (DataSource) ic.lookup("java:comp/env/" + jndiDs);
            conn1 = ds.getConnection();
            
        } else {
            
            try {
                Class.forName(jdbcDriver).newInstance();
            } catch (InstantiationException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            } catch (IllegalAccessException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            } catch (ClassNotFoundException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
            conn1 = DriverManager.getConnection(dbUrl, dbUser, dbPwd);
            conn1.setReadOnly(false);
        }
        
        return conn1;
    }
    
    public Connection open(String namedConnection) throws SQLException,
            NamingException {
        
        cb = (ConnectionBean) ConfigParser.getConnectionDefs().get(namedConnection);
        return open();
        
    }
    
}