/*
 
 TopMind jLynx "jLynx JDBC Framework"
 Copyright (c) 2004-2006. TopMind Systems Inc.
 All rights reserved.
 
 This file is part of TopMind jLynx.
 
 TopMind jLynx is free software; you can redistribute it and/or modify
 it under the terms of the License. See website for License.
 
 */
package net.sf.jlynx;

import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Serializable;
import java.net.URL;
import java.sql.Connection;
import java.util.Date;
import java.util.Enumeration;
import java.util.Map;
import java.util.Stack;
import java.util.TreeMap;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;

/**
 * Parses jLynx configuration file <code>jlynx.xml</code> using SAX XML
 * parser. See website for details.<br>
 * <br>
 * $Revision: 73 $<br>
 * $Date: 2006-11-13 09:58:31 -0500 (Mon, 13 Nov 2006) $
 */
public class ConfigParser extends DefaultHandler implements Serializable {
    
    protected static final String $version = "v1.2.1 (SVN $Revision: 73 $)";
    
    private static String configFile;
    
    static int conn = 0;
    
    private static Map _CONN_MAPPING = new TreeMap();
    
    private static boolean debug = true;
    
    public static final String DEFAULT = "default";
    
    public static final String DEFAULT_CONFIG_FILENAME = "jlynx.xml";
    
    static String dir;
    
    private static String driver;
    
    protected static String fileSep = System.getProperty("file.separator");
    
    private static boolean isParsed = false;
    
    private static String jndiDs;
    
    static Logger logger = LoggerFactory.getLogger(ConfigParser.class);
    
    private static Map mappings = new TreeMap();
    
    private static String password;
    
    protected static String schema;
    
    private static final long serialVersionUID = 3246578199368951463L;
    
    static boolean SingleConnectionMode = true;
    
    protected static Map NAMED_QUERY_MAPPING = new TreeMap();
    
    protected static Map QUERY_CONN_MAPPING = new TreeMap();
    
    protected static Map QUERY_CLASS_MAPPING = new TreeMap();
    
    private static Stack stack = new Stack();
    
    private static Map tableMap;
    
    private static String url;
    
    protected static String userDir = System.getProperty("user.dir");
    
    private static String username;
    
    private static boolean valid = true;
    
    static {
        init();
    }
    
    static String getConfigFile() {
        try {
            if (configFile.indexOf("http:") == -1)
                return new File(configFile).getCanonicalPath();
            else
                return configFile;
        } catch (Exception e) {
            return configFile;
        }
    }
    
    /**
     *
     * Returns a Connection object as configured in jlynx.xml.
     *
     * @param namedConnection -
     *            connection name from jlynx.xml
     * @return java.sql.Connection - a valid Connection object
     *
     */
    public static Connection getConnection(String namedConnection) {
        try {
            if (namedConnection == null)
                namedConnection = ConfigParser.DEFAULT;
            if (ConfigParser.getConnectionDefs().containsKey(namedConnection))
                return new TransactionManager().open(namedConnection);
        } catch (Exception e) {
            logger.error("Connection not available :: " + e.getMessage());
            // e.printStackTrace();
        }
        return null;
    }
    
    static Map getConnectionDefs() {
        return _CONN_MAPPING;
    }
    
    public static String getDriver() {
        return driver;
    }
    
    public static String getJndiDs() {
        return jndiDs;
    }
    
    public static Map getMappings() {
        return mappings;
    }
    
    protected static String getPassword() {
        return password;
    }
    
    /**
     * Returns SQL statement as configured in jlynx.xml.
     *
     * @param namedQuery
     * @return String - SQL statement
     */
    public static String getQuery(String namedQuery) {
        
        String q = (String) ConfigParser.NAMED_QUERY_MAPPING.get(namedQuery);
        if (q == null)
            return "";
        else
            return q;
        
    }
    
    public static String getUrl() {
        return url;
    }
    
    public static String getUsername() {
        return username;
    }
    
    static void init() {
        
        String vendor = (System.getProperty("java.vendor") == null) ? "--" : System.getProperty("java.vendor");
        
        if(!isParsed) {
            logger.info("jLynx JVM " + vendor + " - " + System.getProperty("java.version"));
            logger.info("jLynx java.home - " + System.getProperty("java.home"));
            logger.info("jLynx user.dir - " + System.getProperty("user.dir"));
        }
        
        if (getDriver() != null || getJndiDs() != null) {
            return;
        }
        
        setConfigFile(ConfigParser.configFile);
        
        ConfigParser conf = new ConfigParser();
        SAXParser sp = null;
        XMLReader xr = null;
        
        try {
            
            sp = SAXParserFactory.newInstance().newSAXParser();
            xr = sp.getXMLReader();
            xr.setContentHandler(conf);
            InputSource s = new InputSource();
            s.setSystemId(ConfigParser.configFile);
            xr.parse(s);
            valid = true;
            
        } catch (Exception e) {
            
            valid = false;
            logger.error("Fatal error :: 'jlynx.xml' could not be parsed");
            logger.error(e.getMessage());
            
        }
        
        if(isParsed)
            logger.info("jLynx JDBC Framework " + $version + " initialization complete");
    }
    
    protected static boolean isDebug() {
        return debug;
    }
    
    
    /**
     * File discovery order:
     *
     * 1) WEB-INF/jlynx.xml 2) env lookup 3) system prop 4) user.dir
     *
     */
    private static void setConfigFile(String file) {
        
        ClassLoader contextClassLoader = Thread.currentThread()
        .getContextClassLoader();
        
        String env = null;
        Context ctx = null;
        
        try {
            Enumeration configFiles = contextClassLoader.getResources(".."
                    + fileSep + ConfigParser.DEFAULT_CONFIG_FILENAME);
            
            while (configFiles.hasMoreElements()) {
                
                URL url = (URL) configFiles.nextElement();
                String f = url.getFile();
                
                if (f.indexOf("WEB-INF") != -1 && f.endsWith("jlynx.xml")) {
                    
                    logger.info("jLynx config file - " + f);
                    configFile = url.getFile();
                    return;
                    
                }
            }
            
            ctx = (Context) new InitialContext().lookup("java:comp/env");
            env = (String) ctx.lookup("jlynx.configuration");
            if (env != null) {
                logger
                        .info("Environment variable 'jlynx.configuration' used :: "
                        + env);
                configFile = env;
                return;
            }
            
        } catch (Exception ex) {
            logger
                    .info("Environment variable 'jlynx.configuration' was not found");
        } finally {
            ctx = null;
        }
        
        if (System.getProperty("jlynx.configuration") != null) {
            ConfigParser.configFile = System.getProperty("jlynx.configuration");
            logger.info("System property 'jlynx.configuration' used: "
                    + System.getProperty("jlynx.configuration"));
            return;
        }
        
        if (file != null) {
            
            if (file.indexOf("http:") != -1 && new File(file).canRead()) {
                ConfigParser.configFile = file;
            }
            
        } else {
            
            file = "jlynx.xml";
            configFile = userDir + fileSep + file;
            
            if (!new File(configFile).canRead())
                configFile = DEFAULT_CONFIG_FILENAME;
            // check dir where jlynx.jar is or directory above
            
        }
        logger.info("jlynx.configuration (user.dir): " + configFile);
        
    }
    
    static void setDriver(String driver) {
        ConfigParser.driver = driver;
    }
    
    static void setJndiDs(String jndiDs) {
        ConfigParser.jndiDs = jndiDs;
    }
    
    static void setPassword(String password) {
        ConfigParser.password = password;
    }
    
    static void setUrl(String url) {
        ConfigParser.url = url;
    }
    
    static void setUsername(String username) {
        ConfigParser.username = username;
    }
    
    static void writeConfig(String directory) throws IOException {
        
        if (directory == null)
            directory = ".";
        
        dir = directory;
        StringBuffer out = new StringBuffer();
        out.append("<?xml version=\"1.0\" encoding=\"iso-8859-1\"?>\n");
        out.append("<jlynx>\n");
        out.append("<!-- jlynx config generated on " + new Date().toString()
        + " -->\n");
        out.append(" <connection name=\"" + DEFAULT + "\">\n");
        
        if (jndiDs != null && !"".equals(jndiDs))
            out.append("  <jndi-ds>" + jndiDs.trim() + "</jndi-ds>\n");
        else
            out.append("  <jndi-ds />\n");
        
        if (driver != null && !"".equals(driver))
            out.append("  <driver>" + driver.trim() + "</driver>\n");
        else
            out.append("  <driver />\n");
        
        if (url != null && !"".equals(url))
            out.append("  <db-url>" + url.trim() + "</db-url>\n");
        else
            out.append("  <db-url />\n");
        
        if (username != null && !"".equals(username))
            out.append("  <db-user>" + username.trim() + "</db-user>\n");
        else
            out.append("  <db-user />\n");
        
        if (password != null && !"".equals(password))
            out.append("  <password>" + password.trim() + "</password>\n");
        else
            out.append("  <password />\n");
        
        if (schema != null && !"".equals(schema))
            out.append("  <schema>" + schema.trim() + "</schema>\n");
        else
            out.append("  <schema />\n");
        
        out.append(Generator.e);
        
        out.append(" </connection>\n");
        
        out.append("</jlynx>\n");
        
        // System.out.println(out);
        
        File file = new File(directory + File.separator + "jlynx.xml");
        String millis = new Long(System.currentTimeMillis()).toString();
        File fileSave = new File("jlynx-backup-" + millis + ".xml");
        int c;
        if (file.canRead()) {
            FileReader in = new FileReader(file);
            FileWriter writerBak = new FileWriter(fileSave);
            while ((c = in.read()) != -1)
                writerBak.write(c);
            writerBak.close();
        }
        
        if (file.canWrite() || file.createNewFile()) {
            FileWriter writer = new FileWriter(file);
            writer.write(out.toString());
            writer.flush();
            writer.close();
            System.out.println("Configuration saved to " + directory + fileSep
                    + "jlynx.xml");
        } else
            throw new IOException("cannot write to jlynx.xml");
        
    }
    
    private ConnectionBean cb;
    
    String className = null;
    
    private boolean connectionValid = true;
    
    String connName;
    
    String ds = null;
    
    private Map entities = null;
    
    String tableName = null;
    
    private StringBuffer qry = null;
    
    private String qryConnection = ConfigParser.DEFAULT;
    
    private String qryName = null;
    
    // private String qryResultClass = null;
    
    public void characters(char[] chars, int start, int length)
    throws SAXException {
        
        isParsed = true;
        
        if (length > 0) {
            
            String currElement = (String) stack.peek();
            
            String text = new String(chars, start, length).trim();
            
            if (text != null && text.length() > 0) {
                
                if (currElement.equalsIgnoreCase("jndi-ds")) {
                    if (cb != null && cb.getName().equals(DEFAULT))
                        jndiDs = text;
                    cb.setJndiDataSource(text);
                }
                if (currElement.equalsIgnoreCase("db-url")) {
                    if (cb != null && cb.getName().equals(DEFAULT))
                        url = text;
                    // System.out.println(text);
                    cb.setUrl(text);
                }
                if (currElement.equalsIgnoreCase("db-user")) {
                    if (cb != null && cb.getName().equals(DEFAULT))
                        username = text;
                    cb.setUsername(text);
                }
                if (currElement.equalsIgnoreCase("driver")) {
                    if (cb != null && cb.getName().equals(DEFAULT))
                        driver = text;
                    cb.setDriver(text);
                }
                if (currElement.equalsIgnoreCase("password")) {
                    if (cb != null && cb.getName().equals(DEFAULT))
                        password = text;
                    cb.setPassword(text);
                }
                if (currElement.equalsIgnoreCase("schema")) {
                    if (cb != null && cb.getName().equals(DEFAULT))
                        ConfigParser.schema = text;
                    cb.setSchema(text);
                }
                if (currElement.equalsIgnoreCase("query")) {
                    
                    // saxNormalizeQuery(chars, start, length);
                    String str = new String(chars, start, length);
                    qry.append(str);
                    
                }
                
            }
            
        }
    }
    
    public void endElement(String namespaceURI, String localName, String qName) {
        
        if (qName == "connection" && this.connectionValid) {
            
            cb.setMappings(entities);
            
            if (!_CONN_MAPPING.containsKey(connName)) {
                _CONN_MAPPING.put(connName, cb);
                
                logger.info("jLynx Connection - '" + connName
                        + "'");
            } else {
                logger.warn("Cannot add connection named '" + connName
                        + "' because connection already exists");
            }
        } else if ("query".equalsIgnoreCase(qName) && qryName != null
                && qry != null) {
            
            if (qry != null) {
                
                qryName = qryName.trim();
                
                String q = saxNormalizeQuery(qry.toString());
                
                if (!ConfigParser.NAMED_QUERY_MAPPING.containsKey(qryName)) {
                    
                    ConfigParser.NAMED_QUERY_MAPPING.put(qryName, q);
                    
                    logger.info("jLynx NamedQuery - '" + qryName
                            + "' = "
                            + ConfigParser.NAMED_QUERY_MAPPING.get(qryName)
                            + "");
                }
                
            }
        }
        
    }
    
    private String saxNormalizeQuery(String input) {
        
        if (input == null)
            return "";
        else {
            
            char newLine = '\n';
            char cr = '\r';
            char tab = '\t';
            char space = ' ';
            
            input = input.replace(newLine, space);
            input = input.replace(tab, space);
            input = input.replace(cr, space);
            
            while (input.indexOf("  ") != -1)
                input = StringUtils.replace(input, "  ", " ");
            
            // logger.debug(input);
            return input.trim();
        }
        
    }
    
    public void startElement(String uri, String localName, String element,
            Attributes attribs) throws SAXException {
        
        stack.push(element);
        
        if ((element.equalsIgnoreCase("entity") || element
                .equalsIgnoreCase("table"))) {
            
            tableMap = null;
            
            tableName = attribs.getValue("name");
            className = attribs.getValue("class");
            ds = attribs.getValue("jndi-ds");
            // System.out.println(ds);
            
            if (className != null && tableName != null) {
                mappings.put(className, tableName);
                entities.put(className, tableName);
            }
            
            if (ds != null && tableName != null) {
                mappings.put(tableName + ".ds", ds);
            }
            
        } else if (element.equalsIgnoreCase("column")) {
            
            String j, d;
            
            j = attribs.getValue("property");
            d = attribs.getValue("name");
            
            if (tableMap == null) {
                tableMap = new TreeMap();
            }
            
            tableMap.put(j, d);
            
            mappings.put(tableName, tableMap);
            
            entities.put(tableName, tableMap);
            
        } else if (element.equalsIgnoreCase("connection")) {
            
            // stack.pop();
            entities = new TreeMap();
            
            cb = null;
            cb = new ConnectionBean();
            String cn = attribs.getValue("name");
            if (cn == null) {
                logger
                        .error("Config error! Required attribute name is missing i.e. <connection name='?'>");
                this.connectionValid = false;
            } else {
                this.connectionValid = true;
                logger.debug("Parsing connection name = " + cn);
            }
            if (cn != null)
                connName = cn.trim().toLowerCase();
            
            cb.setName(connName);
            
            String attrib = attribs.getQName(0);
            if (attrib != null && attrib.equalsIgnoreCase("debug")
            && attribs.getValue(attrib).equalsIgnoreCase("true"))
                debug = true;
            
        } else if (element.equalsIgnoreCase("query")) {
            
            // process sql name queries
            // format <query name="name" class="Pojo">SELECT ...</query>
            // query should contain valid sql statements used by applications
            
            // this.qryConnection = attribs.getValue("connection");
            if (qryConnection == null)
                qryConnection = ConfigParser.DEFAULT;
            // this.qryResultClass = attribs.getValue("class");
            this.qryName = attribs.getValue("name");
            this.qry = new StringBuffer("");
            
        }
        
    }
    
}
