<?php

require_once('SimgLog.php');

/**
 * configファイルの内容でdefineを定義する
 * @param $ini 定義ファイル展開配列
 */
function simg_config_load($ini) {
	foreach($ini as $key=>$val) {
		if(!defined($key)) {
			define($key, "{$val}");
		}
	}
}

/**
 * configファイルを指定キーで検索し、その値を返す
 * @param $ini 定義ファイル展開配列
 * @param $key キー
 * @return 値
 */
function simg_find_config_value($ini, $key) {
	foreach($ini as $k=>$v) {
		if(strcasecmp($key, $k) == 0) {
			return $v;
		}
	}
	return null;
}

/**
 * パス組合せ
 * @param $base 基底パス
 * @param $path パス
 * @return 組合せパス
 */
function simg_combine_path($base, $path) {

	if(strlen($path) > 0 && substr($path, 0, 1) == '/')
		return sprintf("%s%s", $base, $path);
	else
		return sprintf("%s/%s", $base, $path);
}

/**
 * 配列から値を取得する
 * @param $ary 配列
 * @param $key キー
 * @param $def デフォルト値
 * @return 値
 */
function simg_get_array_value($ary, $key, $def) {
	$ret = $def;
	if(isset($ary[$key])) {
		$ret = $ary[$key];
	}
	return $ret;
}

/**
 * 値を取得する
 * @param $v 値
 * @param $def デフォルト値
 * @return 値
 */
function simg_get_value($v, $def) {
	$ret = $def;
	if(!is_null($v)) {
		$ret = $v;
	}
	return $ret;
}

/**
 * 現在のURLを取得する
 * @return URL
 */
function simg_current_url() {
	return simg_make_url($_SERVER["REQUEST_URI"]);
}

/**
 * URL作成
 * @param $directory URLディレクトリ部
 * @param $params パラメータマップ（パラメータキー=>パラメータ値）
 * @return URL
 */
function simg_make_url($directory, $params=array()) {
	//SimgLog::getInstance()->put(PEAR_LOG_DEBUG, "HTTP_HOST [%s]", $_SERVER["HTTP_HOST"]);
	//SimgLog::getInstance()->put(PEAR_LOG_DEBUG, "REQUEST_URI [%s]", $_SERVER["REQUEST_URI"]);
	//SimgLog::getInstance()->put(PEAR_LOG_DEBUG, "SCRIPT_NAME [%s]", $_SERVER["SCRIPT_NAME"]);
	$url = empty($_SERVER["HTTPS"])? 'http://': 'https://';
	$url .= $_SERVER["HTTP_HOST"];
	if($directory != null && strlen($directory) > 0) {
		$url = simg_combine_path($url, $directory);
	}
	//////////////////////////////////
	$lanv = simg_get_array_value($_REQUEST, SIMG_PARAM_LANGUAGE, null);
	if($lanv != null) {
		$b = false;
		foreach($params as $k => $v) {
			if($k == SIMG_PARAM_LANGUAGE) {
				$b = true;
				break;
			}
		}
		if(!$b) {
			$params[SIMG_PARAM_LANGUAGE] = $lanv;
		}
	}
	//////////////////////////////////
	if(count($params) > 0) {
		$url .= '?';
		//SimgLog::getInstance()->put(PEAR_LOG_DEBUG, "simg_make_url http_build_query [%s]", print_r($params, true));
		$url .= @http_build_query($params);
	}
	return $url;
}

/**
 * インスタンス生成
 * @param $path クラスファイルが格納されているパス
 * @param $class_name クラス名
 * @param $args コンストラクタ引数
 * @return インスタンス
 */
function simg_create_instance($path, $class_name, $args) {
	$file_name = sprintf('%s/%s.php', $path, $class_name);
	if(!file_exists($file_name)) {
		SimgLog::getInstance()->put(PEAR_LOG_ERR, "simg_create_instance: not exist PHP file [%s]", $file_name);
		return null;
	}
	require_once $file_name;
	if(!class_exists($class_name)) {
		SimgLog::getInstance()->put(PEAR_LOG_ERR, "simg_create_instance: not exist class [%s]", $class_name);
		return null;
	}
	//$instance = new $class_name();
	//$args = func_get_args();
	//array_shift($args);
	//array_shift($args);
	$reflection = new ReflectionClass($class_name);
	$instance = $reflection->newInstanceArgs($args);

	return $instance;
}

/**
 * ディレクトリ配下のPHPファイルからインスタンスを生成する
 * @param $dir ディレクトリ
 * @param $baseclass 基底クラス
 * @param $args コンストラクタ引数
 * @param $instances インスタンス配列
 * @param $cmap クラス重複チェックMAP
 * @return インスタンス配列
 */
function simg_create_instance_on_directory($dir, $baseclass, $args, $instances, $cmap) {
	if(is_null($instances)) {
		$instances = array();
	}
	if(is_null($cmap)) {
		$cmap = array();
	}
	if(file_exists($dir)) {
		$dir_handle = opendir($dir);
		while($file_name = readdir($dir_handle)) {
			$ext = pathinfo($file_name, PATHINFO_EXTENSION);
			if($ext == 'php') {
				$cn = pathinfo($file_name, PATHINFO_FILENAME);
				if(!isset($cmap[$cn])) {
					$cmap[$cn] = "";
					$ins = simg_create_instance($dir, $cn, $args);
					if(!is_null($ins) && $ins instanceof $baseclass) {
						$instances[] = $ins;
					}
				}
			}
		}
	}
	return $instances;
}

/**
 * ディレクトリ配下のPHPファイルからインスタンスを生成する
 * @param $pathes パス配列
 * @param $baseclass 基底クラス
 * @param $args コンストラクタ引数
 * @return インスタンス配列
 */
function simg_create_instance_on_directory_by_pathes($pathes, $baseclass, $args) {
	$instances = array();
	$cmap = array();
	foreach($pathes as $path) {
		$instances = simg_create_instance_on_directory($path, $baseclass, $args, $instances, $cmap);
	}
	return $instances;
}

/**
 * クッキークリア
 * @param $name 項目
 */
function simg_clear_cookie($name) {
	setcookie ($name, "", time() - 3600);
}

/**
 * 303リダイレクト
 * @param $url URL
 */
function simg_redirect_by303($url) {
	header('Location: ' . $url, true, 303);
}

/**
 * SIMG定義ファイルを処理する
 * @param $ini 定義ファイル展開配列
 * @param $targets 処理対象セクション配列
 * @param $maps マップ配列
 * @return 生成したマップ配列
 */
function simg_make_definemap($ini, $targets, $maps) {
	if(!is_null($ini)) {
		if(isset($ini['prefix'])) {
			$ini_prefix = $ini['prefix'];
			foreach($targets as $target) {
				$maps = __make_definemap_inner($ini, $ini_prefix, $target, $maps);
			}
		}
	}
	return $maps;
}
/**
 * SIMG定義ファイルを処理する（内部関数）
 * @param $ini 定義ファイル展開配列
 * @param $ini_prefix prefixセクション展開配列
 * @param $targets 処理対象セクション
 * @param $maps マップ配列
 * @return 生成したマップ配列
 */
function __make_definemap_inner($ini, $ini_prefix, $section_key, $maps) {
	$section = $section_key;
	$sub_section = null;
	$ar = explode(':', $section_key);
	if(count($ar) > 1) {
		$section = $ar[0];
		$sub_section = $ar[1];
		$maps = __make_definemap_inner($ini, $ini_prefix, $sub_section, $maps);
	}

	if(isset($ini[$section_key])) {
		$w = $ini[$section_key];
		if(!is_null($w)) {
			//$map = array();
			$map = (isset($maps[$section]))? $maps[$section]: array();
			foreach($w as $key => $val) {
				$key = trim($key);
				$val = trim($val);
				$ar = explode(':', $val);
				$is_flat = false;
				if(count($ar) > 1 && !is_null($sub_section)) {
					$val = trim($ar[0]);
					if(!array_key_exists($val, $map)) {
						$map[$val] = array();
						$prefix = trim($ini_prefix[$sub_section]);
						foreach(explode(',', $ar[1]) as $el) {
							$name = __make_prefix($prefix, trim($el));
							if(defined($name)) {
								array_push($map[$val], constant($name));
							} else {
								SimgLog::getInstance()->put(PEAR_LOG_WARNING, "__make_definemap_inner: undef [%s] [%s]", $section_key, $name);
							}
						}
					}
				} else {
					$is_flat = true;
				}
				$name = __make_prefix(trim($ini_prefix[$section]), trim($key));
				if(!defined($name)) {
					define($name, $val);
					if($is_flat) {
						$map[$name] = $val;
					}
				}
			}
			$maps[$section] = $map;
		}
	}
	return $maps;
}
/**
 * プレフィクス付加（内部関数）
 * @param $pre プレフィクス
 * @param $name 名称
 * @return プレフィクス付名称
 */
function __make_prefix($pre, $name) {
	if(!is_null($pre)) {
		$name = $pre . $name;
	}
	return $name;
}

/**
 * 設定ファイルをパースする
 * @param $pathes パス配列
 * @param $fn ファイル名
 * @param $process_sections true/false:セクション名と設定が含まれた多次元の配列 / 以外
 * @return パース結果（false:エラー）
 */
function simg_parse_ini_file_pathes($pathes, $fn, $process_sections=false) {
	//SimgLog::getInstance()->put(PEAR_LOG_DEBUG, "simg_parse_ini_file_pathes : pathes=%s", print_r($pathes, true));
	//SimgLog::getInstance()->put(PEAR_LOG_DEBUG, "simg_parse_ini_file_pathes : fn=%s", $fn);
	$ini = array();
	foreach($pathes as $path) {
		$ini = simg_parse_ini_file(simg_combine_path($path, $fn), $process_sections, $ini);
	}
	//SimgLog::getInstance()->put(PEAR_LOG_DEBUG, "simg_parse_ini_file_pathes : ini=%s", print_r($ini, true));
	return $ini;
}

/**
 * 設定ファイルをパースする
 * @param $file ファイルパス
 * @param $process_sections true/false:セクション名と設定が含まれた多次元の配列 / 以外
 * @param $ini パース結果
 * @param $first true/false=最値優先/後値優先
 * @return パース結果
 */
function simg_parse_ini_file($file, $process_sections=false, $ini=null, $first=true) {

	//SimgLog::getInstance()->put(PEAR_LOG_DEBUG, "simg_parse_ini_file : file=%s", $file);
	$res  = is_null($ini)? array(): $ini;
	if(!file_exists($file))
		return $res;
	if(!$lines = file($file, 1))
		return $res;
	$curr = &$res;
	$sectionPattern = '/^\[([^\]]+)\]/';
	$linePattern = '/^\s*([^=\s]*)\s*=\s*"?(.*?)"?\s*$/';

	foreach($lines as $line) {
		if(strlen($line) > 0 && substr($line, 0, 1) != ';') {
			if($process_sections && preg_match($sectionPattern, $line, $m)) {
				if(!isset($res[$m[1]])) {
					$res[$m[1]] = array();
				}
				$curr = &$res[$m[1]];
			} elseif(preg_match($linePattern, $line, $m)) {
				if(!$first || !isset($curr[$m[1]])) {
					$curr[$m[1]] = $m[2];
				}
			}
		}
	}

	return $res;
}

/**
 * 最初に存在するファイルのフルパスを取得する
 * @param $pathes パス配列
 * @param $fn ファイル名
 * @param $def デフォルト値
 * @return ファイルフルパス
 */
function simg_find_first_exist_file($pathes, $fn=null, $def=null) {
	$ret = $def;
	foreach($pathes as $path) {
		$w = (is_null($fn))? $path: simg_combine_path($path, $fn);
		if(file_exists($w)) {
			$ret = $w;
			break;
		}
	}
	return $ret;
}

?>
