<?php

require_once('simg-functions.php');
require_once('SimgLog.php');

define('_ERROR_INDEX', '_e');
define('_INFO_INDEX', '_i');

/**
 * 出力クラス
 * @author System Concept
 *
 */
abstract class SimgOutput {
	/**
	 * Smartyインスタンス
	 */
	protected $smarty;
	/**
	 * ページ情報
	 * key: テンプレートファイル名
	 * value: パーシャル配列
	 */
	protected $page;
	/**
	 * plugin情報配列
	 */
	protected $plugins;
	/**
	 * テンプレートファイル名
	 */
	private $template;
	/**
	 * エラー情報
	 */
	private $error;
	/**
	 * Path管理
	 */
	protected $app_path;

	/**
	 * 構築
	 * @param $ap Path管理
	 */
	public function __construct($ap) {
		$this->initialize($ap);
	}

	/**
	 * ホームテンプレート取得
	 * @return テンプレート
	 */
	abstract public function getHomeTemplate();

	/**
	 * システムエラーテンプレート取得
	 * @return テンプレート
	 */
	abstract public function getSystemErrorTemplate();

	/**
	 * 初期化
	 * @param $ap Path管理
	 */
	private function initialize($ap) {
		$this->app_path = $ap;
		$this->page = $this->createPage();
		$this->smarty = $this->createSmarty();
		$this->plugins = $this->createPlugin();
		$this->tempalte = null;
		$this->error = array();
	}
	/**
	 * ページ情報作成
	 * @return ページ情報
	 */
	protected function createPage() {
		$ap = $this->app_path;
		$maps = simg_make_definemap(simg_parse_ini_file_pathes($ap->getPathes('getConfigPath'), SIMG_FN_CONFIG_PAGE, true), array('partial', 'smarty', 'constant', 'template:partial'), array());
		return simg_get_array_value($maps, 'template', null);
	}
	/**
	 * ページ情報configファイルパス取得
	 * @return ファイルパス
	 */
	//protected function getPageConfigPath() {
	//	return simg_combine_path(SIMG_PATH_CONFIG, SIMG_FN_CONFIG_PAGE);
	//}

	/**
	 * Smarty生成
	 */
	protected function createSmarty() {
		$ap = $this->app_path;
		$smarty = new Smarty();
		$smarty->template_dir = $ap->findFirstExistFile($ap::getTemplatePath()) . '/';
		$smarty->compile_dir = $ap->findFirstExistFile($ap::getTemplateCompilePath()) . '/';
		$smarty->cache_dir = $ap->findFirstExistFile($ap::getTemplateCachePath()) . '/';
		$this->loadResourceOnDirectory($smarty);
		if(defined('SIMG_LEFT_DELIMITER')) {
			$smarty->left_delimiter = SIMG_LEFT_DELIMITER;
		}
		if(defined('SIMG_RIGHT_DELIMITER')) {
			$smarty->right_delimiter = SIMG_RIGHT_DELIMITER;
		}
		return $smarty;
	}
	/**
	 * Plugin生成
	 */
	protected function createPlugin() {
		$ap = $this->app_path;
		$pmap = simg_make_definemap(simg_parse_ini_file(simg_combine_path(SIMG_ROOT, 'simg-smarty-plugin.ini'), true), array('plugin', 'parameter'), array());
		foreach(simg_get_array_value($pmap, 'plugin', array()) as $k => $v) {
			$this->activePlaginInner(array(SIMG_ROOT), $v);
		}
		return simg_get_array_value(simg_make_definemap(simg_parse_ini_file_pathes($ap->getPathes('getConfigPath'), SIMG_FN_CONFIG_PLUGIN, true), array('plugin', 'parameter'), array()), 'plugin', array());
	}
	/**
	 * ディレクトリ配下のリソースファイルをSmarty定数として読み込む
	 * @param $smarty ビュー
	 */
	protected function loadResourceOnDirectory($smarty) {
		$ap = $this->app_path;
		$rfils = array();
		foreach($ap->getPathes('getResourcePath') as $dir) {
			$dir_handle = opendir($dir);
			while($file_name = readdir($dir_handle)) {
				$ext = pathinfo($file_name, PATHINFO_EXTENSION);
				if($ext == SIMG_RESOURCE_EXT) {
					$w = simg_combine_path($dir, pathinfo($file_name, PATHINFO_BASENAME));
					//$smarty->configLoad($w);
					simg_config_load(simg_parse_ini_file($w));
					array_unshift($rfils, $w);
				}
			}
		}
		foreach($rfils as $f) {
			$smarty->configLoad($f);
		}
	}
	/**
	 * テンプレートファイル名設定
	 * @param $v テンプレートファイル名
	 */
	public function setTemplate($v) {
		$this->template = $v;
	}
	/**
	 * テンプレートファイル名取得
	 * @return テンプレートファイル名
	 */
	public function getTemplate() {
		return $this->template;
	}
	/**
	 * テンプレート未設定か否か
	 * @return true/false=テンプレート空き/以外
	 */
	public function isTemplateEmpty() {
		return is_null($this->template);
	}
	/**
	 * パーシャルを保持しているかを調べる
	 * @param $partial パーシャル
	 * @param $template テンプレート名
	 * @return true/false=保持している/していない
	 */
	public function hasPartial($partial, $template = null) {
		$template = (is_null($template))? $this->template: $template;
		//$ar = simg_get_value($this->page[$template], array());
		$ar = simg_get_array_value($this->page, $template, array());
		return in_array($partial, $ar);
	}
	/**
	 * テンプレートに値を割り当てる
	 * @param $name 名前
	 * @param $val 値
	 */
	public function assign($name, $val) {
		$this->smarty->assign($name, $val);
	}
	/**
	 * テンプレートを表示する
	 * @return true/false=表示した/テンプレートが無い
	 */
	public function display() {
		$ret = false;
		$temp = $this->template;
		//SimgLog::getInstance()->put(PEAR_LOG_DEBUG, "display template: [%s]", $temp);
		if(!is_null($temp)) {
			//$ctype = simg_get_value(simg_find_config_value(SIMG_ROOT . '/simg-content-type.ini', pathinfo(basename($temp), PATHINFO_EXTENSION)), '');
			$ctype = simg_get_value(simg_find_config_value(simg_parse_ini_file(SIMG_ROOT . '/simg-content-type.ini'), pathinfo(basename($temp), PATHINFO_EXTENSION)), '');
			if(defined('SIMG_TEMPLATE_EXTENSION') && strlen(SIMG_TEMPLATE_EXTENSION) > 0) {
				$temp .= '.';
				$temp .= SIMG_TEMPLATE_EXTENSION;
			}
			//SimgLog::getInstance()->put(PEAR_LOG_DEBUG, "display : temp [%s]", $temp);
			if($this->smarty->templateExists($temp)) {
				if(strlen($ctype) > 0) {
					header('Content-Type: ' . $ctype);
				}
				$this->smarty->display($temp);
			}
			$ret = true;
		}
		return $ret;
	}
	/**
	 * プラグイン有効化
	 * @param $info プラグイン情報（配列可、null:全て）
	 */
	public function activePlagin($info = null) {
		//SimgLog::getInstance()->put(PEAR_LOG_DEBUG, "activePlagin : info [%s]", $info);
		$pathes = $this->app_path->getPathes('getPluginPath');
		if(!is_null($info) && !is_array($info)) {
			$this->activePlaginInner($pathes, $info);
		} else {
			foreach((is_null($info))? $this->plugins : $info as $el) {
				$this->activePlaginInner($pathes, $el);
			}
		}
	}
	/**
	 * プラグイン有効化（内部関数）
	 * @param $pathes パス配列
	 * @param $info プラグイン情報
	 */
	protected function activePlaginInner($pathes, $info) {
		//SimgLog::getInstance()->put(PEAR_LOG_DEBUG, "activePlaginInner : info [%s]", $info);
		$ar = explode(':', $info);
		if(count($ar) >= 3) {
			$prefix = trim($ar[0]);
			//$php = simg_combine_path((!is_null($directory))? $directory: SIMG_PATH_PLUGIN, trim($ar[1]) . '.php');
			//$php = $ap->findFirstExistFile($ap::getPluginPath(), trim($ar[1]) . '.php');
			$php = simg_find_first_exist_file($pathes, trim($ar[1]) . '.php');
			if(!is_null($php)) {
				//SimgLog::getInstance()->put(PEAR_LOG_DEBUG, "> : php [%s]", $php);
				if(file_exists($php)) {
					require_once($php);
					$ptypes = explode(',', SIMG_PLUGIN_TYPES);
					//SimgLog::getInstance()->put(PEAR_LOG_DEBUG, "activePlaginInner> : ptypes [%s]", print_r($ptypes, true));
					foreach(explode(',', $ar[2]) as $item) {
						$func = trim($item);
						//foreach(array('function', 'block', 'compiler', 'modifier') as $type) {
						foreach($ptypes as $t) {
							$type = trim($t);
							$func_name = $type . '_' . $func;
							if(function_exists($func_name)) {
								$name = $prefix . $func;
								$this->smarty->registerPlugin($type, $name, $func_name);
								//SimgLog::getInstance()->put(PEAR_LOG_DEBUG, "activePlagin : %s [%s]-[%s]", $type, $name, $func_name);
							}
						}
					}
				}
			}
		}
	}
	/**
	 * ホーム遷移
	 */
	public function toHome() {
		$this->setTemplate($this->getHomeTemplate());
	}
	/**
	 * システムエラー遷移
	 */
	public function toSystemError() {
		$this->setTemplate($this->getSystemErrorTemplate());
	}
	/**
	 * エラーコード設定
	 * @param $err エラーコード
	 */
	public function setError($err) {
		$this->error[_ERROR_INDEX] = $err;
	}
	/**
	 * エラーコード取得
	 * @return エラーコード
	 */
	public function getError() {
		return simg_get_array_value($this->error, _ERROR_INDEX, null);
	}
	/**
	 * エラーコード判定
	 * @return true/false=エラーあり/なし
	 */
	public function isExistError() {
		return !is_null($this->getError());
	}
	/**
	 * エラー情報設定
	 * @param $info エラーコード
	 */
	public function setErrorInfo($info) {
		$this->error[_INFO_INDEX] = $err;
	}
	/**
	 * エラー情報取得
	 * @return エラー情報
	 */
	public function getErrorInfo() {
		return simg_get_array_value($this->error, _INFO_INDEX, null);
	}
	/**
	 * エラー情報有無判定
	 * @return true/false=エラー情報あり/なし
	 */
	public function isExistErrorInfo() {
		return !is_null($this->getErrorInfo());
	}
	/**
	 * エラー処理
	 * @param $e 例外
	 */
	public function error($e) {
		$this->toSystemError();
		$this->assign(SIMG_ASSIGN_ERROR, $e->getMessage());
	}
	/**
	 * レンダリング前処理
	 * @param $event イベント
	 * @param $parameter パラメータ
	 * @param $db DB
	 */
	public function preRendering($event, $parameter, $db) {
		if($event == SIMG_EVENT_SCRIPT) {
			$src = simg_get_value($parameter->get(SIMG_PARAM_SRC), '');
			if(strlen($src) > 0) {
				$this->setTemplate($src);
				//$this->page[$src] = explode(',', simg_get_value($parameter->get(SIMG_PARAM_PARTIAL), ''));
			}
		}
	}
}

?>

