/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 文字列を基本的なデータ型とその集合に変換する機能を提供します。<br>
 * shohaku.core.lang.ValueOf からの転用です。
 */
final class ValueOf {

    /* 負数記号。 */
    private static final String NEGATIVE_SIGN = "-";

    /* 日付を解析、要素の抽出を行う正規表現。 */
    private static final Pattern dateTimePattern;

    static {
        // 2000-01-01[T| ]23:01:01.000 +00:00
        final StringBuffer p = new StringBuffer();
        p.append("^");
        p.append("(?:(\\d{4})-(\\d{2})-(\\d{2}))?");
        p.append("(T|\\u0020+)?");
        p.append("(?:(?:(\\d{2}):(\\d{2}):(\\d{2}))(?:(?:\\.(\\d{3}))?)?(?:(?:\\u0020*)(?:([-+])(\\d{2}):(\\d{2})))?)?(l|L|d|D|g|G)?");
        p.append("$");
        dateTimePattern = Pattern.compile(p.toString());
    }

    /*
     * decode
     */

    /**
     * Unicode escapes の文字表現を文字型へ変換して返却します。
     * 
     * @param src
     *            Unicode escapes 形式の文字列
     * @return デコードされた文字
     * @throws IllegalArgumentException
     *             構文が不正の場合
     */
    public static Character decodeCharacter(String src) {
        if (src == null) {
            throw newIllegalArgumentEx("Malformed character encoding.", src, 0);
        }
        final int srcLength = src.length();
        if (srcLength == 0) {
            throw newIllegalArgumentEx("Malformed character encoding.", src, 0);
        }
        char ch;
        ch = src.charAt(0);
        if (ch == '\\') {
            if (srcLength < 2) {
                throw newIllegalArgumentEx("Malformed character encoding.", src, 0);
            }
            ch = src.charAt(1);
            switch (ch) {
            case 0x30: // 0
            case 0x31: // 1
            case 0x32: // 2
            case 0x33: // 3
            case 0x34: // 4
            case 0x35: // 5
            case 0x36: // 6
            case 0x37: // 7
                return Boxing.box(decodeOctalCharacter(src, 1));
            case 0x75: // u
                return Boxing.box(decodeUnicodeCharacter(src, 2));
            case 0x78: // x
                return Boxing.box(decodeHexCharacter(src, 2));
            default:
                return Boxing.box(decodeCntrlCharacter(src, ch, 1));
            }
        } else {
            if (srcLength != 1) {
                throw newIllegalArgumentEx("Malformed 'x' encoding.", src, 1);
            }
            return Boxing.box(ch);
        }
    }

    /**
     * 文字列を日付オブジェクトに変換して返却します。<br>
     * 有効な書式は 'yyyy-mm-dd[T| ]hh:mi:ss.zzz[+|-]hh:mi[l|L|g|G|d|D]' です。<br>
     * 日付と時刻の両方または片方の指定が可能です。またミリ秒とタイムゾーン、型接尾辞は省略可能です。<br>
     * 型接尾辞 'l' or 'L' を指定すると日付の結果は、グリニッジ標準日時（1970-01-01）からのミリ秒を示す Long 型で返却されます。<br>
     * 型接尾辞 'g' or 'G' を指定すると結果は、グレゴリオカレンダー型で返却されます。<br>
     * 型接尾辞 'd' or 'D' を指定すると結果は、日付オブジェクトで返却されます。<br>
     * 省略時は、日付オブジェクトで返却されます。<br>
     * <br>
     * 有効な書式例：<br>
     * "2007-05-23T09:51:29.345+09:00"<br>
     * "2007-05-23 09:51:29.345-09:00L"<br>
     * "2007-05-23 09:51:29"<br>
     * "2007-05-23 +05:30"<br>
     * "2007-05-23"<br>
     * "09:51:29.345+09:00"<br>
     * "09:51:29 +09:00L"<br>
     * 
     * @param src
     *            生成元の文字列
     * @return 日付オブジェクト
     * @throws IllegalArgumentException
     *             構文が不正の場合
     */
    public static Object decodeDateTime(String src) {
        if (src.length() == 0) {
            throw newIllegalArgumentEx("pattern err.", src, 0);
        }
        // 書式が有効ならば、構文用の書式に束ねる
        final Matcher match = dateTimePattern.matcher(src);
        if (match.find()) {
            final String year = match.group(1);
            final String month = match.group(2);
            final String day = match.group(3);
            final String deli = match.group(4);
            final String hour = match.group(5);
            final String minute = match.group(6);
            final String second = match.group(7);
            final String nano = match.group(8);
            final String zoneSign = match.group(9);
            final String zoneHour = match.group(10);
            final String zoneMinute = match.group(11);
            final String type = match.group(12);
            // 日付と時刻が指定された場合は区切り必須
            if ((year != null && hour != null) && deli == null) {
                throw newIllegalArgumentEx("pattern err.", src, 0);
            }
            // 区切りの有る場合は日付と時刻は必須
            if (deli != null && (year == null || hour == null)) {
                throw newIllegalArgumentEx("pattern err.", src, 0);
            }

            final int y;
            final int m;
            final int d;
            if (year != null) {
                y = Integer.parseInt(year);
                m = Integer.parseInt(month) - 1;
                d = Integer.parseInt(day);
            } else {
                y = 1970;
                m = 0;
                d = 1;
            }
            final int h;
            final int i;
            final int s;
            final int n;
            if (hour != null) {
                h = Integer.parseInt(hour);
                i = Integer.parseInt(minute);
                s = Integer.parseInt(second);
                n = (nano != null) ? Integer.parseInt(nano) : 0;
            } else {
                h = 0;
                i = 0;
                s = 0;
                n = 0;
            }

            final GregorianCalendar cal = new GregorianCalendar();
            cal.set(Calendar.YEAR, y);
            cal.set(Calendar.MONTH, m);
            cal.set(Calendar.DATE, d);
            cal.set(Calendar.HOUR_OF_DAY, h);
            cal.set(Calendar.MINUTE, i);
            cal.set(Calendar.SECOND, s);
            cal.set(Calendar.MILLISECOND, n);
            if (zoneSign != null) {
                final int zh = Integer.parseInt(zoneHour);
                final int zm = Integer.parseInt(zoneMinute);
                final int zone = (zh * 60 * 60 * 1000) + (zm * 60 * 1000);
                final int z = (zoneSign.equals("+")) ? zone : -zone;
                cal.set(Calendar.ZONE_OFFSET, z);
            }
            if (type != null) {
                final int _type = toUpper(type.charAt(0));
                switch (_type) {
                case 0x44: // D
                    return cal.getTime();
                case 0x47: // G
                    return cal;
                case 0x4C: // L
                    return Boxing.box(cal.getTimeInMillis());
                default:
                    break;
                }
            }
            // type is null
            return cal.getTime();
        }
        throw newIllegalArgumentEx("pattern err.", src, 0);
    }

    /**
     * 拡張規則で文字列を Integer に変換します。<br>
     * 10進数，16進数（[0x|0X]n），又は8進数（[0]n），2進数（[0b|0B]n）, 接尾辞（[i|I]）の表記法をサポートします。
     * 
     * @param src
     *            生成元の文字列
     * @return 数値オブジェクト
     */
    public static Integer decodeInteger(String src) {
        int index = 0;
        boolean negative = false;
        if (src.startsWith(NEGATIVE_SIGN)) {
            negative = true;
            index++;
        }
        int radix = 10;
        if (src.startsWith("0x", index) || src.startsWith("0X", index)) {
            index += 2;
            radix = 16;
        } else if (src.startsWith("0b", index) || src.startsWith("0B", index)) {
            index += 2;
            radix = 2;
        } else if (src.startsWith("0", index) && src.length() > 1 + index) {
            index++;
            radix = 8;
        }
        final int suffix = (src.endsWith("i") || src.endsWith("I")) ? src.length() - 1 : src.length();
        final String _num = (negative) ? NEGATIVE_SIGN.concat(src.substring(index, suffix)) : src.substring(index, suffix);
        return Integer.valueOf(_num, radix);
    }

    /**
     * 拡張規則で文字列を数値オブジェクトに変換します。<br>
     * 整数に対しては、10進数，16進数（[0x|0X]n），又は8進数（[0]n），2進数（[0b|0B]n）の表記法をサポートします。<br>
     * また, 型指定子として接尾辞（[u|U|s|S|i|I|l|L|f|F|d|D|h|H|g|G]）の表記法をサポートします。<br>
     * <br>
     * 型指定子の表記規則：<br>
     * &nbsp;&nbsp;1.&nbsp;u or U = Byte<br>
     * &nbsp;&nbsp;2.&nbsp;s or S = Short<br>
     * &nbsp;&nbsp;3.&nbsp;i or I = Integer<br>
     * &nbsp;&nbsp;4.&nbsp;l or L = Long<br>
     * &nbsp;&nbsp;5.&nbsp;f or F = Float<br>
     * &nbsp;&nbsp;6.&nbsp;d or D = Double<br>
     * &nbsp;&nbsp;7.&nbsp;h or H = BigInteger<br>
     * &nbsp;&nbsp;8.&nbsp;g or G = BigDecimal<br>
     * &nbsp;&nbsp;9.&nbsp;接尾辞が省略された整数型 = Integer<br>
     * &nbsp;10.&nbsp;接尾辞が省略された浮動小数点型 = Double<br>
     * 
     * @param src
     *            生成元の文字列
     * @return 数値オブジェクト
     */
    public static Number decodeNumber(String src) {
        final int srcLength = src.length();
        if (srcLength == 1) {
            return Integer.valueOf(src, 10); // 一文字、十進数整数
        }
        // 符号と接尾辞を検出
        final int negative = (src.startsWith(NEGATIVE_SIGN)) ? 1 : 0;
        final int suffix = toUpper(src.charAt(srcLength - 1));
        // 浮動小数点表記か検証
        if (-1 < src.indexOf('.', negative)) {
            // 浮動小数点表記
            switch (suffix) {
            case 0x44: // D
                return Double.valueOf(numstr(negative, src, 0, true));
            case 0x46: // F
                return Float.valueOf(numstr(negative, src, 0, true));
            case 0x47: // G
                return new BigDecimal(numstr(negative, src, 0, true));
            default:
                return Double.valueOf(numstr(negative, src, 0, false));
            }
        }
        // 整数表記の二進数、八進数または十六進数か検証
        if (src.startsWith("0", negative)) {
            final int radixNm = toUpper(src.charAt(negative + 1));
            // 十六進数
            if ('X' == radixNm) {
                switch (suffix) {
                case 0x48: // H
                    return new BigInteger(numstr(negative, src, 16, true), 16);
                case 0x49: // I
                    return Integer.valueOf(numstr(negative, src, 16, true), 16);
                case 0x4C: // L
                    return Long.valueOf(numstr(negative, src, 16, true), 16);
                case 0x53: // S
                    return Short.valueOf(numstr(negative, src, 16, true), 16);
                case 0x55: // U
                    return Byte.valueOf(numstr(negative, src, 16, true), 16);
                default:
                    return Integer.valueOf(numstr(negative, src, 16, false), 16);
                }
            }
            // 二進数
            if ('B' == radixNm) {
                switch (suffix) {
                case 0x48: // H
                    return new BigInteger(numstr(negative, src, 2, true), 2);
                case 0x49: // I
                    return Integer.valueOf(numstr(negative, src, 2, true), 2);
                case 0x4C: // L
                    return Long.valueOf(numstr(negative, src, 2, true), 2);
                case 0x53: // S
                    return Short.valueOf(numstr(negative, src, 2, true), 2);
                case 0x55: // U
                    return Byte.valueOf(numstr(negative, src, 2, true), 2);
                default:
                    return Integer.valueOf(numstr(negative, src, 2, false), 2);
                }
            }
            // '0' + 接尾辞
            if (src.length() == 2 + negative) {
                switch (suffix) {
                case 0x44: // D
                    return new Double((negative == 0) ? 0.0 : -0.0);
                case 0x46: // F
                    return new Float((negative == 0) ? 0.0f : -0.0f);
                case 0x47: // G
                    return new BigDecimal(BigInteger.ZERO, 0);
                case 0x48: // H
                    return BigInteger.ZERO;
                case 0x49: // I
                    return new Integer(0);
                case 0x4C: // L
                    return new Long(0L);
                case 0x53: // S
                    return new Short((short) 0);
                case 0x55: // U
                    return new Byte((byte) 0);
                default:
                    break;
                }
            }
            // 八進数
            switch (suffix) {
            case 0x48: // H
                return new BigInteger(numstr(negative, src, 8, true), 8);
            case 0x49: // I
                return Integer.valueOf(numstr(negative, src, 8, true), 8);
            case 0x4C: // L
                return Long.valueOf(numstr(negative, src, 8, true), 8);
            case 0x53: // S
                return Short.valueOf(numstr(negative, src, 8, true), 8);
            case 0x55: // U
                return Byte.valueOf(numstr(negative, src, 8, true), 8);
            default:
                return Integer.valueOf(numstr(negative, src, 8, false), 8);
            }
        }

        // 整数表記の十進数、科学表記、無限大、非数
        switch (suffix) {
        case 0x44: // D
            return Double.valueOf(numstr(negative, src, 0, true));
        case 0x46: // F
            return Float.valueOf(numstr(negative, src, 0, true));
        case 0x47: // G
            return new BigDecimal(numstr(negative, src, 0, true));
        case 0x48: // H
            return new BigInteger(numstr(negative, src, 10, true), 10);
        case 0x49: // I
            return Integer.valueOf(numstr(negative, src, 10, true), 10);
        case 0x4C: // L
            return Long.valueOf(numstr(negative, src, 10, true), 10);
        case 0x53: // S
            return Short.valueOf(numstr(negative, src, 10, true), 10);
        case 0x55: // U
            return Byte.valueOf(numstr(negative, src, 10, true), 10);
        default:
            if (src.startsWith("I", negative)) {// 無限大
                return Double.valueOf(src);
            }
            if (src.startsWith("N", negative)) {// 非数
                return Double.valueOf(src);
            }
            if (-1 < src.indexOf('e', negative) || -1 < src.indexOf('E', negative)) {// 科学表記
                return Double.valueOf(numstr(negative, src, 0, false));
            }
            return Integer.valueOf(numstr(negative, src, 10, false), 10);// 整数表記の十進数
        }
    }

    /**
     * Unicode escapes の文字表現を文字列型へ変換して返却します。
     * 
     * @param src
     *            Unicode escapes 形式の文字列
     * @return デコードされた文字列
     * @throws IllegalArgumentException
     *             構文が不正の場合
     */
    public static String decodeString(String src) {
        if (src == null) {
            throw newIllegalArgumentEx("Malformed character encoding.", src, 0);
        }
        final int srcLength = src.length();
        final StringBuffer sb = new StringBuffer(srcLength);
        for (int i = 0; i < srcLength; i++) {
            final char ch = src.charAt(i);
            if (ch == '\\') {
                if (!(i + 1 < srcLength)) {
                    throw newIllegalArgumentEx("Malformed character encoding.", src, i);
                }
                final char next = src.charAt(i + 1);
                switch (next) {
                case 0x30: // 0
                case 0x31: // 1
                case 0x32: // 2
                case 0x33: // 3
                case 0x34: // 4
                case 0x35: // 5
                case 0x36: // 6
                case 0x37: // 7
                    sb.append(decodeOctalCharacter(src, i + 1));
                    i += 3;
                    break;
                case 0x75: // u
                    sb.append(decodeUnicodeCharacter(src, i + 2));
                    i += 5;
                    break;
                case 0x78: // x
                    sb.append(decodeHexCharacter(src, i + 2));
                    i += 3;
                    break;
                default:
                    sb.append(decodeCntrlCharacter(src, next, i));
                    i += 1;
                    break;
                }
            } else {
                sb.append(ch);
            }
        }
        return sb.toString();
    }

    /*
     * private
     */

    private static char decodeCntrlCharacter(CharSequence src, char ch, int index) {
        switch (ch) {
        case 0x22: // "
            return '\"';
        case 0x27: // '
            return '\'';
        case 0x5C: // \
            return '\\';
        case 0x60: // `
            return '`';
        case 0x62: // b
            return '\b';
        case 0x66: // f
            return '\f';
        case 0x6E: // n
            return '\n';
        case 0x72: // r
            return '\r';
        case 0x74: // t
            return '\t';
        default:
            throw newIllegalArgumentEx("Malformed \\X encoding.", src, index);
        }
    }

    private static char decodeOctalCharacter(CharSequence src, int index) {
        if ((index + 3) > src.length()) {
            throw newIllegalArgumentEx("Malformed \\XXX encoding.", src, index);
        }
        int value = 0;
        char ch;
        for (int i = index; i < (index + 3); i++) {
            ch = src.charAt(i);
            if (i == index) {
                switch (ch) {
                case 0x30: // 0
                case 0x31: // 1
                case 0x32: // 2
                case 0x33: // 3
                    value = (value << 3) + ch - '0';
                    break;
                default:
                    throw newIllegalArgumentEx("Malformed \\XXX encoding.", src, index);
                }
            } else {
                switch (ch) {
                case 0x30: // 0
                case 0x31: // 1
                case 0x32: // 2
                case 0x33: // 3
                case 0x34: // 4
                case 0x35: // 5
                case 0x36: // 6
                case 0x37: // 7
                    value = (value << 3) + ch - '0';
                    break;
                default:
                    throw newIllegalArgumentEx("Malformed \\XXX encoding.", src, index);
                }
            }
        }
        return (char) value;
    }

    private static char decodeHexCharacter(CharSequence src, int index) {
        if ((index + 2) > src.length()) {
            throw newIllegalArgumentEx("Malformed \\xXX encoding.", src, index);
        }
        int value = 0;
        char ch;
        for (int i = index; i < (index + 2); i++) {
            ch = src.charAt(i);
            switch (ch) {
            case 0x30: // 0
            case 0x31: // 1
            case 0x32: // 2
            case 0x33: // 3
            case 0x34: // 4
            case 0x35: // 5
            case 0x36: // 6
            case 0x37: // 7
            case 0x38: // 8
            case 0x39: // 9
                value = (value << 4) + ch - '0';
                break;
            case 0x41: // A
            case 0x42: // B
            case 0x43: // C
            case 0x44: // D
            case 0x45: // E
            case 0x46: // F
                value = (value << 4) + 10 + ch - 'A';
                break;
            case 0x61: // a
            case 0x62: // b
            case 0x63: // c
            case 0x64: // d
            case 0x65: // e
            case 0x66: // f
                value = (value << 4) + 10 + ch - 'a';
                break;
            default:
                throw newIllegalArgumentEx("Malformed \\xXX encoding.", src, index);
            }
        }
        return (char) value;
    }

    private static char decodeUnicodeCharacter(CharSequence src, int index) {
        if ((index + 4) > src.length()) {
            throw newIllegalArgumentEx("Malformed \\uXXXX encoding.", src, index);
        }
        int value = 0;
        char ch;
        for (int i = index; i < (index + 4); i++) {
            ch = src.charAt(i);
            switch (ch) {
            case 0x30: // 0
            case 0x31: // 1
            case 0x32: // 2
            case 0x33: // 3
            case 0x34: // 4
            case 0x35: // 5
            case 0x36: // 6
            case 0x37: // 7
            case 0x38: // 8
            case 0x39: // 9
                value = (value << 4) + ch - '0';
                break;
            case 0x41: // A
            case 0x42: // B
            case 0x43: // C
            case 0x44: // D
            case 0x45: // E
            case 0x46: // F
                value = (value << 4) + 10 + ch - 'A';
                break;
            case 0x61: // a
            case 0x62: // b
            case 0x63: // c
            case 0x64: // d
            case 0x65: // e
            case 0x66: // f
                value = (value << 4) + 10 + ch - 'a';
                break;
            default:
                throw newIllegalArgumentEx("Malformed \\uXXXX encoding.", src, index);
            }
        }
        return (char) value;
    }

    private static IllegalArgumentException newIllegalArgumentEx(String msg, Object src, int index) {
        throw new IllegalArgumentException(msg + "　src=" + src + ", pos=" + index);
    }

    /* Javaで認識できる形式で数値の文字列を構築する */
    private static String numstr(int negative, String src, int radix, boolean hasSuffix) {
        final int endIndex = (hasSuffix) ? (src.length() - 1) : src.length();
        final String _num;
        switch (radix) {
        case 2:
            _num = src.substring(negative + 2, endIndex);
            break;
        case 8:
            _num = src.substring(negative + 1, endIndex);
            break;
        case 10:
            _num = src.substring(negative + 0, endIndex);
            break;
        case 16:
            _num = src.substring(negative + 2, endIndex);
            break;
        default:
            return src.substring(0, endIndex);
        }
        return (negative == 1) ? NEGATIVE_SIGN.concat(_num) : _num;
    }

    /* 指定の文字が小文字アルファベットの場合は大文字に変換します。 */
    private static int toUpper(int ch) {
        return ((ch - 0x61) | (0x7A - ch)) >= 0 ? (ch - 0x20) : ch;
    }

}
