/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
/**
 * 
 */
package shohaku.ogdl;

/**
 * OGDLの構文例外を提供します。
 */
public class OgdlSyntaxException extends RuntimeException {

    /* serialVersionUID */
    private static final long serialVersionUID = -8246970984193657017L;

    /* 解析イベント */
    private OgdlEvent _event;

    /**
     * デフォルトコンストラクタ。
     */
    public OgdlSyntaxException() {
        super();
    }

    /**
     * エラーメッセージを設定して初期化します。
     * 
     * @param message
     *            エラーメッセージ
     */
    public OgdlSyntaxException(String message) {
        super(message);
    }

    /**
     * エラーメッセージと起因例外を設定して初期化します。
     * 
     * @param message
     *            エラーメッセージ
     * @param cause
     *            起因例外
     */
    public OgdlSyntaxException(String message, Throwable cause) {
        super(message, cause);
    }

    /**
     * イベントとエラーメッセージを設定して初期化します。
     * 
     * @param event
     *            解析イベント
     * @param message
     *            エラーメッセージ
     */
    public OgdlSyntaxException(OgdlEvent event, String message) {
        super(message);
        this._event = event;
    }

    /**
     * イベントとエラーメッセージと起因例外を設定して初期化します。
     * 
     * @param event
     *            解析イベント
     * @param message
     *            エラーメッセージ
     * @param cause
     *            起因例外
     */
    public OgdlSyntaxException(OgdlEvent event, String message, Throwable cause) {
        super(message, cause);
        this._event = event;
    }

    /**
     * イベントを返却します。
     * 
     * @return イベント
     */
    OgdlEvent getEvent() {
        return this._event;
    }

    /**
     * イベントを格納します。
     * 
     * @param event
     *            イベント
     */
    void setEvent(OgdlEvent event) {
        this._event = event;
    }

    /**
     * 例外発生時のインデックスを返却します。<br>
     * インデックスが不明の場合は -1 が返却されます。
     * 
     * @return 例外発生時のインデックス
     */
    public int getIndex() {
        return (_event == null) ? -1 : _event.index();
    }

    /**
     * 例外発生時の式構文を返却します。get
     * 
     * @return 例外発生時の式構文
     */
    public String getPattern() {
        return (_event == null) ? null : _event.ptn;
    }

    /**
     * 例外発生時のクラスローダを返却します。
     * 
     * @return 例外発生時のクラスローダ
     */
    public ClassLoader getClassLoader() {
        return (_event == null) ? null : _event.loader;
    }

    /**
     * 例外発生時のコンテキストを返却します。
     * 
     * @return 例外発生時のコンテキスト
     */
    public OgdlContext getContext() {
        return (_event == null) ? null : _event.context;
    }

    /**
     * 例外発生時の処理対象のオブジェクトを返却します。
     * 
     * @return 例外発生時の処理対象のオブジェクト
     */
    public Object getTargetObject() {
        return (_event == null) ? null : _event.target;
    }

    /**
     * プロパティを設定して自身の参照を返却します。
     * 
     * @param event
     *            イベント
     * @return 自身の参照
     */
    OgdlSyntaxException throwFor(OgdlEvent event) {
        this.setEvent(event);
        return this;
    }

    /**
     * super.getMessage(), event={pattern=getPattern(),offset=getOgdlParseIndex(),target=getTargetObject()} の表現で返却します。
     * 
     * @return このオブジェクトの文字列表現
     * @see java.lang.Object#toString()
     */
    public String toString() {
        final StringBuffer msg = new StringBuffer();
        msg.append(super.getMessage());
        msg.append(", event=");
        msg.append(_event);
        return msg.toString();
    }

}
