/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package shohaku.ogdl;

/**
 * 構文解析の共通的な機能を提供します。
 */
class OgdlSyntax {

    /** 大文字アルファベット。 */
    static final int UPPER = 0x00000001;

    /** 小文字アルファベット。 */
    static final int LOWER = 0x00000002;

    /** ２進数字。 */
    static final int BINARY = 0x00000004;

    /** ８進数字。 */
    static final int OCTAL = 0x00000008;

    /** １０進数字。 */
    static final int DIGIT = 0x00000010;

    /** １６進数字。 */
    static final int HEX = 0x00000020;

    /** スペース文字（\x20\t\n\x0B\f\r）。 */
    static final int SPACE = 0x00000040;

    /** 記号文字。 */
    static final int PUNCT = 0x00000080;

    /** 制御文字。 */
    static final int CNTRL = 0x00000100;

    /** 余白文字（\x20\t）。 */
    static final int BLANK = 0x00000200;

    /** 印刷可能文字。 */
    static final int PRINT = 0x00000400;

    /** アンダーバー。 */
    static final int UNDER = 0x0000800;

    /** アルファベット。 */
    static final int ALPHA = (UPPER | LOWER);

    /** アルファベットと数字。 */
    static final int ALNUM = (UPPER | LOWER | DIGIT);

    /** グリフ文字。 */
    static final int GRAPH = (PUNCT | UPPER | LOWER | DIGIT);

    /** 単語を構成する文字。 */
    static final int WORD = (UPPER | LOWER | UNDER | DIGIT);

    /** 浮動小数点数値。 */
    static final int FLOAT = 0x0001000;

    /** ドット。 */
    static final int DOT = 0x0002000;

    /** ドル。 */
    static final int DOLLAR = 0x0004000;

    /** コロン。 */
    static final int COLON = 0x0008000;

    /** マイナス。 */
    static final int MINUS = 0x0010000;

    /** 数値の有効な開始文字。 */
    static final int NUMHD = 0x0020000;

    /** 数値の有効な型指定子。 */
    static final int NUMTL = 0x0040000;

    /** メンバー（フィールド名、メソッド名、クラス名）。 */
    static final int MEMBER = (UPPER | LOWER | DIGIT | UNDER);

    /** メンバー（フィールド名、メソッド名、クラス名）の開始文字。 */
    static final int MENHD = (UPPER | LOWER | UNDER);

    /** パッケージを含む完全クラス名。 */
    static final int CLAZZ = (UPPER | LOWER | DIGIT | UNDER | DOT | DOLLAR);

    /** 数値。 */
    static final int NUMBER = (BINARY | OCTAL | DIGIT | HEX | FLOAT | MINUS);

    /** 拡張関数名。 */
    static final int FUNC = (UPPER | LOWER | DIGIT | UNDER | COLON);

    /** オペレータ。 */
    static final int OPERATOR = (PUNCT | LOWER);

    /** 等差数列の差値。 */
    static final int DIFFERENCE = (UPPER | LOWER | UNDER | BINARY | OCTAL | DIGIT | HEX | FLOAT | MINUS);

    /** 文字をマッピングするビットセット。 */
    private static final int[] CHAR_CLAZZ = new int[0xFF];
    static {
        CHAR_CLAZZ[0x00] = CNTRL; // (NUL)
        CHAR_CLAZZ[0x01] = CNTRL; // (SOH)
        CHAR_CLAZZ[0x02] = CNTRL; // (STX)
        CHAR_CLAZZ[0x03] = CNTRL; // (ETX)
        CHAR_CLAZZ[0x04] = CNTRL; // (EOT)
        CHAR_CLAZZ[0x05] = CNTRL; // (ENQ)
        CHAR_CLAZZ[0x06] = CNTRL; // (ACK)
        CHAR_CLAZZ[0x07] = CNTRL; // (BEL)
        CHAR_CLAZZ[0x08] = CNTRL; // (BS)
        CHAR_CLAZZ[0x09] = SPACE + CNTRL + BLANK; // (HT)
        CHAR_CLAZZ[0x0A] = SPACE + CNTRL; // (LF)
        CHAR_CLAZZ[0x0B] = SPACE + CNTRL; // (VT)
        CHAR_CLAZZ[0x0C] = SPACE + CNTRL; // (FF)
        CHAR_CLAZZ[0x0D] = SPACE + CNTRL; // (CR)
        CHAR_CLAZZ[0x0E] = CNTRL; // (SI)
        CHAR_CLAZZ[0x0F] = CNTRL; // (SO)
        CHAR_CLAZZ[0x10] = CNTRL; // (DLE)
        CHAR_CLAZZ[0x11] = CNTRL; // (DC1)
        CHAR_CLAZZ[0x12] = CNTRL; // (DC2)
        CHAR_CLAZZ[0x13] = CNTRL; // (DC3)
        CHAR_CLAZZ[0x14] = CNTRL; // (DC4)
        CHAR_CLAZZ[0x15] = CNTRL; // (NAK)
        CHAR_CLAZZ[0x16] = CNTRL; // (SYN)
        CHAR_CLAZZ[0x17] = CNTRL; // (ETB)
        CHAR_CLAZZ[0x18] = CNTRL; // (CAN)
        CHAR_CLAZZ[0x19] = CNTRL; // (EM)
        CHAR_CLAZZ[0x1A] = CNTRL; // (SUB)
        CHAR_CLAZZ[0x1B] = CNTRL; // (ESC)
        CHAR_CLAZZ[0x1C] = CNTRL; // (FS)
        CHAR_CLAZZ[0x1D] = CNTRL; // (GS)
        CHAR_CLAZZ[0x1E] = CNTRL; // (RS)
        CHAR_CLAZZ[0x1F] = CNTRL; // (US)
        CHAR_CLAZZ[0x20] = PRINT + SPACE + BLANK; // SPACE
        CHAR_CLAZZ[0x21] = PRINT + PUNCT; // !
        CHAR_CLAZZ[0x22] = PRINT + PUNCT; // "
        CHAR_CLAZZ[0x23] = PRINT + PUNCT; // #
        CHAR_CLAZZ[0x24] = PRINT + PUNCT + DOLLAR; // $
        CHAR_CLAZZ[0x25] = PRINT + PUNCT; // %
        CHAR_CLAZZ[0x26] = PRINT + PUNCT; // &
        CHAR_CLAZZ[0x27] = PRINT + PUNCT; // '
        CHAR_CLAZZ[0x28] = PRINT + PUNCT; // (
        CHAR_CLAZZ[0x29] = PRINT + PUNCT; // )
        CHAR_CLAZZ[0x2A] = PRINT + PUNCT; // *
        CHAR_CLAZZ[0x2B] = PRINT + PUNCT + FLOAT; // +
        CHAR_CLAZZ[0x2C] = PRINT + PUNCT; // ,
        CHAR_CLAZZ[0x2D] = PRINT + PUNCT + FLOAT + MINUS; // -
        CHAR_CLAZZ[0x2E] = PRINT + PUNCT + FLOAT + DOT; // .
        CHAR_CLAZZ[0x2F] = PRINT + PUNCT; // /
        CHAR_CLAZZ[0x30] = PRINT + NUMHD + DIGIT + FLOAT + HEX + OCTAL + BINARY; // 0
        CHAR_CLAZZ[0x31] = PRINT + NUMHD + DIGIT + FLOAT + HEX + OCTAL + BINARY; // 1
        CHAR_CLAZZ[0x32] = PRINT + NUMHD + DIGIT + FLOAT + HEX + OCTAL; // 2
        CHAR_CLAZZ[0x33] = PRINT + NUMHD + DIGIT + FLOAT + HEX + OCTAL; // 3
        CHAR_CLAZZ[0x34] = PRINT + NUMHD + DIGIT + FLOAT + HEX + OCTAL; // 4
        CHAR_CLAZZ[0x35] = PRINT + NUMHD + DIGIT + FLOAT + HEX + OCTAL; // 5
        CHAR_CLAZZ[0x36] = PRINT + NUMHD + DIGIT + FLOAT + HEX + OCTAL; // 6
        CHAR_CLAZZ[0x37] = PRINT + NUMHD + DIGIT + FLOAT + HEX + OCTAL; // 7
        CHAR_CLAZZ[0x38] = PRINT + NUMHD + DIGIT + FLOAT + HEX; // 8
        CHAR_CLAZZ[0x39] = PRINT + NUMHD + DIGIT + FLOAT + HEX; // 9
        CHAR_CLAZZ[0x3A] = PRINT + PUNCT + COLON; // :
        CHAR_CLAZZ[0x3B] = PRINT + PUNCT; // ;
        CHAR_CLAZZ[0x3C] = PRINT + PUNCT; // <
        CHAR_CLAZZ[0x3D] = PRINT + PUNCT; // =
        CHAR_CLAZZ[0x3E] = PRINT + PUNCT; // >
        CHAR_CLAZZ[0x3F] = PRINT + PUNCT; // ?
        CHAR_CLAZZ[0x40] = PRINT + PUNCT; // @
        CHAR_CLAZZ[0x41] = PRINT + UPPER + HEX; // A
        CHAR_CLAZZ[0x42] = PRINT + UPPER + HEX + BINARY; // B
        CHAR_CLAZZ[0x43] = PRINT + UPPER + HEX; // C
        CHAR_CLAZZ[0x44] = PRINT + UPPER + HEX + NUMTL; // D
        CHAR_CLAZZ[0x45] = PRINT + UPPER + HEX + FLOAT; // E
        CHAR_CLAZZ[0x46] = PRINT + UPPER + HEX + NUMTL; // F
        CHAR_CLAZZ[0x47] = PRINT + UPPER + NUMTL; // G
        CHAR_CLAZZ[0x48] = PRINT + UPPER + NUMTL; // H
        CHAR_CLAZZ[0x49] = PRINT + UPPER + FLOAT + NUMHD + NUMTL; // I
        CHAR_CLAZZ[0x4A] = PRINT + UPPER; // J
        CHAR_CLAZZ[0x4B] = PRINT + UPPER; // K
        CHAR_CLAZZ[0x4C] = PRINT + UPPER + NUMTL; // L
        CHAR_CLAZZ[0x4D] = PRINT + UPPER; // M
        CHAR_CLAZZ[0x4E] = PRINT + UPPER + FLOAT + NUMHD; // N
        CHAR_CLAZZ[0x4F] = PRINT + UPPER; // O
        CHAR_CLAZZ[0x50] = PRINT + UPPER; // P
        CHAR_CLAZZ[0x51] = PRINT + UPPER; // Q
        CHAR_CLAZZ[0x52] = PRINT + UPPER; // R
        CHAR_CLAZZ[0x53] = PRINT + UPPER + NUMTL; // S
        CHAR_CLAZZ[0x54] = PRINT + UPPER; // T
        CHAR_CLAZZ[0x55] = PRINT + UPPER + NUMTL; // U
        CHAR_CLAZZ[0x56] = PRINT + UPPER; // V
        CHAR_CLAZZ[0x57] = PRINT + UPPER; // W
        CHAR_CLAZZ[0x58] = PRINT + UPPER + HEX; // X
        CHAR_CLAZZ[0x59] = PRINT + UPPER; // Y
        CHAR_CLAZZ[0x5A] = PRINT + UPPER; // Z
        CHAR_CLAZZ[0x5B] = PRINT + PUNCT; // [
        CHAR_CLAZZ[0x5C] = PRINT + PUNCT; // \
        CHAR_CLAZZ[0x5D] = PRINT + PUNCT; // ]
        CHAR_CLAZZ[0x5E] = PRINT + PUNCT; // ^
        CHAR_CLAZZ[0x5F] = PRINT + PUNCT + UNDER; // _
        CHAR_CLAZZ[0x60] = PRINT + PUNCT; // `
        CHAR_CLAZZ[0x61] = PRINT + LOWER + HEX + FLOAT; // a
        CHAR_CLAZZ[0x62] = PRINT + LOWER + HEX + BINARY; // b
        CHAR_CLAZZ[0x63] = PRINT + LOWER + HEX; // c
        CHAR_CLAZZ[0x64] = PRINT + LOWER + HEX + NUMTL; // d
        CHAR_CLAZZ[0x65] = PRINT + LOWER + HEX + FLOAT; // e
        CHAR_CLAZZ[0x66] = PRINT + LOWER + HEX + FLOAT + NUMTL; // f
        CHAR_CLAZZ[0x67] = PRINT + LOWER + NUMTL; // g
        CHAR_CLAZZ[0x68] = PRINT + LOWER + NUMTL; // h
        CHAR_CLAZZ[0x69] = PRINT + LOWER + FLOAT + NUMTL; // i
        CHAR_CLAZZ[0x6A] = PRINT + LOWER; // j
        CHAR_CLAZZ[0x6B] = PRINT + LOWER; // k
        CHAR_CLAZZ[0x6C] = PRINT + LOWER + NUMTL; // l
        CHAR_CLAZZ[0x6D] = PRINT + LOWER; // m
        CHAR_CLAZZ[0x6E] = PRINT + LOWER + FLOAT; // n
        CHAR_CLAZZ[0x6F] = PRINT + LOWER; // o
        CHAR_CLAZZ[0x70] = PRINT + LOWER; // p
        CHAR_CLAZZ[0x71] = PRINT + LOWER; // q
        CHAR_CLAZZ[0x72] = PRINT + LOWER; // r
        CHAR_CLAZZ[0x73] = PRINT + LOWER + NUMTL; // s
        CHAR_CLAZZ[0x74] = PRINT + LOWER + FLOAT; // t
        CHAR_CLAZZ[0x75] = PRINT + LOWER + NUMTL; // u
        CHAR_CLAZZ[0x76] = PRINT + LOWER; // v
        CHAR_CLAZZ[0x77] = PRINT + LOWER; // w
        CHAR_CLAZZ[0x78] = PRINT + LOWER + HEX; // x
        CHAR_CLAZZ[0x79] = PRINT + LOWER + FLOAT; // y
        CHAR_CLAZZ[0x7A] = PRINT + LOWER; // z
        CHAR_CLAZZ[0x7B] = PRINT + PUNCT; // {
        CHAR_CLAZZ[0x7C] = PRINT + PUNCT; // |
        CHAR_CLAZZ[0x7D] = PRINT + PUNCT; // }
        CHAR_CLAZZ[0x7E] = PRINT + PUNCT; // ~
        CHAR_CLAZZ[0x7F] = CNTRL; // (DEL)
    }

    private static final String ARRAY_LENGTH_SYMBOL = "length";

    private static final String ARRAY_SYMBOL = "[]";

    private static final int ARRAY_SYMBOL_LENGTH = 2;

    /**
     * 指定の文字のビットセットにクラスのマッピングを返却します。<br>
     * アスキー文字以外が指定された場合は 0 を返却します。
     * 
     * @param ch
     *            対象の文字
     * @return ビットセット
     */
    static long getClazz(int ch) {
        return isAscii(ch) ? CHAR_CLAZZ[ch] : 0;
    }

    /**
     * 指定の文字が指定のビットセットにクラスのマッピングされているか検証します。
     * 
     * @param ch
     *            対象の文字
     * @param clazz
     *            検証するクラス
     * @return マッピングされている場合は true
     */
    static boolean isClazz(int ch, int clazz) {
        return (getClazz(ch) & clazz) != 0;
    }

    /**
     * 指定の文字がアスキー文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return アスキー文字の場合は true
     */
    static boolean isAscii(int ch) {
        return ((ch & 0xFFFFFF80) == 0);
    }

    /**
     * 指定の文字がアルファベット文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return アルファベット文字の場合は true
     */
    static boolean isAlpha(int ch) {
        return isClazz(ch, ALPHA);
    }

    /**
     * 指定の文字が２進数字文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return ２進数字文字の場合は true
     */
    static boolean isBinaryDigit(int ch) {
        return isClazz(ch, BINARY);
    }

    /**
     * 指定の文字が８進数字文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return ８進数字文字の場合は true
     */
    static boolean isOctDigit(int ch) {
        return isClazz(ch, OCTAL);
    }

    /**
     * 指定の文字が１０進数字文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return １０進数字文字の場合は true
     */
    static boolean isDigit(int ch) {
        return isClazz(ch, DIGIT);
    }

    /**
     * 指定の文字が１６進数字文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return １６進数字文字の場合は true
     */
    static boolean isHexDigit(int ch) {
        return isClazz(ch, HEX);
    }

    /**
     * 指定の文字がアルファベット文字か数字文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return アルファベット文字か数字文字の場合は true
     */
    static boolean isAlnum(int ch) {
        return isClazz(ch, ALNUM);
    }

    /**
     * 指定の文字がグリフ文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return グリフ文字の場合は true
     */
    static boolean isGraph(int ch) {
        return isClazz(ch, GRAPH);
    }

    /**
     * 指定の文字が印刷可能文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return 印刷可能文字の場合は true
     */
    static boolean isPrint(int ch) {
        return isClazz(ch, PRINT);
    }

    /**
     * 指定の文字が記号文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return 記号文字の場合は true
     */
    static boolean isPunct(int ch) {
        return isClazz(ch, PUNCT);
    }

    /**
     * 指定の文字がスペース文字（\x20\t\n\x0B\f\r）か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return スペース文字の場合は true
     */
    static boolean isSpace(int ch) {
        return isClazz(ch, SPACE);
    }

    /**
     * 指定の文字が余白文字（\x20\t）か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return 余白文字の場合は true
     */
    static boolean isBlank(int ch) {
        return ch == 0x20 || ch == 0x09;
    }

    /**
     * 指定の文字が制御文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return 制御文字の場合は true
     */
    static boolean isCntrl(int ch) {
        return isClazz(ch, CNTRL);
    }

    /**
     * 指定の文字が小文字アルファベット文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return 小文字アルファベット文字の場合は true
     */
    static boolean isLower(int ch) {
        return isClazz(ch, LOWER);
    }

    /**
     * 指定の文字が大文字アルファベット文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return 大文字アルファベット文字の場合は true
     */
    static boolean isUpper(int ch) {
        return isClazz(ch, UPPER);
    }

    /**
     * 指定の文字が単語を構成する文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return 単語を構成する文字の場合は true
     */
    static boolean isWord(int ch) {
        return isClazz(ch, WORD);
    }

    /**
     * 指定の文字を小文字のアルファベットに変換します。<br>
     * 文字が大文字のアルファベット以外の場合は、そのまま返却されます。
     * 
     * @param ch
     *            変換する文字
     * @return 変換後の文字
     */
    static int toLower(int ch) {
        return isUpper(ch) ? (ch + 0x20) : ch;
    }

    /**
     * 指定の文字を大文字のアルファベットに変換します。<br>
     * 文字が小文字のアルファベット以外の場合は、そのまま返却されます。
     * 
     * @param ch
     *            変換する文字
     * @return 変換後の文字
     */
    static int toUpper(int ch) {
        return isLower(ch) ? (ch - 0x20) : ch;
    }

    /**
     * 指定の文字がメンバー名を構成する文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return メンバー名を構成する文字の場合は true
     */
    static boolean isNumHead(int ch) {
        return isClazz(ch, NUMHD);
    }

    /* ビットマップの文字をスキップしたインデックスを検索して返却します。 */
    static int skip(OgdlEvent ev, int clazz) {
        return skip(ev, ev.index(), clazz);
    }

    /* ビットマップの文字をスキップしたインデックスを検索して返却します。 */
    static int skip(OgdlEvent ev, int index, int clazz) {
        int i = index;
        while (i < ev.len) {
            final int ch = ev.charAtTo(i);
            if (!isClazz(ch, clazz)) {
                break;
            }
            i++;
        }
        return i;
    }

    /* 空白文字（組込コメントを含む）以外のインデックスを検索し返却します。 */
    static int skipSpace(OgdlEvent ev) {
        int i = ev.index();
        while (i < ev.len) {
            i = skip(ev, i, SPACE);
            if (ev.hasNextTo(i)) {
                final int ch = ev.charAtTo(i);
                switch (ch) {
                case '#':
                    i = skipEncloseString(ev, i);
                    break;
                case 'N':
                    final int next = i + 1;
                    if (ev.hasNextTo(next) && isEncloseOpenChar(ev.ptn, next)) {
                        i = skipEncloseString(ev, ++i);
                    } else {
                        return i;
                    }
                    break;
                default:
                    return i;
                }
            } else {
                return i;
            }
        }
        return i;
    }

    /* 演算子の終了インデックスを解析して返却します。 */
    static int skipOperator(OgdlEvent ev) {
        final int i = skip(ev, OPERATOR);
        if (ev.len > i && isClazz(ev.charAtTo(i), SPACE)) {
            return i;
        }
        return ev.index();
    }

    /* 数値領域（等差数列や時間領域）式の文字列を返却します。 */
    static String cutDifferenceString(OgdlEvent ev) {
        return ev.cutForShift(skip(ev, DIFFERENCE));
    }

    /* 開始文字に囲まれた文字列として文字列を切り出して返却します。 */
    static String cutEncloseString(OgdlEvent ev) {
        final int close = getEncloseCloseChar(ev);
        ev.shift();

        int begin = ev.index();
        final StringBuffer buff = new StringBuffer(ev.len - begin);
        int i = begin;
        while (i < ev.len) {
            i = ev.ptn.indexOf(close, i);
            if (-1 == i) {
                throw new OgdlSyntaxException(ev, Utils.log("not find close char. close=", close));
            }
            final int next = i + 1;
            if (next < ev.len && ev.charAtTo(next) == close) {
                buff.append(ev.ptn.substring(begin, i));
                begin = i + 1;
                i++;
            } else {
                buff.append(ev.ptn.substring(begin, i));
                i++;
                break;
            }
            i++;
        }
        ev.setIndex(i);
        return buff.toString();
    }

    /* クラス名を解析して返却します。 */
    static String cutExtendClassName(OgdlEvent ev) {
        int i = skip(ev, CLAZZ);
        while (i < ev.len) {
            if (!ev.isPrefixTo(ARRAY_SYMBOL, i)) {
                break;
            }
            i += ARRAY_SYMBOL_LENGTH;
        }
        return ev.cutForShift(i);
    }

    /* Javaのメンバ (フィールドやメソッド、コンストラクタ)の識別名を切り出して返却します。 */
    static String cutMemberName(OgdlEvent ev) {
        return ev.cutForShift(skip(ev, MEMBER));
    }

    /* 組込み関数の完全名を切り出して返却します。 */
    static String cutFunctionName(OgdlEvent ev) {
        return ev.cutForShift(skip(ev, FUNC));
    }

    /* 数値文字列を切り出して返却します。 */
    static String cutNumberString(OgdlEvent ev) {
        int i = skip(ev, NUMBER);
        if (ev.len > i && isClazz(ev.charAtTo(i), NUMTL)) {
            i++;
        }
        return ev.cutForShift(i);
    }

    /* 配列の length フィールドの指定か検証します。 */
    static boolean isArraylength(Class clazz, String memberName) {
        return clazz.isArray() && ARRAY_LENGTH_SYMBOL.equals(memberName);
    }

    /* 囲み文字列の開始文字として有効な文字か検証します。 */
    static boolean isEncloseOpenChar(OgdlEvent ev) {
        return isEncloseOpenChar(ev.ptn, ev.index());
    }

    /* Javaのメンバ（フィールドやメソッド、コンストラクタ）として認識出来るか検証します。 */
    static boolean isMemberHead(OgdlEvent ev) {
        return isClazz(ev.charAt(), MENHD);
    }

    /* 囲み文字列の終了文字を返却します。 */
    static int getEncloseCloseChar(OgdlEvent ev) {
        return getEncloseCloseChar(ev.ptn, ev.index());
    }

    /*
     * private
     */

    /* 開始文字に囲まれた文字列として終端インデックスを計算して返却します。 */
    private static int skipEncloseString(OgdlEvent ev, int index) {
        int i = index;
        final int close = getEncloseCloseChar(ev.ptn, i);
        i++;
        while (i < ev.len) {
            i = ev.ptn.indexOf(close, i);
            if (-1 == i) {
                throw new OgdlSyntaxException(ev, Utils.log("not find close char. close=", close));
            }
            final int next = i + 1;
            if (next < ev.len && ev.charAtTo(next) == close) {
                i++;
            } else {
                i++;
                break;
            }
            i++;
        }
        return i;
    }

    /* 囲み文字列の開始文字として有効な文字か検証します。 */
    private static boolean isEncloseOpenChar(CharSequence cs, int index) {
        final int ch = cs.charAt(index);
        switch (ch) {
        case '!':
        case '"':
        case '%':
        case '\'':
        case '(':
        case '/':
        case '<':
        case '?':
        case '@':
        case '[':
        case '`':
        case '{':
        case '|':
            return true;
        default:
            return false;
        }
    }

    /* 囲み文字列の終了文字を返却します。 */
    private static int getEncloseCloseChar(CharSequence cs, int index) {
        final int ch = cs.charAt(index);
        switch (ch) {
        case '(':
            return ')';
        case '<':
            return '>';
        case '[':
            return ']';
        case '{':
            return '}';
        default:
            return ch;
        }
    }
}
