/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * OGDLの構文解析機能を提供します。<br>
 * このパーザが個別機能を提供する、他のパーザの入口として機能します。
 */
class OgdlParser {

    /*
     * Prefix Expression
     */

    static Object evaluate(OgdlEvent ev) {
        final int open = ev.charAt();
        switch (open) {
        case 0x21: // !
            // Logical Complement Operator
            return OgdlUnaryOperatorParser.evaluateLogicalComplement(ev);
        case 0x25: // %
            // Regex
            return OgdlPrimitiveParser.evaluateRegexPattern(ev);
        case 0x27: // '
            // String
            return OgdlPrimitiveParser.evaluateString(ev);
        case 0x28: // (
            // Set
            return OgdlCollectionParser.evaluateDefaultSet(ev);
        case 0x2B: // +
            // Unary Plus Operator
            return OgdlUnaryOperatorParser.evaluateUnaryPlus(ev);
        case 0x2D: // -
            // Number
            if (ev.hasNext() && OgdlSyntax.isNumHead(ev.charAtBy(1))) {
                return OgdlPrimitiveParser.evaluateNumber(ev);
            }
            // Unary Minus Operator
            return OgdlUnaryOperatorParser.evaluateUnaryMinus(ev);
        case 0x2F: // /
            // Date Time
            return OgdlPrimitiveParser.evaluateDateTime(ev);
        case 0x30: // 0
        case 0x31: // 1
        case 0x32: // 2
        case 0x33: // 3
        case 0x34: // 4
        case 0x35: // 5
        case 0x36: // 6
        case 0x37: // 7
        case 0x38: // 8
        case 0x39: // 9
            // Number
            return OgdlPrimitiveParser.evaluateNumber(ev);
        case 0x40: // @
            // Reference or Class Operator
            if (isLiteral(ev, "@@")) {
                // Class Operator
                return OgdlPrimitiveParser.evaluateClassByExtendName(ev);
            }
            // Reference Operator
            ev.shift();
            return OgdlPrimitiveParser.evaluateReference(ev);
        case 0x42: // B
            ev.shift();
            return OgdlBlockOperatorParser.evaluateOperator(ev);
        case 0x44: // D
            // Date Time
            ev.shift();
            return OgdlPrimitiveParser.evaluateDateTime(ev);
        case 0x46: // F
            // Operator Function
            ev.shift();
            return OgdlOperatorParser.evaluateOperator(ev);
        case 0x49: // I
            // Infinity
            return OgdlPrimitiveParser.evaluateNumber(ev);
        case 0x4E: // N
            // NaN
            return OgdlPrimitiveParser.evaluateNumber(ev);
        case 0x52: // R
            // Regex
            ev.shift();
            return OgdlPrimitiveParser.evaluateRegexPattern(ev);
        case 0x53: // S
            // String
            ev.shift();
            return OgdlPrimitiveParser.evaluateString(ev);
        case 0x55: // U
            // Character
            ev.shift();
            return OgdlPrimitiveParser.evaluateCharacter(ev);
        case 0x5B: // [
            // List
            return OgdlCollectionParser.evaluateDefaultList(ev);
        case 0x60: // `
            // Character
            return OgdlPrimitiveParser.evaluateCharacter(ev);
        case 0x61: // a
            // array
            if (isPrefixName(ev, "array:")) {
                return OgdlCollectionParser.evaluateArray(ev);
            }
            break;
        case 0x63: // c
            // class
            if (isPrefixName(ev, "class:")) {
                return OgdlPrimitiveParser.evaluateClassByExtendName(ev);
            }
            // coll
            if (isPrefixName(ev, "coll:")) {
                return OgdlCollectionParser.evaluateCollection(ev);
            }
            // constructor
            if (isPrefixName(ev, "cons:")) {
                return OgdlIntrospectParser.evaluateConstructor(ev);
            }
            break;
        case 0x66: // f
            // Boolean.FALSE
            if (isLiteral(ev, "false")) {
                return Boolean.FALSE;
            }
            // field
            if (isPrefixName(ev, "field:")) {
                return OgdlIntrospectParser.evaluateField(ev);
            }
            // function
            if (isPrefixName(ev, "f:")) {
                return OgdlFunctionParser.evaluateFunction(ev);
            }
            break;
        case 0x6C: // l
            // list
            if (isPrefixName(ev, "list:")) {
                return OgdlCollectionParser.evaluateList(ev);
            }
            break;
        case 0x6D: // m
            // map
            if (isPrefixName(ev, "map:")) {
                return OgdlCollectionParser.evaluateMap(ev);
            }
            // method
            if (isPrefixName(ev, "method:")) {
                return OgdlIntrospectParser.evaluateMethod(ev);
            }
            break;
        case 0x6E: // n
            // Null
            if (isLiteral(ev, "null")) {
                return null;
            }
            // new
            if (isPrefixName(ev, "new:")) {
                return OgdlIntrospectParser.evaluateNewInstance(ev);
            }
            break;
        case 0x72: // r
            // Reference Operator
            if (isPrefixName(ev, "ref:")) {
                return OgdlPrimitiveParser.evaluateReference(ev);
            }
            break;
        case 0x73: // s
            // set
            if (isPrefixName(ev, "set:")) {
                return OgdlCollectionParser.evaluateSet(ev);
            }
            // sortedSet
            if (isPrefixName(ev, "sortedSet:")) {
                return OgdlCollectionParser.evaluateSortedSet(ev);
            }
            // sortedMap
            if (isPrefixName(ev, "sortedMap:")) {
                return OgdlCollectionParser.evaluateSortedMap(ev);
            }
            break;
        case 0x74: // t
            // Boolean.TRUE
            if (isLiteral(ev, "true")) {
                return Boolean.TRUE;
            }
            break;
        case 0x7B: // {
            // Map
            return OgdlCollectionParser.evaluateDefaultMap(ev);
        case 0x7E: // ~
            // Bitwise Complement Operator
            return OgdlUnaryOperatorParser.evaluateBitwiseComplement(ev);
        default:
            break;
        }
        throw new OgdlSyntaxException(ev, "not find expression.");
    }

    /*
     * Collection
     */

    /* コンマ区切りのコレクション書式として解析して返却します。 */
    static Collection evaluateEncloseCollection(OgdlEvent ev) {
        return OgdlCollectionParser.evaluateEncloseCollection(ev);
    }

    /* 開始文字を検証せずに、コンマ区切りのコレクション書式として解析して返却します。 */
    static Collection evaluateCloseCollection(OgdlEvent ev) {
        return OgdlCollectionParser.evaluateCloseCollection(ev);
    }

    /* マップ書式として解析して指定のマップに追加して返却します。 */
    static Map evaluateEncloseMap(OgdlEvent ev) {
        return OgdlCollectionParser.evaluateEncloseMap(ev);
    }

    /* 開始文字を検証せずに、マップ書式として解析して指定のマップに追加して返却します。 */
    static Map evaluateCloseMap(OgdlEvent ev) {
        return OgdlCollectionParser.evaluateCloseMap(ev);
    }

    /* 数値領域（等差数列や時間領域）式を有効とするコンマ区切りのセット書式として引数のセットに格納します。 */
    static Set evaluateEncloseProgressionSet(OgdlEvent ev) {
        return (Set) OgdlCollectionParser.evaluateEncloseProgressionCollection(ev);
    }

    /* 開始文字を検証せずに、数値領域（等差数列や時間領域）式を有効とするコンマ区切りのセット書式として解析して引数のセットに格納して返却します。 */
    static Set evaluateCloseProgressionSet(OgdlEvent ev) {
        return (Set) OgdlCollectionParser.evaluateCloseProgressionCollection(ev);
    }

    /* 数値領域（等差数列や時間領域）式を有効とするコンマ区切りのリスト書式として引数のリストに格納します。 */
    static List evaluateEncloseProgressionList(OgdlEvent ev) {
        return (List) OgdlCollectionParser.evaluateEncloseProgressionCollection(ev);
    }

    /* 開始文字を検証せずに、数値領域（等差数列や時間領域）式を有効とするコンマ区切りのリスト書式として解析して引数のリストに格納して返却します。 */
    static List evaluateCloseProgressionList(OgdlEvent ev) {
        return (List) OgdlCollectionParser.evaluateCloseProgressionCollection(ev);
    }

    static Class evaluateClass(OgdlEvent ev) {
        return OgdlPrimitiveParser.evaluateClassByExtendName(ev);
    }

    /*
     * Postfix Expression
     */

    /* 後置式を実行します。 */
    static Object evaluatePostfixExpression(OgdlEvent ev, Object returnValue) {
        final int open = ev.charAt();
        switch (open) {
        case '[':
        case '.':
        case '#':
            ev.target = returnValue;
            return OgdlIntrospectParser.evaluateObjectNavigate(ev);
        default:
            break;
        }
        return returnValue;
    }

    /*
     * private
     */

    private static boolean isLiteral(OgdlEvent ev, String literal) {
        if (ev.isPrefix(literal)) {
            ev.shiftBy(literal.length());
            return true;
        }
        return false;
    }

    /*   */
    private static boolean isPrefixName(OgdlEvent ev, String id) {
        if (ev.isPrefix(id)) {
            ev.shiftBy(id.length());
            ev.shiftSpace();
            return true;
        }
        return false;
    }

}
