/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * 演算子の構文解析機能を提供します。
 */
class OgdlOperatorParser {

    static final Map operators;
    static {
        final Map ops = new HashMap(16 * 8);
        ops.put("+", Boxing.box(Arithmetic.ARITHMETIC_ADDITION));
        ops.put("add", Boxing.box(Arithmetic.ARITHMETIC_ADDITION));
        ops.put("-", Boxing.box(Arithmetic.ARITHMETIC_SUBTRACTION));
        ops.put("sub", Boxing.box(Arithmetic.ARITHMETIC_SUBTRACTION));
        ops.put("*", Boxing.box(Arithmetic.ARITHMETIC_MULTIPLICATION));
        ops.put("mul", Boxing.box(Arithmetic.ARITHMETIC_MULTIPLICATION));
        ops.put("/", Boxing.box(Arithmetic.ARITHMETIC_DIVISION));
        ops.put("div", Boxing.box(Arithmetic.ARITHMETIC_DIVISION));
        ops.put("%", Boxing.box(Arithmetic.ARITHMETIC_REMAINDER));
        ops.put("mod", Boxing.box(Arithmetic.ARITHMETIC_REMAINDER));
        ops.put("+>", Boxing.box(Arithmetic.ARITHMETIC_STRING_BUILDING));
        ops.put("cnct", Boxing.box(Arithmetic.ARITHMETIC_STRING_BUILDING));
        ops.put("&", Boxing.box(Arithmetic.BITWISE_AND));
        ops.put("band", Boxing.box(Arithmetic.BITWISE_AND));
        ops.put("|", Boxing.box(Arithmetic.BITWISE_OR));
        ops.put("bor", Boxing.box(Arithmetic.BITWISE_OR));
        ops.put("<<", Boxing.box(Arithmetic.BIT_SHIFT_LEFT));
        ops.put("shl", Boxing.box(Arithmetic.BIT_SHIFT_LEFT));
        ops.put("^", Boxing.box(Arithmetic.BITWISE_XOR));
        ops.put("xor", Boxing.box(Arithmetic.BITWISE_XOR));
        ops.put(">>", Boxing.box(Arithmetic.BIT_SHIFT_RIGHT));
        ops.put("shr", Boxing.box(Arithmetic.BIT_SHIFT_RIGHT));
        ops.put(">>>", Boxing.box(Arithmetic.BIT_SHIFT_LOGICAL_RIGHT));
        ops.put("ushr", Boxing.box(Arithmetic.BIT_SHIFT_LOGICAL_RIGHT));
        ops.put("==", Boxing.box(Arithmetic.COMPARATIVE_EQUALITY));
        ops.put("eq", Boxing.box(Arithmetic.COMPARATIVE_EQUALITY));
        ops.put("!=", Boxing.box(Arithmetic.COMPARATIVE_INEQUALITY));
        ops.put("ne", Boxing.box(Arithmetic.COMPARATIVE_INEQUALITY));
        ops.put("<", Boxing.box(Arithmetic.COMPARATIVE_LESS_THAN));
        ops.put("lt", Boxing.box(Arithmetic.COMPARATIVE_LESS_THAN));
        ops.put("<=", Boxing.box(Arithmetic.COMPARATIVE_GREATER_THAN));
        ops.put("le", Boxing.box(Arithmetic.COMPARATIVE_GREATER_THAN));
        ops.put(">", Boxing.box(Arithmetic.COMPARATIVE_LESS_EQUALITY));
        ops.put("gt", Boxing.box(Arithmetic.COMPARATIVE_LESS_EQUALITY));
        ops.put(">=", Boxing.box(Arithmetic.COMPARATIVE_GREATER_EQUALITY));
        ops.put("ge", Boxing.box(Arithmetic.COMPARATIVE_GREATER_EQUALITY));
        ops.put("===", Boxing.box(Arithmetic.RELATIONAL_OBJECT_EQUALITY));
        ops.put("oeq", Boxing.box(Arithmetic.RELATIONAL_OBJECT_EQUALITY));
        ops.put("!==", Boxing.box(Arithmetic.RELATIONAL_OBJECT_INEQUALITY));
        ops.put("one", Boxing.box(Arithmetic.RELATIONAL_OBJECT_INEQUALITY));
        ops.put("req", Boxing.box(Arithmetic.RELATIONAL_REFERENCE_EQUALITY));
        ops.put("====", Boxing.box(Arithmetic.RELATIONAL_REFERENCE_EQUALITY));
        ops.put("rne", Boxing.box(Arithmetic.RELATIONAL_REFERENCE_INEQUALITY));
        ops.put("!===", Boxing.box(Arithmetic.RELATIONAL_REFERENCE_INEQUALITY));
        ops.put("?>", Boxing.box(Arithmetic.RELATIONAL_INSTANCE_OF));
        ops.put("iof", Boxing.box(Arithmetic.RELATIONAL_INSTANCE_OF));
        operators = ops;
    }

    /* 演算式を実行します。 */
    static Object evaluateOperator(OgdlEvent ev) {
        if (OgdlSyntax.isEncloseOpenChar(ev)) {
            return evaluateEncloseOperator(ev);
        } else {
            throw new OgdlSyntaxException(ev, "is not enclose literal.");
        }
    }

    /* 演算式を実行します。 */
    private static Object evaluateEncloseOperator(OgdlEvent ev) {

        final int close = OgdlSyntax.getEncloseCloseChar(ev);
        final int begin = ev.shift();// next index
        final int next = OgdlSyntax.skipOperator(ev);

        if (next == begin) {
            // 演算子が検出できない場合
            throw new OgdlSyntaxException(ev, "no find operator. ");
        }

        // 演算子の種別を検出する
        final String name = ev.cutForEnd(next);
        final int type = asOperatorType(name);

        if (-1 == type) {
            // 演算子が検出できない場合
            throw new OgdlSyntaxException(ev, "no find operator. ");
        }

        // 演算処理を実行する
        ev.setIndex(next);
        final List args = getArguments(ev, close);
        if (args.isEmpty()) {
            final OgdlFunctionException fe = new OgdlFunctionException(ev, "empty arguments. ");
            throw fe.throwFor(ev, name, args);
        }
        try {
            return Arithmetic.evaluate(type, args);
        } catch (final ArithmeticException e) {
            final OgdlFunctionException fe = new OgdlFunctionException("arithmetic err.", e);
            throw fe.throwFor(ev, name, args);
        } catch (final RuntimeException e) {
            final OgdlFunctionException fe = new OgdlFunctionException("java runtime err.", e);
            throw fe.throwFor(ev, name, args);
        }

    }

    /* オペレータのタイプを示す数値を返却します。 */
    private static int asOperatorType(String name) {
        final Object type = operators.get(name);
        if (null == type) {
            return -1;
        }
        return ((Integer) type).intValue();
    }

    /* (method: arg, arg, ...) の引数を解析して返却します。 */
    private static List getArguments(OgdlEvent ev, int close) {
        return (List) OgdlParser.evaluateCloseCollection(ev.get('\u0000', close, new LinkedList()));
    }

}
