/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Iterator;
import java.util.List;

/**
 * オブジェクトの算術演算機能を提供します。
 */
class ArithmeticHelper {

    static Object add(Object v1, Object v2) {
        if (isStringOperand(v1, v2)) {
            return concat(v1, v2);
        } else {
            return arithmetic(Arithmetic.ARITHMETIC_ADDITION, v1, v2);
        }
    }

    static Object arithmetic(int type, Object v1, Object v2) {
        try {
            if (isNumberOperand(v1, v2)) {
                final Number[] nums = toNumber(v1, v2);
                final int arithmeticType = Widening.arithmetic(nums);
                switch (arithmeticType) {
                case Widening.TYPE_BIGDECIMAL:
                    return arithmeticBigDecimal(type, (BigDecimal) nums[0], (BigDecimal) nums[1]);
                case Widening.TYPE_BIGINTEGER:
                    return arithmeticBigInteger(type, (BigInteger) nums[0], (BigInteger) nums[1]);
                case Widening.TYPE_DOUBLE:
                    return Boxing.box(arithmeticDouble(type, nums[0], nums[1]));
                case Widening.TYPE_FLOAT:
                    return Boxing.box(arithmeticFloat(type, nums[0], nums[1]));
                case Widening.TYPE_LONG:
                    return Boxing.box(arithmeticLong(type, nums[0], nums[1]));
                case Widening.TYPE_INTEGER:
                    return Boxing.box(arithmeticInteger(type, nums[0], nums[1]));
                default:
                    break;
                }
            }
        } catch (final ClassCastException e) {
            throw buildArithmeticException("is not Number.", e);
        } catch (final NumberFormatException e) {
            throw buildArithmeticException("number format error.", e);
        }
        throw new ArithmeticException("illegal operand type.");
    }

    static Object add(List vs) {
        ifIllegalArgumentsSize(vs);
        if (isStringOperand(vs.get(0), vs.get(1))) {
            return concat(vs);
        }
        return arithmetic(Arithmetic.ARITHMETIC_ADDITION, vs);
    }

    static Object arithmetic(int type, List vs) {
        try {
            if (isNumberOperand(vs)) {
                final Number[] nums = new Number[2];
                final Iterator i = vs.iterator();
                nums[0] = toNumber(i.next());
                while (i.hasNext()) {
                    nums[1] = toNumber(i.next());
                    final int arithmeticType = Widening.arithmetic(nums);
                    switch (arithmeticType) {
                    case Widening.TYPE_BIGDECIMAL:
                        nums[0] = arithmeticBigDecimal(type, (BigDecimal) nums[0], (BigDecimal) nums[1]);
                        break;
                    case Widening.TYPE_BIGINTEGER:
                        nums[0] = arithmeticBigInteger(type, (BigInteger) nums[0], (BigInteger) nums[1]);
                        break;
                    case Widening.TYPE_DOUBLE:
                        nums[0] = Boxing.box(arithmeticDouble(type, nums[0], nums[1]));
                        break;
                    case Widening.TYPE_FLOAT:
                        nums[0] = Boxing.box(arithmeticFloat(type, nums[0], nums[1]));
                        break;
                    case Widening.TYPE_LONG:
                        nums[0] = Boxing.box(arithmeticLong(type, nums[0], nums[1]));
                        break;
                    case Widening.TYPE_INTEGER:
                        nums[0] = Boxing.box(arithmeticInteger(type, nums[0], nums[1]));
                        break;
                    default:
                        throw new ArithmeticException("illegal operand type.");
                    }
                }
                return nums[0];
            }
        } catch (final ClassCastException e) {
            throw buildArithmeticException("is not Number.", e);
        } catch (final NumberFormatException e) {
            throw buildArithmeticException("number format error.", e);
        }
        throw new ArithmeticException("illegal operand type.");
    }

    private static Number arithmeticBigDecimal(int type, BigDecimal v1, BigDecimal v2) {
        switch (type) {
        case Arithmetic.ARITHMETIC_ADDITION:
            return v1.add(v2);
        case Arithmetic.ARITHMETIC_SUBTRACTION:
            return v1.subtract(v2);
        case Arithmetic.ARITHMETIC_MULTIPLICATION:
            return v1.multiply(v2);
        case Arithmetic.ARITHMETIC_DIVISION:
            return v1.divide(v2, BigDecimal.ROUND_HALF_EVEN);
        case Arithmetic.ARITHMETIC_REMAINDER:
            return v1.subtract(new BigDecimal(v1.divide(v2, BigDecimal.ROUND_DOWN).toBigInteger()).multiply(v2));
        default:
            throw new ArithmeticException("illegal operand type.");
        }
    }

    private static Number arithmeticBigInteger(int type, BigInteger v1, BigInteger v2) {
        switch (type) {
        case Arithmetic.ARITHMETIC_ADDITION:
            return v1.add(v2);
        case Arithmetic.ARITHMETIC_SUBTRACTION:
            return v1.subtract(v2);
        case Arithmetic.ARITHMETIC_MULTIPLICATION:
            return v1.multiply(v2);
        case Arithmetic.ARITHMETIC_DIVISION:
            return v1.divide(v2);
        case Arithmetic.ARITHMETIC_REMAINDER:
            return v1.mod(v2);
        default:
            throw new ArithmeticException("illegal operand type.");
        }
    }

    private static double arithmeticDouble(int type, Number v1, Number v2) {
        switch (type) {
        case Arithmetic.ARITHMETIC_ADDITION:
            return v1.doubleValue() + v2.doubleValue();
        case Arithmetic.ARITHMETIC_SUBTRACTION:
            return v1.doubleValue() - v2.doubleValue();
        case Arithmetic.ARITHMETIC_MULTIPLICATION:
            return v1.doubleValue() * v2.doubleValue();
        case Arithmetic.ARITHMETIC_DIVISION:
            return v1.doubleValue() / v2.doubleValue();
        case Arithmetic.ARITHMETIC_REMAINDER:
            return v1.doubleValue() % v2.doubleValue();
        default:
            throw new ArithmeticException("illegal operand type.");
        }
    }

    private static float arithmeticFloat(int type, Number v1, Number v2) {
        switch (type) {
        case Arithmetic.ARITHMETIC_ADDITION:
            return v1.floatValue() + v2.floatValue();
        case Arithmetic.ARITHMETIC_SUBTRACTION:
            return v1.floatValue() - v2.floatValue();
        case Arithmetic.ARITHMETIC_MULTIPLICATION:
            return v1.floatValue() * v2.floatValue();
        case Arithmetic.ARITHMETIC_DIVISION:
            return v1.floatValue() / v2.floatValue();
        case Arithmetic.ARITHMETIC_REMAINDER:
            return v1.floatValue() % v2.floatValue();
        default:
            throw new ArithmeticException("illegal operand type.");
        }
    }

    private static long arithmeticLong(int type, Number v1, Number v2) {
        switch (type) {
        case Arithmetic.ARITHMETIC_ADDITION:
            return v1.longValue() + v2.longValue();
        case Arithmetic.ARITHMETIC_SUBTRACTION:
            return v1.longValue() - v2.longValue();
        case Arithmetic.ARITHMETIC_MULTIPLICATION:
            return v1.longValue() * v2.longValue();
        case Arithmetic.ARITHMETIC_DIVISION:
            return v1.longValue() / v2.longValue();
        case Arithmetic.ARITHMETIC_REMAINDER:
            return v1.longValue() % v2.longValue();
        default:
            throw new ArithmeticException("illegal operand type.");
        }
    }

    private static int arithmeticInteger(int type, Number v1, Number v2) {
        switch (type) {
        case Arithmetic.ARITHMETIC_ADDITION:
            return v1.intValue() + v2.intValue();
        case Arithmetic.ARITHMETIC_SUBTRACTION:
            return v1.intValue() - v2.intValue();
        case Arithmetic.ARITHMETIC_MULTIPLICATION:
            return v1.intValue() * v2.intValue();
        case Arithmetic.ARITHMETIC_DIVISION:
            return v1.intValue() / v2.intValue();
        case Arithmetic.ARITHMETIC_REMAINDER:
            return v1.intValue() % v2.intValue();
        default:
            throw new ArithmeticException("illegal operand type.");
        }
    }

    /*
     * Comparable
     */

    /**
     * 一項に対する二項の比較演算を実行し結果を返却します。
     * 
     * @param v1
     *            値１
     * @param v2
     *            値２
     * @return 演算結果
     */
    static int compare(Object v1, Object v2) {
        if (isNumberOperand(v1, v2)) {
            try {
                final Number[] nums = toNumber(v1, v2);
                final int arithmeticType = Widening.arithmetic(nums);
                switch (arithmeticType) {
                case Widening.TYPE_BIGDECIMAL:
                    return ((BigDecimal) nums[0]).compareTo((BigDecimal) nums[1]);
                case Widening.TYPE_BIGINTEGER:
                    return ((BigInteger) nums[0]).compareTo((BigInteger) nums[1]);
                case Widening.TYPE_DOUBLE:
                    return ((Double) nums[0]).compareTo((Double) nums[1]);
                case Widening.TYPE_FLOAT:
                    return ((Float) nums[0]).compareTo((Float) nums[1]);
                case Widening.TYPE_LONG:
                    return ((Long) nums[0]).compareTo((Long) nums[1]);
                case Widening.TYPE_INTEGER:
                    return ((Integer) nums[0]).compareTo((Integer) nums[1]);
                default:
                    break;
                }
            } catch (final ClassCastException e) {
                throw buildArithmeticException("is not Number.", e);
            } catch (final NumberFormatException e) {
                throw buildArithmeticException("number format error.", e);
            }
        } else if (isComparableOperand(v1, v2)) {
            return ((Comparable) v1).compareTo(v2);
        }
        throw new ArithmeticException("illegal operand type.");
    }

    /**
     * 一項に対する二項の比較演算を実行し結果を返却します。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    static Boolean compare(int type, List vs) {
        ifIllegalArgumentsSize(vs);

        final Object v1 = vs.get(0);
        final Object v2 = vs.get(1);
        if (isNumberOperand(v1, v2)) {
            return compareNumber(type, vs);
        }
        if (isComparableOperand(v1, v2)) {
            return compareComparable(type, vs);
        }
        throw new ArithmeticException("illegal operand type.");
    }

    private static Boolean compareComparable(int type, List vs) {
        try {
            final Iterator i = vs.iterator();
            final Comparable comp1 = (Comparable) i.next();
            Comparable comp2 = null;
            while (i.hasNext()) {
                comp2 = (Comparable) i.next();
                if (!compare(type, comp1, comp2)) {
                    return Boolean.FALSE;
                }
            }
            return Boolean.TRUE;
        } catch (final ClassCastException e) {
            throw buildArithmeticException("is not Comparable.", e);
        }
    }

    private static Boolean compareNumber(int type, List vs) {
        try {
            final Number[] nums = new Number[2];
            final Iterator i = vs.iterator();
            nums[0] = toNumber(i.next());
            while (i.hasNext()) {
                nums[1] = toNumber(i.next());
                Widening.arithmetic(nums);
                if (!compare(type, (Comparable) nums[0], (Comparable) nums[1])) {
                    return Boolean.FALSE;
                }
                break;
            }
            return Boolean.TRUE;
        } catch (final ClassCastException e) {
            throw buildArithmeticException("is not Number.", e);
        } catch (final NumberFormatException e) {
            throw buildArithmeticException("is infinite or NaN.", e);
        }
    }

    private static boolean compare(int type, Comparable comp1, Comparable comp2) {
        switch (type) {
        case Arithmetic.COMPARATIVE_EQUALITY:
            return comp1.compareTo(comp2) == 0;
        case Arithmetic.COMPARATIVE_LESS_THAN:
            return comp1.compareTo(comp2) < 0;
        case Arithmetic.COMPARATIVE_GREATER_THAN:
            return comp1.compareTo(comp2) <= 0;
        case Arithmetic.COMPARATIVE_LESS_EQUALITY:
            return comp1.compareTo(comp2) > 0;
        case Arithmetic.COMPARATIVE_GREATER_EQUALITY:
            return comp1.compareTo(comp2) >= 0;
        default:
            throw new ArithmeticException("illegal operand type.");
        }
    }

    /*
     * bitwise
     */

    static Object bitwise(int type, Object v1, Object v2) {
        if (isNumberOperand(v1, v2)) {
            try {
                final Number[] nums = toNumber(v1, v2);
                final int arithmeticType = Widening.arithmetic(nums);
                switch (arithmeticType) {
                case Widening.TYPE_BIGINTEGER:
                    switch (type) {
                    case Arithmetic.BITWISE_AND:
                        return ((BigInteger) nums[0]).and((BigInteger) nums[1]);
                    case Arithmetic.BITWISE_XOR:
                        return ((BigInteger) nums[0]).xor((BigInteger) nums[1]);
                    case Arithmetic.BITWISE_OR:
                        return ((BigInteger) nums[0]).or((BigInteger) nums[1]);
                    default:
                        break;
                    }
                    break;
                case Widening.TYPE_LONG:
                    switch (type) {
                    case Arithmetic.BITWISE_AND:
                        return Boxing.box(nums[0].longValue() & nums[1].longValue());
                    case Arithmetic.BITWISE_XOR:
                        return Boxing.box(nums[0].longValue() ^ nums[1].longValue());
                    case Arithmetic.BITWISE_OR:
                        return Boxing.box(nums[0].longValue() | nums[1].longValue());
                    default:
                        break;
                    }
                    break;
                case Widening.TYPE_INTEGER:
                    switch (type) {
                    case Arithmetic.BITWISE_AND:
                        return Boxing.box(nums[0].intValue() & nums[1].intValue());
                    case Arithmetic.BITWISE_XOR:
                        return Boxing.box(nums[0].intValue() ^ nums[1].intValue());
                    case Arithmetic.BITWISE_OR:
                        return Boxing.box(nums[0].intValue() | nums[1].intValue());
                    default:
                        break;
                    }
                    break;
                default:
                    throw new ArithmeticException("illegal operand type.");
                }
            } catch (final ClassCastException e) {
                throw buildArithmeticException("is not Number.", e);
            } catch (final NumberFormatException e) {
                throw buildArithmeticException("number format error.", e);
            }
        }
        if (isBooleanOperand(v1, v2)) {
            final Boolean bool1 = (Boolean) v1;
            final Boolean bool2 = (Boolean) v2;
            switch (type) {
            case Arithmetic.BITWISE_AND:
                return Boxing.box(bool1.booleanValue() & bool2.booleanValue());
            case Arithmetic.BITWISE_XOR:
                return Boxing.box(bool1.booleanValue() ^ bool2.booleanValue());
            case Arithmetic.BITWISE_OR:
                return Boxing.box(bool1.booleanValue() | bool2.booleanValue());
            default:
                throw new ArithmeticException("illegal operand type.");
            }
        }
        throw new ArithmeticException("illegal operand type.");
    }

    static Object bitwise(int type, List vs) {
        ifIllegalArgumentsSize(vs);
        final Object v1 = vs.get(0);
        final Object v2 = vs.get(1);
        if (isNumberOperand(v1, v2)) {
            return bitwiseNumber(type, vs);
        }
        if (isBooleanOperand(v1, v2)) {
            return bitwiseBoolean(type, vs);
        }
        throw new ArithmeticException("illegal operand type.");
    }

    private static Object bitwiseBoolean(int type, List vs) {
        try {
            final Iterator i = vs.iterator();
            Boolean bool1 = (Boolean) i.next();
            Boolean bool2 = null;
            while (i.hasNext()) {
                bool2 = (Boolean) i.next();
                switch (type) {
                case Arithmetic.BITWISE_AND:
                    bool1 = Boxing.box(bool1.booleanValue() & bool2.booleanValue());
                    break;
                case Arithmetic.BITWISE_XOR:
                    bool1 = Boxing.box(bool1.booleanValue() ^ bool2.booleanValue());
                    break;
                case Arithmetic.BITWISE_OR:
                    bool1 = Boxing.box(bool1.booleanValue() | bool2.booleanValue());
                    break;
                default:
                    throw new ArithmeticException("illegal operand type.");
                }
            }
            return bool1;
        } catch (final ClassCastException e) {
            throw buildArithmeticException("is not Boolean.", e);
        }
    }

    private static Object bitwiseNumber(int type, List vs) {
        try {
            final Number[] nums = new Number[2];
            final Iterator i = vs.iterator();
            nums[0] = toNumber(i.next());
            while (i.hasNext()) {
                nums[1] = toNumber(i.next());
                final int arithmeticType = Widening.arithmetic(nums);
                switch (arithmeticType) {
                case Widening.TYPE_BIGINTEGER:
                    switch (type) {
                    case Arithmetic.BITWISE_AND:
                        nums[0] = ((BigInteger) nums[0]).and((BigInteger) nums[1]);
                        break;
                    case Arithmetic.BITWISE_XOR:
                        nums[0] = ((BigInteger) nums[0]).xor((BigInteger) nums[1]);
                        break;
                    case Arithmetic.BITWISE_OR:
                        nums[0] = ((BigInteger) nums[0]).or((BigInteger) nums[1]);
                        break;
                    default:
                        break;
                    }
                    break;
                case Widening.TYPE_LONG:
                    switch (type) {
                    case Arithmetic.BITWISE_AND:
                        nums[0] = Boxing.box(nums[0].longValue() & nums[1].longValue());
                        break;
                    case Arithmetic.BITWISE_XOR:
                        nums[0] = Boxing.box(nums[0].longValue() ^ nums[1].longValue());
                        break;
                    case Arithmetic.BITWISE_OR:
                        nums[0] = Boxing.box(nums[0].longValue() | nums[1].longValue());
                        break;
                    default:
                        break;
                    }
                    break;
                case Widening.TYPE_INTEGER:
                    switch (type) {
                    case Arithmetic.BITWISE_AND:
                        nums[0] = Boxing.box(nums[0].intValue() & nums[1].intValue());
                        break;
                    case Arithmetic.BITWISE_XOR:
                        nums[0] = Boxing.box(nums[0].intValue() ^ nums[1].intValue());
                        break;
                    case Arithmetic.BITWISE_OR:
                        nums[0] = Boxing.box(nums[0].intValue() | nums[1].intValue());
                        break;
                    default:
                        break;
                    }
                    break;
                default:
                    throw new ArithmeticException("illegal operand type.");
                }
            }
            return nums[0];
        } catch (final ClassCastException e) {
            throw buildArithmeticException("is not Number.", e);
        } catch (final NumberFormatException e) {
            throw buildArithmeticException("number format error.", e);
        }
    }

    static Object shift(int type, Object v, Object n) {
        try {
            final Number val = toNumber(v);
            final Number shift = toNumber(n);
            // BigDecimal or Double or Float is illegal
            if (!isShiftOperand(val, shift)) {
                throw new ArithmeticException("illegal operand type.");
            }
            // BigInteger
            if (val instanceof BigInteger && isIntegerOperand(shift)) {
                switch (type) {
                case Arithmetic.BIT_SHIFT_LEFT:
                    return ((BigInteger) val).shiftLeft(shift.intValue());
                case Arithmetic.BIT_SHIFT_RIGHT:
                    return ((BigInteger) val).shiftRight(shift.intValue());
                case Arithmetic.BIT_SHIFT_LOGICAL_RIGHT:
                    throw new ArithmeticException("illegal operand type.");
                default:
                    break;
                }
            } else
            // Long
            if (val instanceof Long && shift instanceof Long) {
                switch (type) {
                case Arithmetic.BIT_SHIFT_LEFT:
                    return Boxing.box(val.longValue() << shift.longValue());
                case Arithmetic.BIT_SHIFT_RIGHT:
                    return Boxing.box(val.longValue() >> shift.longValue());
                case Arithmetic.BIT_SHIFT_LOGICAL_RIGHT:
                    return Boxing.box(val.longValue() >>> shift.longValue());
                default:
                    break;
                }
            } else
            // (Integer or Short or Byte) and Long
            if (isIntegerOperand(val) && shift instanceof Long) {
                switch (type) {
                case Arithmetic.BIT_SHIFT_LEFT:
                    return Boxing.box(val.intValue() << shift.longValue());
                case Arithmetic.BIT_SHIFT_RIGHT:
                    return Boxing.box(val.intValue() >> shift.longValue());
                case Arithmetic.BIT_SHIFT_LOGICAL_RIGHT:
                    return Boxing.box(val.intValue() >>> shift.longValue());
                default:
                    break;
                }
            } else
            // Long and (Integer or Short or Byte)
            if (val instanceof Long && isIntegerOperand(shift)) {
                switch (type) {
                case Arithmetic.BIT_SHIFT_LEFT:
                    return Boxing.box(val.longValue() << shift.intValue());
                case Arithmetic.BIT_SHIFT_RIGHT:
                    return Boxing.box(val.longValue() >> shift.intValue());
                case Arithmetic.BIT_SHIFT_LOGICAL_RIGHT:
                    return Boxing.box(val.longValue() >>> shift.intValue());
                default:
                    break;
                }
            } else
            // Integer or Short or Byte
            if (isIntegerOperand(val, shift)) {
                switch (type) {
                case Arithmetic.BIT_SHIFT_LEFT:
                    return Boxing.box(val.intValue() << shift.intValue());
                case Arithmetic.BIT_SHIFT_RIGHT:
                    return Boxing.box(val.intValue() >> shift.intValue());
                case Arithmetic.BIT_SHIFT_LOGICAL_RIGHT:
                    return Boxing.box(val.intValue() >>> shift.intValue());
                default:
                    break;
                }
            }
        } catch (final ClassCastException e) {
            throw buildArithmeticException("is not Number.", e);
        } catch (final NumberFormatException e) {
            throw buildArithmeticException("number format error.", e);
        }
        throw new ArithmeticException("illegal operand type.");
    }

    static Object shift(int type, List vs) {
        if (isNumberOperand(vs)) {
            return shiftNumber(type, vs);
        }
        throw new ArithmeticException("illegal operand type.");
    }

    private static Object shiftNumber(int type, List vs) {
        try {
            final Iterator i = vs.iterator();
            Number val = toNumber(i.next());
            Number shift = null;
            while (i.hasNext()) {
                shift = toNumber(i.next());
                // BigDecimal or Double or Float is illegal
                if (!isShiftOperand(val, shift)) {
                    throw new ArithmeticException("illegal operand type.");
                }
                // BigInteger
                if (val instanceof BigInteger && isIntegerOperand(shift)) {
                    switch (type) {
                    case Arithmetic.BIT_SHIFT_LEFT:
                        val = ((BigInteger) val).shiftLeft(shift.intValue());
                        break;
                    case Arithmetic.BIT_SHIFT_RIGHT:
                        val = ((BigInteger) val).shiftRight(shift.intValue());
                        break;
                    case Arithmetic.BIT_SHIFT_LOGICAL_RIGHT:
                        throw new ArithmeticException("illegal operand type.");
                    default:
                        break;
                    }
                    break;
                } else
                // Long
                if (val instanceof Long && shift instanceof Long) {
                    switch (type) {
                    case Arithmetic.BIT_SHIFT_LEFT:
                        val = Boxing.box(val.longValue() << shift.longValue());
                        break;
                    case Arithmetic.BIT_SHIFT_RIGHT:
                        val = Boxing.box(val.longValue() >> shift.longValue());
                        break;
                    case Arithmetic.BIT_SHIFT_LOGICAL_RIGHT:
                        val = Boxing.box(val.longValue() >>> shift.longValue());
                        break;
                    default:
                        break;
                    }
                    break;
                } else
                // (Integer or Short or Byte) and Long
                if (isIntegerOperand(val) && shift instanceof Long) {
                    switch (type) {
                    case Arithmetic.BIT_SHIFT_LEFT:
                        val = Boxing.box(val.intValue() << shift.longValue());
                        break;
                    case Arithmetic.BIT_SHIFT_RIGHT:
                        val = Boxing.box(val.intValue() >> shift.longValue());
                        break;
                    case Arithmetic.BIT_SHIFT_LOGICAL_RIGHT:
                        val = Boxing.box(val.intValue() >>> shift.longValue());
                        break;
                    default:
                        break;
                    }
                    break;
                } else
                // Long and (Integer or Short or Byte)
                if (val instanceof Long && isIntegerOperand(shift)) {
                    switch (type) {
                    case Arithmetic.BIT_SHIFT_LEFT:
                        val = Boxing.box(val.longValue() << shift.intValue());
                        break;
                    case Arithmetic.BIT_SHIFT_RIGHT:
                        val = Boxing.box(val.longValue() >> shift.intValue());
                        break;
                    case Arithmetic.BIT_SHIFT_LOGICAL_RIGHT:
                        val = Boxing.box(val.longValue() >>> shift.intValue());
                        break;
                    default:
                        break;
                    }
                    break;
                } else
                // Integer or Short or Byte
                if (isIntegerOperand(val, shift)) {
                    switch (type) {
                    case Arithmetic.BIT_SHIFT_LEFT:
                        val = Boxing.box(val.intValue() << shift.intValue());
                        break;
                    case Arithmetic.BIT_SHIFT_RIGHT:
                        val = Boxing.box(val.intValue() >> shift.intValue());
                        break;
                    case Arithmetic.BIT_SHIFT_LOGICAL_RIGHT:
                        val = Boxing.box(val.intValue() >>> shift.intValue());
                        break;
                    default:
                        break;
                    }
                    break;
                } else {
                    throw new ArithmeticException("illegal operand type.");
                }
            }
            return val;
        } catch (final ClassCastException e) {
            throw buildArithmeticException("is not Number.", e);
        } catch (final NumberFormatException e) {
            throw buildArithmeticException("number format error.", e);
        }
    }

    static Boolean instanceOf(List vs) {
        try {
            final Iterator i = vs.iterator();
            final Object e = i.next();
            while (i.hasNext()) {
                if (!((Class) i.next()).isInstance(e)) {
                    return Boolean.FALSE;
                }
            }
            return Boolean.TRUE;
        } catch (final ClassCastException e) {
            throw buildArithmeticException("is not java.lang.Class.", e);
        }
    }

    static Boolean oeq(Object v1, Object v2) {
        return Boxing.box(isEquals(v1, v2));
    }

    static Boolean oeq(List vs) {
        final Iterator i = vs.iterator();
        final Object e = i.next();
        while (i.hasNext()) {
            if (!isEquals(e, i.next())) {
                return Boolean.FALSE;
            }
        }
        return Boolean.TRUE;
    }

    static Boolean req(List vs) {
        final Iterator i = vs.iterator();
        final Object e = i.next();
        while (i.hasNext()) {
            if (e != i.next()) {
                return Boolean.FALSE;
            }
        }
        return Boolean.TRUE;
    }

    static String concat(Object v1, Object v2) {
        final String s1 = String.valueOf(v1);
        final String s2 = String.valueOf(v2);
        return s1.concat(s2);
    }

    static String concat(List vs) {
        final StringBuffer sb = new StringBuffer(16 * vs.size());
        final Iterator i = vs.iterator();
        while (i.hasNext()) {
            sb.append(i.next());
        }
        return sb.toString();
    }

    static String concatNvl(Object v1, Object v2) {
        final StringBuffer sb = new StringBuffer(16 * 2);
        if (v1 != null) {
            sb.append(v1);
        }
        if (v2 != null) {
            sb.append(v2);
        }
        return sb.toString();
    }

    static String concatNvl(List vs) {
        final StringBuffer sb = new StringBuffer(16 * vs.size());
        final Iterator i = vs.iterator();
        while (i.hasNext()) {
            final Object o = i.next();
            if (o != null) {
                sb.append(o);
            }
        }
        return sb.toString();
    }

    /*
     * Helper
     */

    static void ifIllegalArgumentsSize(List vs) {
        if (2 > vs.size()) {
            throw new ArithmeticException("illegal arguments size. 2 > size.");
        }
    }

    private static boolean isEquals(Object o1, Object o2) {
        return o1 == o2 || (o1 != null && o1.equals(o2));
    }

    /* 数値型の配列に変換して返却します */
    private static Number[] toNumber(Object v1, Object v2) {
        return new Number[] { toNumber(v1), toNumber(v2) };
    }

    /* 数値型に変換して返却します */
    private static Number toNumber(Object o) {
        return (o instanceof Character) ? Boxing.box((int) ((Character) o).charValue()) : (Number) o;
    }

    private static boolean isStringOperand(Object v1, Object v2) {
        return (v1 instanceof String) || (v2 instanceof String);
    }

    private static boolean isBooleanOperand(Object v1, Object v2) {
        return (v1 instanceof Boolean) && (v2 instanceof Boolean);
    }

    private static boolean isComparableOperand(Object v1, Object v2) {
        return (v1 instanceof Comparable) && (v2 instanceof Comparable);
    }

    private static boolean isNumberOperand(Object v1, Object v2) {
        return isNumberOperand(v1) && isNumberOperand(v2);
    }

    private static boolean isNumberOperand(Object v) {
        return (v instanceof Number || v instanceof Character);
    }

    private static boolean isShiftOperand(Object v1, Object v2) {
        return isShiftOperand(v1) && isShiftOperand(v2);
    }

    private static boolean isShiftOperand(Object v) {
        return !(v instanceof BigDecimal || v instanceof Double || v instanceof Float);
    }

    private static boolean isIntegerOperand(Object v1, Object v2) {
        return isIntegerOperand(v1) && isIntegerOperand(v2);
    }

    private static boolean isIntegerOperand(Object v) {
        return (v instanceof Integer || v instanceof Short || v instanceof Byte);
    }

    private static boolean isNumberOperand(List vs) {
        ifIllegalArgumentsSize(vs);
        return isNumberOperand(vs.get(0), vs.get(1));
    }

    private static ArithmeticException buildArithmeticException(String msg, Exception cause) {
        final ArithmeticException e = new ArithmeticException(msg);
        e.initCause(cause);
        return e;
    }
}
