/*
SRWebHistory.m

Author: Makoto Kinoshita

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "SRWebHistory.h"

#import "SRConstants.h"

@interface SRWebHistory (private)
- (void)_updateTreeWebHistory;
@end

@implementation SRWebHistory

//--------------------------------------------------------------//
#pragma mark -- Initialize --
//--------------------------------------------------------------//

+ (id)sharedInstance
{
    static SRWebHistory*    _sharedInstance = nil;
    if (!_sharedInstance) {
        _sharedInstance = [[SRWebHistory alloc] init];
    }
    
    return _sharedInstance;
}

- (id)init
{
    self = [super init];
    if (!self) {
        return nil;
    }
    
    // Initialize instance variables
    _treeWebHistory = [[NSMutableArray array] retain];
    
    // Register notification
    NSNotificationCenter*   center;
    center = [NSNotificationCenter defaultCenter];
    [center addObserver:self selector:@selector(webHistoryItemsUpdated:) 
            name:WebHistoryItemsAddedNotification object:nil];
    [center addObserver:self selector:@selector(webHistoryItemsUpdated:) 
            name:WebHistoryItemsRemovedNotification object:nil];
    [center addObserver:self selector:@selector(webHistoryItemsUpdated:) 
            name:WebHistoryAllItemsRemovedNotification object:nil];
    
    // Update tree web history
    [self _updateTreeWebHistory];
    
    return self;
}

- (void)dealloc
{
    // Release instance variables
    [_treeWebHistory release], _treeWebHistory = nil;
    
    // Remove observer
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    
    [super dealloc];
}

//--------------------------------------------------------------//
#pragma mark -- Web history --
//--------------------------------------------------------------//

- (void)_updateTreeWebHistory
{
    // Remove old history
    [_treeWebHistory removeAllObjects];
    
    // Add last visited days
    WebHistory*     webHistory;
    NSEnumerator*   enumerator;
    NSCalendarDate* calendarDate;
    webHistory = [WebHistory optionalSharedHistory];
    enumerator = [[webHistory orderedLastVisitedDays] objectEnumerator];
    while (calendarDate = [enumerator nextObject]) {
        NSString*   altTitle;
        altTitle = [calendarDate descriptionWithCalendarFormat:NSLocalizedString(@"%a, %B %e", nil) 
                locale:[[NSUserDefaults standardUserDefaults] dictionaryRepresentation]];
        
        NSMutableDictionary*    dateInfo;
        dateInfo = [NSMutableDictionary dictionary];
        [dateInfo setObject:calendarDate forKey:@"date"];
        [dateInfo setObject:altTitle forKey:@"altTitle"];
        [dateInfo setObject:[webHistory orderedItemsLastVisitedOnDay:calendarDate] forKey:@"children"];
        
        [_treeWebHistory addObject:dateInfo];
    }
    
    // Notify update
    [[NSNotificationCenter defaultCenter] 
            postNotificationName:SRWebHistoryUpdated object:self];
}

- (NSArray*)treeWebHistory
{
    return _treeWebHistory;
}

//--------------------------------------------------------------//
#pragma mark -- WebHistory notification --
//--------------------------------------------------------------//

- (void)webHistoryItemsUpdated:(NSNotification*)notification
{
    // Update tree web history
    [self _updateTreeWebHistory];
}

@end

#pragma mark -

@implementation WebHistoryItem (AltTitle)

//--------------------------------------------------------------//
#pragma mark -- Title --
//--------------------------------------------------------------//

- (NSString*)altTitle
{
    // Get URL string
    NSString*   urlString;
    urlString = [self URLString];
    if ([urlString hasPrefix:@"shiira://shelf"]) {
        return NSLocalizedString(@"Shelf", nil);
    }
    
    // Get title
    NSString*   title;
    title = [self title];
    if (title && [title length] > 0) {
        return title;
    }
    
    if (urlString && [urlString length] > 0) {
        return urlString;
    }
    
    return [NSString stringWithFormat:@"(%@)", NSLocalizedString(@"Untitled", nil)];
}

- (NSImage*)altIcon
{
    // Get URL string
    NSString*   urlString;
    urlString = [self URLString];
    if ([urlString hasPrefix:@"shiira://shelf"]) {
        return [NSImage imageNamed:@"shelfPage"];
    }
    
    return [self icon];
}

@end
