/* Fix various `stat' maladies on Windows, and improve it with
   native security-related features to read owner and group info.  */

#include <stddef.h>
#include <time.h>
#include <sys/time.h>
#include <sys/utime.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <limits.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <pwd.h>
#include <grp.h>

#include <sys/param.h>
#include <dirent.h>
#include <fts.h>

#ifndef S_IFLNK
#define S_ISLNK(S)  (0)
#endif

#include "archive.h"
#define _WIN32_WINNT 0x0500
#define WINVER 0x0501
#include <windows.h>
#include <lmcons.h>
#include <accctrl.h>
#include <aclapi.h>
#include <sddl.h>
#include <lmcons.h>

struct w32stat_internal {
  DWORD w32vol_serial;
  FILETIME w32ctime;
  FILETIME w32atime;
  FILETIME w32mtime;
  int w32nlink;
  __int64 w32ino;
  int w32user_rid;
  int w32group_rid;
  char w32uname[UNLEN+1];
  char w32gname[GNLEN+1];
};

struct w32stat_id {
  DWORD rid;
  struct w32stat_id *next;
  char name[GNLEN+1];
  unsigned char sid[];
};

static struct w32stat_id *w32stat_idlist;

#define alloca(n) __builtin_alloca(n)

static int
w32stat_cached_id (PSID sid, DWORD *id, char *name)
{
  struct w32stat_id *tail, *found;

  for (found = NULL, tail = w32stat_idlist; tail; tail = tail->next)
    {
      if (EqualSid ((PSID)tail->sid, sid))
	{
	  found = tail;
	  break;
	}
    }
  if (found)
    {
      *id = found->rid;
      strcpy (name, found->name);
      return 1;
    }
  else
    return 0;
}

static void
w32stat_add_to_cache (PSID sid, DWORD id, char *name)
{
  DWORD sid_len = GetLengthSid (sid);
  struct w32stat_id *new_entry =
    malloc (offsetof (struct w32stat_id, sid) + sid_len);

  /* We ignore malloc failures: it just means the entry will not be
     cached, i.e. graceful degradation.  */
  if (new_entry)
    {
      new_entry->rid = id;
      strcpy (new_entry->name, name);
      CopySid (sid_len, (PSID)new_entry->sid, sid);
      new_entry->next = w32stat_idlist;
      w32stat_idlist = new_entry;
    }
}

/* User & Group IDs: we use the "relative SID" (RID) that is the
   last subauthority encoded in the respective binary SID.  */
static long
w32stat_get_id (PSID sid)
{
  unsigned n_subauthorities;

  n_subauthorities = *GetSidSubAuthorityCount (sid);
  if (n_subauthorities < 1)
    return 0;	/* the "World" RID */
  return *(long *)GetSidSubAuthority (sid, n_subauthorities - 1);
}

static void
w32stat_get_file_security (const char *fname,
			   int *suid, int *sgid,
			   char *uname, char *gname)
{
  PSECURITY_DESCRIPTOR psd = NULL;
  DWORD sd_len;
  PSID owner_sid, group_sid;
  DWORD uid, gid;
  DWORD name_len = UNLEN+1;
  BOOL dflt;
  char domain[1024];
  DWORD domain_len = sizeof(domain);
  SID_NAME_USE ignore;
  SECURITY_INFORMATION si =
    OWNER_SECURITY_INFORMATION | GROUP_SECURITY_INFORMATION;
  char machine[MAX_COMPUTERNAME_LENGTH+1];
  char *cmp = NULL;

  if (!GetFileSecurity (fname, si, psd, 0, &sd_len))
    {
      if (GetLastError () != ERROR_INSUFFICIENT_BUFFER)
	return;
    }

  psd = alloca (sd_len);
  if (!GetFileSecurity (fname, si, psd, sd_len, &sd_len))
    return;

  if (!GetSecurityDescriptorOwner (psd, &owner_sid, &dflt)
      || !GetSecurityDescriptorGroup (psd, &group_sid, &dflt))
    return;

  if (!IsValidSid (owner_sid) || !IsValidSid (group_sid))
    return;

  /* Try the cache first, to avoid costly LookupAccountSid call.  */
  if (!w32stat_cached_id (owner_sid, &uid, uname))
    {
      uid = w32stat_get_id (owner_sid);
      /* If FNAME is a UNC, we need to lookup account on the specified
	 machine.  */
      if (fname[0] == '\\' && fname[1] == '\\' && fname[2] != '\0')
	{
	  const char *s;
	  char *p;

	  for (s = fname + 2, p = machine; *s && *s != '\\'; s++, p++)
	    *p = *s;
	  *p = '\0';
	  cmp = machine;
	}

      if (LookupAccountSid (cmp, owner_sid, uname, &name_len,
			    domain, &domain_len, &ignore)
	  && name_len <= UNLEN+1)
	w32stat_add_to_cache (owner_sid, uid, uname);
    }

  if (!w32stat_cached_id (group_sid, &gid, gname))
    {
      gid = w32stat_get_id (group_sid);

      /* If FNAME is a UNC, we need to lookup account on the specified
	 machine.  */
      if (!cmp && fname[0] == '\\' && fname[1] == '\\' && fname[2] != '\0')
	{
	  const char *s;
	  char *p;

	  for (s = fname + 2, p = machine; *s && *s != '\\'; s++, p++)
	    *p = *s;
	  *p = '\0';
	  cmp = machine;
	}

      name_len = GNLEN+1;
      domain_len = sizeof(domain);
      if (LookupAccountSid (cmp, group_sid, gname, &name_len,
			    NULL, &domain_len, &ignore)
	  && name_len <= GNLEN+1)
	w32stat_add_to_cache (group_sid, gid, gname);
    }

  *suid = uid;
  *sgid = gid;
}

static int
w32stat (const char *name, struct w32stat_internal *w32st)
{
  static OSVERSIONINFO verinfo;
  HANDLE fh;

  fh = CreateFile (name, 0, 0, NULL, OPEN_EXISTING,
		   FILE_FLAG_BACKUP_SEMANTICS, NULL);
  if (fh == INVALID_HANDLE_VALUE)
    return -1;
  else
    {
      BY_HANDLE_FILE_INFORMATION finfo;
      BOOL rv = GetFileInformationByHandle (fh, &finfo);

      if (rv)
	{
	  w32st->w32ctime = finfo.ftCreationTime;
	  w32st->w32atime = finfo.ftLastAccessTime;
	  w32st->w32mtime = finfo.ftLastWriteTime;
	  w32st->w32vol_serial = finfo.dwVolumeSerialNumber;
	  w32st->w32nlink = finfo.nNumberOfLinks;
	  w32st->w32ino = finfo.nFileIndexHigh;
	  w32st->w32ino <<= 32;
	  w32st->w32ino += finfo.nFileIndexLow;
	}
      CloseHandle (fh);
      if (!rv)
	return -1;
    }

  w32st->w32uname[0] = '\0';
  w32st->w32gname[0] = '\0';

  /* Security APIs are not supported on Windows 9x.  */
  if (verinfo.dwOSVersionInfoSize == 0)
    {
      verinfo.dwOSVersionInfoSize = sizeof (OSVERSIONINFO);
      GetVersionEx (&verinfo);
    }
  if (verinfo.dwPlatformId != VER_PLATFORM_WIN32_WINDOWS)
    w32stat_get_file_security (name,
			       &w32st->w32user_rid, &w32st->w32group_rid,
			       w32st->w32uname, w32st->w32gname);
  else
    {
      w32st->w32user_rid = 513;
      if (stricmp (w32st->w32uname, "administrator") == 0)
	w32st->w32user_rid = 500;
      else
	w32st->w32group_rid = 1004;
    }

  if (!*w32st->w32uname)
    {
      DWORD ulen = UNLEN+1;

      if (!GetUserName (w32st->w32uname, &ulen))
	strcpy (w32st->w32uname, "Unknown");
    }
  if (!*w32st->w32gname)
    strcpy (w32st->w32gname, "None");

  return 0;
}

int
lstat (const char *fn, struct xar_stat *xst)
{
  size_t nlen = strlen (fn);
  int retval = -1;
  char tmp[MAX_PATH];
  struct _stati64 st;
  struct w32stat_internal wst;

  /* If we have empty or very long file name, let `stat' handle that.  */
  if (nlen >= MAX_PATH || nlen == 0)
    retval = _stati64 (fn, &st);
  else
    {
      size_t prefix = fn[1] == ':' ? 2 : 0;

      /* Windows `stat' does not like trailing slashes.  */
      memcpy (tmp, fn, nlen + 1);
      for (--nlen; nlen > prefix && (tmp[nlen] == '/' || tmp[nlen] == '\\');
	   --nlen)
	tmp[nlen] = '\0';

      retval = _stati64 (tmp, &st);
    }

  if (retval)
    return retval;

  xst->st_mode = st.st_mode;
  xst->st_size = st.st_size;
  xst->st_atime = st.st_atime;
  xst->st_mtime = st.st_mtime;
  xst->st_ctime = st.st_ctime;
  xst->st_rdev = st.st_rdev;

  retval = w32stat (tmp, &wst);

  xst->st_dev = wst.w32vol_serial;
  xst->st_ino = wst.w32ino;
  xst->st_nlink = wst.w32nlink;
  xst->st_uid = wst.w32user_rid;
  xst->st_gid = wst.w32group_rid;
  strcpy (xst->st_uname, wst.w32uname);
  strcpy (xst->st_gname, wst.w32gname);

  return retval;
}

/* NTSec implementations for pwd/grp functions.  */

#define SID_MAX_SUB_AUTHORITIES  15
#define SECURITY_MAX_SID_SIZE  \
  (sizeof(SID) - sizeof(DWORD) + (SID_MAX_SUB_AUTHORITIES * sizeof(DWORD)))

static HANDLE
get_process_token (void)
{
  HANDLE       token = NULL;
  BOOL         result;

  result = OpenProcessToken (GetCurrentProcess (), TOKEN_QUERY, &token);
  if (!result)
    token = NULL;
  return token;
}

uid_t
geteuid (void)
{
  HANDLE         token = get_process_token ();
  unsigned char *buf = NULL;
  DWORD          needed;
  DWORD          blen = 0;
  BOOL           result;

  if (token)
    {
      result = GetTokenInformation (token, TokenUser, NULL, 0, &blen);
      if (!result && GetLastError () == ERROR_INSUFFICIENT_BUFFER)
	{
	  buf = malloc (blen);
	  if (buf)
	    {
	      result = GetTokenInformation (token, TokenUser,
					    (LPVOID)buf, blen, &needed);
	      if (result)
		return w32stat_get_id (((TOKEN_USER *)buf)->User.Sid);
	    }
	}
    }
  return 0;	/* root */
}

gid_t
getegid (void)
{
  HANDLE         token = get_process_token ();
  unsigned char *buf = NULL;
  DWORD          needed;
  DWORD          blen = 0;
  BOOL           result;

  if (token)
    {
      result = GetTokenInformation (token, TokenPrimaryGroup, NULL, 0, &blen);
      if (!result && GetLastError () == ERROR_INSUFFICIENT_BUFFER)
	{
	  buf = malloc (blen);
	  if (buf)
	    {
	      result = GetTokenInformation (token, TokenPrimaryGroup,
					    (LPVOID)buf, blen, &needed);
	      if (result)
		return w32stat_get_id (((TOKEN_PRIMARY_GROUP *)buf)->PrimaryGroup);
	    }
	}
    }
  return 0;	/* root */
}

struct passwd *
getpwnam (const char *u)
{
  char  domain[1025];
  DWORD dlength = sizeof (domain);
  DWORD sid_size = SECURITY_MAX_SID_SIZE;
  PSID  psid;
  BOOL  result;
  SID_NAME_USE dtype;
  struct passwd *pw = NULL;

  psid = LocalAlloc (LMEM_FIXED, sid_size);
  result = LookupAccountName (NULL, u, psid, &sid_size,
			      domain, &dlength, &dtype);
  if (result)
    {
      pw = malloc (sizeof(struct passwd) + strlen (u) + 1);
      if (pw)
	{
	  pw->pw_name = strcpy ((char *)pw + sizeof(struct passwd), u);
	  pw->pw_uid = w32stat_get_id (psid);
	}
    }
  LocalFree (psid);
  return pw;
}

struct group *
getgrnam (const char *g)
{
  char  domain[1025];
  DWORD dlength = sizeof (domain);
  DWORD sid_size = SECURITY_MAX_SID_SIZE;
  PSID  psid;
  BOOL  result;
  SID_NAME_USE dtype;
  struct group *gr = NULL;

  psid = LocalAlloc (LMEM_FIXED, sid_size);
  result = LookupAccountName (NULL, g, psid, &sid_size,
			      domain, &dlength, &dtype);
  if (result)
    {
      gr = malloc (sizeof(struct group) + strlen (g) + 1);
      if (gr)
	{
	  gr->gr_name = strcpy ((char *)gr + sizeof(struct group), g);
	  gr->gr_gid = w32stat_get_id (psid);
	}
    }
  LocalFree (psid);
  return gr;
}

/* A simple enough emulation of strptime.  */

char *
strptime (const char *buf, const char *fmt, struct tm *tm)
{
  char *retval = (char *)buf;

  /* We only support the format descriptors produced by stat.c.  */
  if (0 == strcmp (fmt, "%Y-%m-%dT%H:%M:%S")
      && 6 == sscanf (buf, "%4d-%2d-%2dT%2d:%2d:%2d",
		      &tm->tm_year, &tm->tm_mon, &tm->tm_mday,
		      &tm->tm_hour, &tm->tm_min, &tm->tm_sec))
    {
      tm->tm_year -= 1900;
      retval = (char *)buf + strlen (buf);
    }

  return retval;
}

/* An emulation of timegm.  The idea stolen from libarchive.  */

time_t
timegm (struct tm *t)
{
  mktime(t);
  return (t->tm_sec + t->tm_min * 60 + t->tm_hour * 3600
	  + t->tm_yday * 86400 + (t->tm_year - 70) * 31536000
	  + ((t->tm_year - 69) / 4) * 86400 -
	  ((t->tm_year - 1) / 100) * 86400
	  + ((t->tm_year + 299) / 400) * 86400);
}

/* An amulation of utimes.  */
int
utimes (const char *fn, struct timeval tvp[2])
{
  struct utimbuf ut;

  ut.actime = tvp[0].tv_sec;
  ut.modtime = tvp[1].tv_sec;
  return utime (fn, &ut);
}

/* An emulation of link.  */
int
link (const char *oldname, const char *newname)
{
  if (!CreateHardLink(newname, oldname, NULL)) {
    errno = EACCES;
    return -1;
  }
  return 0;
}

int
symlink (const char *old, const char *new)
{
  if (!CopyFile (old, new, TRUE))
    {
      DWORD err = GetLastError ();

      switch (err)
	{
	  case ERROR_FILE_EXISTS:
	  case ERROR_ACCESS_DENIED:
	    errno = EEXIST;
	    break;
	  case ERROR_CANNOT_COPY:
	    errno = EACCES;
	    break;
	  default:
	    errno = ENOSPC;
	    break;
	}
      return -1;
    }
  return 0;
}

int
mkstemp (char *tpl)
{
  char *tmp_fn = mktemp(tpl);

  if (!tmp_fn || tmp_fn[0] == '\0')
    return -1;
  return open(tmp_fn, O_RDWR | O_BINARY | O_CREAT | O_EXCL, 0600);
}

struct tm *
gmtime_r(const time_t *t, struct tm *res)
{
  struct tm *tem = gmtime(t);

  if (tem)
    *res = *tem;
  return tem;
}

/* Shamelessly stolen from gnulib.  */
char *
strsep(char **stringp, const char *delim)
{
  char *start = *stringp;
  char *ptr;

  if (start == NULL)
    return NULL;

  /* Optimize the case of no delimiters.  */
  if (delim[0] == '\0') {
    *stringp = NULL;
    return start;
  }

  /* Optimize the case of one delimiter.  */
  if (delim[1] == '\0')
    ptr = strchr (start, delim[0]);
  else
    /* The general case.  */
    ptr = strpbrk (start, delim);
  if (ptr == NULL) {
    *stringp = NULL;
    return start;
  }

  *ptr = '\0';
  *stringp = ptr + 1;

  return start;
}

/* Of all the possible methods of traversing a directory tree, why, oh
   why did they choose the least portable one??

   The below was stolen from libgw32c, the Windows port of glibc, with
   minimal changes.  */

/*-
 * Copyright (c) 1990, 1993, 1994
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */


/* Largest alignment size needed, minus one.
   Usually long double is the worst case.  */
#ifndef ALIGNBYTES
#define ALIGNBYTES	(__alignof__ (long double) - 1)
#endif
/* Align P to that size.  */
#ifndef ALIGN
#define	ALIGN(p)	(((unsigned long int) (p) + ALIGNBYTES) & ~ALIGNBYTES)
#endif

#define stat(F,B)   lstat(F,B)

static FTSENT	*fts_alloc (FTS *, const char *, int) ;
static FTSENT	*fts_build (FTS *, int) ;
static void	 fts_lfree (FTSENT *) ;
static void	 fts_load (FTS *, FTSENT *) ;
static size_t	 fts_maxarglen (char * const *) ;
static void	 fts_padjust (FTS *, FTSENT *) ;
static int	 fts_palloc (FTS *, size_t) ;
static FTSENT	*fts_sort (FTS *, FTSENT *, int) ;
static u_short	 fts_stat (FTS *, FTSENT *, int) ;
static int       fts_safe_changedir (FTS *, FTSENT *, int, const char *);

#ifndef MAX
#define MAX(a, b)	({ __typeof__ (a) _a = (a); \
			   __typeof__ (b) _b = (b); \
			   _a > _b ? _a : _b; })
#endif

#define	ISDOT(a)	(a[0] == '.' && (!a[1] || (a[1] == '.' && !a[2])))

#define CLR(opt)	(sp->fts_options &= ~(opt))
#define	ISSET(opt)	(sp->fts_options & (opt))
#define	SET(opt)	(sp->fts_options |= (opt))

#define	FCHDIR(sp, fd)	(!ISSET(FTS_NOCHDIR) && fchdir(fd))

/* fts_build flags */
#define	BCHILD		1		/* fts_children */
#define	BNAMES		2		/* fts_children, names only */
#define	BREAD		3		/* fts_read */

static inline int
fchdir (int fd)
{
	errno = ENOSYS;
	return -1;
}

static inline long
dirfd (DIR *d)
{
	return d->dd_handle;
}

static inline int
d_namelen (struct dirent *de)
{
	return de->d_namlen;
}

FTS *
fts_open(argv, options, compar)
	char * const *argv;
	register int options;
	int (*compar) (const FTSENT **, const FTSENT **);
{
	register FTS *sp;
	register FTSENT *p, *root;
	register int nitems;
	FTSENT *parent, *tmp;
	int len;

	/* Options check. */
	if (options & ~FTS_OPTIONMASK) {
		errno = EINVAL;
		return (NULL);
	}

	/* Allocate/initialize the stream */
	if ((sp = malloc((u_int)sizeof(FTS))) == NULL)
		return (NULL);
	memset(sp, 0, sizeof(FTS));
	sp->fts_compar = (int (*) (const void *, const void *)) compar;
	sp->fts_options = options;

	/* Logical walks turn on NOCHDIR; symbolic links are too hard. */
	if (ISSET(FTS_LOGICAL))
		SET(FTS_NOCHDIR);

	/*
	 * Start out with 1K of path space, and enough, in any case,
	 * to hold the user's paths.
	 */
#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif
	if (fts_palloc(sp, MAX(fts_maxarglen(argv), MAXPATHLEN)))
		goto mem1;

	/* Allocate/initialize root's parent. */
	if ((parent = fts_alloc(sp, "", 0)) == NULL)
		goto mem2;
	parent->fts_level = FTS_ROOTPARENTLEVEL;

	/* Allocate/initialize root(s). */
	for (root = NULL, nitems = 0; *argv != NULL; ++argv, ++nitems) {
		/* Don't allow zero-length paths. */
		if ((len = strlen(*argv)) == 0) {
			errno = ENOENT;
			goto mem3;
		}

		p = fts_alloc(sp, *argv, len);
		p->fts_level = FTS_ROOTLEVEL;
		p->fts_parent = parent;
		p->fts_accpath = p->fts_name;
		p->fts_info = fts_stat(sp, p, ISSET(FTS_COMFOLLOW));

		/* Command-line "." and ".." are real directories. */
		if (p->fts_info == FTS_DOT)
			p->fts_info = FTS_D;

		/*
		 * If comparison routine supplied, traverse in sorted
		 * order; otherwise traverse in the order specified.
		 */
		if (compar) {
			p->fts_link = root;
			root = p;
		} else {
			p->fts_link = NULL;
			if (root == NULL)
				tmp = root = p;
			else {
				tmp->fts_link = p;
				tmp = p;
			}
		}
	}
	if (compar && nitems > 1)
		root = fts_sort(sp, root, nitems);

	/*
	 * Allocate a dummy pointer and make fts_read think that we've just
	 * finished the node before the root(s); set p->fts_info to FTS_INIT
	 * so that everything about the "current" node is ignored.
	 */
	if ((sp->fts_cur = fts_alloc(sp, "", 0)) == NULL)
		goto mem3;
	sp->fts_cur->fts_link = root;
	sp->fts_cur->fts_info = FTS_INIT;

	/*
	 * If using chdir(2), grab a file descriptor pointing to dot to ensure
	 * that we can get back here; this could be avoided for some paths,
	 * but almost certainly not worth the effort.  Slashes, symbolic links,
	 * and ".." are all fairly nasty problems.  Note, if we can't get the
	 * descriptor we run anyway, just more slowly.
	 */
	if (!ISSET(FTS_NOCHDIR)
	    && (sp->fts_rfd = open(".", O_RDONLY, 0)) < 0)
		SET(FTS_NOCHDIR);

	return (sp);

mem3:	fts_lfree(root);
	free(parent);
mem2:	free(sp->fts_path);
mem1:	free(sp);
	return (NULL);
}

static void
fts_load(sp, p)
	FTS *sp;
	register FTSENT *p;
{
	register int len;
	register char *cp;

	/*
	 * Load the stream structure for the next traversal.  Since we don't
	 * actually enter the directory until after the preorder visit, set
	 * the fts_accpath field specially so the chdir gets done to the right
	 * place and the user can access the first node.  From fts_open it's
	 * known that the path will fit.
	 */
	len = p->fts_pathlen = p->fts_namelen;
	memmove(sp->fts_path, p->fts_name, len + 1);
	if ((cp = strrchr(p->fts_name, '/')) && (cp != p->fts_name || cp[1])) {
		len = strlen(++cp);
		memmove(p->fts_name, cp, len + 1);
		p->fts_namelen = len;
	}
	p->fts_accpath = p->fts_path = sp->fts_path;
	sp->fts_dev = p->fts_dev;
}

int
fts_close(sp)
	FTS *sp;
{
	register FTSENT *freep, *p;
	int saved_errno;

	/*
	 * This still works if we haven't read anything -- the dummy structure
	 * points to the root list, so we step through to the end of the root
	 * list which has a valid parent pointer.
	 */
	if (sp->fts_cur) {
		for (p = sp->fts_cur; p->fts_level >= FTS_ROOTLEVEL;) {
			freep = p;
			p = p->fts_link != NULL ? p->fts_link : p->fts_parent;
			free(freep);
		}
		free(p);
	}

	/* Free up child linked list, sort array, path buffer. */
	if (sp->fts_child)
		fts_lfree(sp->fts_child);
	if (sp->fts_array)
		free(sp->fts_array);
	free(sp->fts_path);

	/* Return to original directory, save errno if necessary. */
	if (!ISSET(FTS_NOCHDIR)) {
		saved_errno = fchdir(sp->fts_rfd) ? errno : 0;
		(void)close(sp->fts_rfd);

		/* Set errno and return. */
		if (saved_errno != 0) {
			/* Free up the stream pointer. */
			free(sp);
			errno = saved_errno;
			return (-1);
		}
	}

	/* Free up the stream pointer. */
	free(sp);
	return (0);
}

/*
 * Special case of "/" at the end of the path so that slashes aren't
 * appended which would cause paths to be written as "....//foo".
 */
#define	NAPPEND(p)							\
	(p->fts_path[p->fts_pathlen - 1] == '/'				\
	    ? p->fts_pathlen - 1 : p->fts_pathlen)

FTSENT *
fts_read(sp)
	register FTS *sp;
{
	register FTSENT *p, *tmp;
	register int instr;
	register char *t;
	int saved_errno;

	/* If finished or unrecoverable error, return NULL. */
	if (sp->fts_cur == NULL || ISSET(FTS_STOP))
		return (NULL);

	/* Set current node pointer. */
	p = sp->fts_cur;

	/* Save and zero out user instructions. */
	instr = p->fts_instr;
	p->fts_instr = FTS_NOINSTR;

	/* Any type of file may be re-visited; re-stat and re-turn. */
	if (instr == FTS_AGAIN) {
		p->fts_info = fts_stat(sp, p, 0);
		return (p);
	}

	/*
	 * Following a symlink -- SLNONE test allows application to see
	 * SLNONE and recover.  If indirecting through a symlink, have
	 * keep a pointer to current location.  If unable to get that
	 * pointer, follow fails.
	 */
	if (instr == FTS_FOLLOW &&
	    (p->fts_info == FTS_SL || p->fts_info == FTS_SLNONE)) {
		p->fts_info = fts_stat(sp, p, 1);
		if (p->fts_info == FTS_D && !ISSET(FTS_NOCHDIR)) {
			if ((p->fts_symfd = open(".", O_RDONLY, 0)) < 0) {
				p->fts_errno = errno;
				p->fts_info = FTS_ERR;
			} else
				p->fts_flags |= FTS_SYMFOLLOW;
		}
		return (p);
	}

	/* Directory in pre-order. */
	if (p->fts_info == FTS_D) {
		/* If skipped or crossed mount point, do post-order visit. */
		if (instr == FTS_SKIP ||
		    (ISSET(FTS_XDEV) && p->fts_dev != sp->fts_dev)) {
			if (p->fts_flags & FTS_SYMFOLLOW)
				(void)close(p->fts_symfd);
			if (sp->fts_child) {
				fts_lfree(sp->fts_child);
				sp->fts_child = NULL;
			}
			p->fts_info = FTS_DP;
			return (p);
		}

		/* Rebuild if only read the names and now traversing. */
		if (sp->fts_child != NULL && ISSET(FTS_NAMEONLY)) {
			CLR(FTS_NAMEONLY);
			fts_lfree(sp->fts_child);
			sp->fts_child = NULL;
		}

		/*
		 * Cd to the subdirectory.
		 *
		 * If have already read and now fail to chdir, whack the list
		 * to make the names come out right, and set the parent errno
		 * so the application will eventually get an error condition.
		 * Set the FTS_DONTCHDIR flag so that when we logically change
		 * directories back to the parent we don't do a chdir.
		 *
		 * If haven't read do so.  If the read fails, fts_build sets
		 * FTS_STOP or the fts_info field of the node.
		 */
		if (sp->fts_child != NULL) {
			if (fts_safe_changedir(sp, p, -1, p->fts_accpath)) {
				p->fts_errno = errno;
				p->fts_flags |= FTS_DONTCHDIR;
				for (p = sp->fts_child; p != NULL;
				     p = p->fts_link)
					p->fts_accpath =
					    p->fts_parent->fts_accpath;
			}
		} else if ((sp->fts_child = fts_build(sp, BREAD)) == NULL) {
			if (ISSET(FTS_STOP))
				return (NULL);
			return (p);
		}
		p = sp->fts_child;
		sp->fts_child = NULL;
		goto name;
	}

	/* Move to the next node on this level. */
next:	tmp = p;
	if ((p = p->fts_link) != NULL) {
		free(tmp);

		/*
		 * If reached the top, return to the original directory (or
		 * the root of the tree), and load the paths for the next root.
		 */
		if (p->fts_level == FTS_ROOTLEVEL) {
			if (FCHDIR(sp, sp->fts_rfd)) {
				SET(FTS_STOP);
				return (NULL);
			}
			fts_load(sp, p);
			return (sp->fts_cur = p);
		}

		/*
		 * User may have called fts_set on the node.  If skipped,
		 * ignore.  If followed, get a file descriptor so we can
		 * get back if necessary.
		 */
		if (p->fts_instr == FTS_SKIP)
			goto next;
		if (p->fts_instr == FTS_FOLLOW) {
			p->fts_info = fts_stat(sp, p, 1);
			if (p->fts_info == FTS_D && !ISSET(FTS_NOCHDIR)) {
				if ((p->fts_symfd =
				    open(".", O_RDONLY, 0)) < 0) {
					p->fts_errno = errno;
					p->fts_info = FTS_ERR;
				} else
					p->fts_flags |= FTS_SYMFOLLOW;
			}
			p->fts_instr = FTS_NOINSTR;
		}

name:		t = sp->fts_path + NAPPEND(p->fts_parent);
		*t++ = '/';
		memmove(t, p->fts_name, p->fts_namelen + 1);
		return (sp->fts_cur = p);
	}

	/* Move up to the parent node. */
	p = tmp->fts_parent;
	free(tmp);

	if (p->fts_level == FTS_ROOTPARENTLEVEL) {
		/*
		 * Done; free everything up and set errno to 0 so the user
		 * can distinguish between error and EOF.
		 */
		free(p);
		errno = 0;
		return (sp->fts_cur = NULL);
	}

	/* NUL terminate the pathname. */
	sp->fts_path[p->fts_pathlen] = '\0';

	/*
	 * Return to the parent directory.  If at a root node or came through
	 * a symlink, go back through the file descriptor.  Otherwise, cd up
	 * one directory.
	 */
	if (p->fts_level == FTS_ROOTLEVEL) {
		if (FCHDIR(sp, sp->fts_rfd)) {
			SET(FTS_STOP);
			return (NULL);
		}
	} else if (p->fts_flags & FTS_SYMFOLLOW) {
		if (FCHDIR(sp, p->fts_symfd)) {
			saved_errno = errno;
			(void)close(p->fts_symfd);
			errno = saved_errno;
			SET(FTS_STOP);
			return (NULL);
		}
		(void)close(p->fts_symfd);
	} else if (!(p->fts_flags & FTS_DONTCHDIR) &&
		   fts_safe_changedir(sp, p->fts_parent, -1, "..")) {
		SET(FTS_STOP);
		return (NULL);
	}
	p->fts_info = p->fts_errno ? FTS_ERR : FTS_DP;
	return (sp->fts_cur = p);
}

/*
 * Fts_set takes the stream as an argument although it's not used in this
 * implementation; it would be necessary if anyone wanted to add global
 * semantics to fts using fts_set.  An error return is allowed for similar
 * reasons.
 */
/* ARGSUSED */
int
fts_set(sp, p, instr)
	FTS *sp;
	FTSENT *p;
	int instr;
{
	if (instr != 0 && instr != FTS_AGAIN && instr != FTS_FOLLOW &&
	    instr != FTS_NOINSTR && instr != FTS_SKIP) {
		errno = EINVAL;
		return (1);
	}
	p->fts_instr = instr;
	return (0);
}

FTSENT *
fts_children(sp, instr)
	register FTS *sp;
	int instr;
{
	register FTSENT *p;
	int fd;

	if (instr != 0 && instr != FTS_NAMEONLY) {
		errno = EINVAL;
		return (NULL);
	}

	/* Set current node pointer. */
	p = sp->fts_cur;

	/*
	 * Errno set to 0 so user can distinguish empty directory from
	 * an error.
	 */
	errno = 0;

	/* Fatal errors stop here. */
	if (ISSET(FTS_STOP))
		return (NULL);

	/* Return logical hierarchy of user's arguments. */
	if (p->fts_info == FTS_INIT)
		return (p->fts_link);

	/*
	 * If not a directory being visited in pre-order, stop here.  Could
	 * allow FTS_DNR, assuming the user has fixed the problem, but the
	 * same effect is available with FTS_AGAIN.
	 */
	if (p->fts_info != FTS_D /* && p->fts_info != FTS_DNR */)
		return (NULL);

	/* Free up any previous child list. */
	if (sp->fts_child != NULL)
		fts_lfree(sp->fts_child);

	if (instr == FTS_NAMEONLY) {
		SET(FTS_NAMEONLY);
		instr = BNAMES;
	} else
		instr = BCHILD;

	/*
	 * If using chdir on a relative path and called BEFORE fts_read does
	 * its chdir to the root of a traversal, we can lose -- we need to
	 * chdir into the subdirectory, and we don't know where the current
	 * directory is, so we can't get back so that the upcoming chdir by
	 * fts_read will work.
	 */
	if (p->fts_level != FTS_ROOTLEVEL || p->fts_accpath[0] == '/' ||
	    ISSET(FTS_NOCHDIR))
		return (sp->fts_child = fts_build(sp, instr));

	if ((fd = open(".", O_RDONLY, 0)) < 0)
		return (NULL);
	sp->fts_child = fts_build(sp, instr);
	if (fchdir(fd))
		return (NULL);
	(void)close(fd);
	return (sp->fts_child);
}

/*
 * This is the tricky part -- do not casually change *anything* in here.  The
 * idea is to build the linked list of entries that are used by fts_children
 * and fts_read.  There are lots of special cases.
 *
 * The real slowdown in walking the tree is the stat calls.  If FTS_NOSTAT is
 * set and it's a physical walk (so that symbolic links can't be directories),
 * we can do things quickly.  First, if it's a 4.4BSD file system, the type
 * of the file is in the directory entry.  Otherwise, we assume that the number
 * of subdirectories in a node is equal to the number of links to the parent.
 * The former skips all stat calls.  The latter skips stat calls in any leaf
 * directories and for any files after the subdirectories in the directory have
 * been found, cutting the stat calls by about 2/3.
 */
static FTSENT *

fts_build(sp, type)
	register FTS *sp;
	int type;
{
	register struct dirent *dp;
	register FTSENT *p, *head;
	register int nitems;
	FTSENT *cur, *tail;
	DIR *dirp;
	void *oldaddr;
	int cderrno, descend, len, level, maxlen, nlinks, saved_errno,
	    nostat, doadjust;
	char *cp;

	/* Set current node pointer. */
	cur = sp->fts_cur;

	/*
	 * Open the directory for reading.  If this fails, we're done.
	 * If being called from fts_read, set the fts_info field.
	 */
#if defined FTS_WHITEOUT && 0
	if (ISSET(FTS_WHITEOUT))
		oflag = DTF_NODUP|DTF_REWIND;
	else
		oflag = DTF_HIDEW|DTF_NODUP|DTF_REWIND;
#else
# define opendir2(path, flag) opendir(path)
#endif
       if ((dirp = opendir2(cur->fts_accpath, oflag)) == NULL) {
		if (type == BREAD) {
			cur->fts_info = FTS_DNR;
			cur->fts_errno = errno;
		}
		return (NULL);
	}

	/*
	 * Nlinks is the number of possible entries of type directory in the
	 * directory if we're cheating on stat calls, 0 if we're not doing
	 * any stat calls at all, -1 if we're doing stats on everything.
	 */
	if (type == BNAMES) {
		nlinks = 0;
		/* Be quiet about nostat, GCC. */
		nostat = 0;
	} else if (ISSET(FTS_NOSTAT) && ISSET(FTS_PHYSICAL)) {
		nlinks = cur->fts_nlink - (ISSET(FTS_SEEDOT) ? 0 : 2);
		nostat = 1;
	} else {
		nlinks = -1;
		nostat = 0;
	}

#ifdef notdef
	(void)printf("nlinks == %d (cur: %d)\n", nlinks, cur->fts_nlink);
	(void)printf("NOSTAT %d PHYSICAL %d SEEDOT %d\n",
	    ISSET(FTS_NOSTAT), ISSET(FTS_PHYSICAL), ISSET(FTS_SEEDOT));
#endif
	/*
	 * If we're going to need to stat anything or we want to descend
	 * and stay in the directory, chdir.  If this fails we keep going,
	 * but set a flag so we don't chdir after the post-order visit.
	 * We won't be able to stat anything, but we can still return the
	 * names themselves.  Note, that since fts_read won't be able to
	 * chdir into the directory, it will have to return different path
	 * names than before, i.e. "a/b" instead of "b".  Since the node
	 * has already been visited in pre-order, have to wait until the
	 * post-order visit to return the error.  There is a special case
	 * here, if there was nothing to stat then it's not an error to
	 * not be able to stat.  This is all fairly nasty.  If a program
	 * needed sorted entries or stat information, they had better be
	 * checking FTS_NS on the returned nodes.
	 */
	cderrno = 0;
	if (nlinks || type == BREAD) {
		if (fts_safe_changedir(sp, cur, dirfd(dirp), NULL)) {
			if (nlinks && type == BREAD)
				cur->fts_errno = errno;
			cur->fts_flags |= FTS_DONTCHDIR;
			descend = 0;
			cderrno = errno;
			(void)closedir(dirp);
			dirp = NULL;
		} else
			descend = 1;
	} else
		descend = 0;

	/*
	 * Figure out the max file name length that can be stored in the
	 * current path -- the inner loop allocates more path as necessary.
	 * We really wouldn't have to do the maxlen calculations here, we
	 * could do them in fts_read before returning the path, but it's a
	 * lot easier here since the length is part of the dirent structure.
	 *
	 * If not changing directories set a pointer so that can just append
	 * each new name into the path.
	 */
	len = NAPPEND(cur);
	if (ISSET(FTS_NOCHDIR)) {
		cp = sp->fts_path + len;
		*cp++ = '/';
	} else {
		/* GCC, you're too verbose. */
		cp = NULL;
	}
	len++;
	maxlen = sp->fts_pathlen - len;

	level = cur->fts_level + 1;

	/* Read the directory, attaching each entry to the `link' pointer. */
	doadjust = 0;
	for (head = tail = NULL, nitems = 0; dirp && (dp = readdir(dirp));) {
		if (!ISSET(FTS_SEEDOT) && ISDOT(dp->d_name))
			continue;

		if ((p = fts_alloc(sp, dp->d_name, (int)d_namelen (dp))) == NULL)
			goto mem1;
		if (d_namelen (dp) >= maxlen) {/* include space for NUL */
			oldaddr = sp->fts_path;
			if (fts_palloc(sp, d_namelen (dp) + len + 1)) {
				/*
				 * No more memory for path or structures.  Save
				 * errno, free up the current structure and the
				 * structures already allocated.
				 */
mem1:				saved_errno = errno;
				if (p)
					free(p);
				fts_lfree(head);
				(void)closedir(dirp);
				cur->fts_info = FTS_ERR;
				SET(FTS_STOP);
				errno = saved_errno;
				return (NULL);
			}
			/* Did realloc() change the pointer? */
			if (oldaddr != sp->fts_path) {
				doadjust = 1;
				if (ISSET(FTS_NOCHDIR))
					cp = sp->fts_path + len;
			}
			maxlen = sp->fts_pathlen - len;
		}

		if (len + d_namelen (dp) >= USHRT_MAX) {
			/*
			 * In an FTSENT, fts_pathlen is a u_short so it is
			 * possible to wraparound here.  If we do, free up
			 * the current structure and the structures already
			 * allocated, then error out with ENAMETOOLONG.
			 */
			free(p);
			fts_lfree(head);
			(void)closedir(dirp);
			cur->fts_info = FTS_ERR;
			SET(FTS_STOP);
			errno = ENAMETOOLONG;
			return (NULL);
		}
		p->fts_level = level;
		p->fts_parent = sp->fts_cur;
		p->fts_pathlen = len + d_namelen (dp);

#if defined FTS_WHITEOUT && 0
		if (dp->d_type == DT_WHT)
			p->fts_flags |= FTS_ISW;
#endif

		if (cderrno) {
			if (nlinks) {
				p->fts_info = FTS_NS;
				p->fts_errno = cderrno;
			} else
				p->fts_info = FTS_NSOK;
			p->fts_accpath = cur->fts_accpath;
		} else if (nlinks == 0
#if defined DT_DIR && defined _DIRENT_HAVE_D_TYPE
			   || (nostat &&
			       dp->d_type != DT_DIR && dp->d_type != DT_UNKNOWN)
#endif
		    ) {
			p->fts_accpath =
			    ISSET(FTS_NOCHDIR) ? p->fts_path : p->fts_name;
			p->fts_info = FTS_NSOK;
		} else {
			/* Build a file name for fts_stat to stat. */
			if (ISSET(FTS_NOCHDIR)) {
				p->fts_accpath = p->fts_path;
				memmove(cp, p->fts_name, p->fts_namelen + 1);
			} else
				p->fts_accpath = p->fts_name;
			/* Stat it. */
			p->fts_info = fts_stat(sp, p, 0);

			/* Decrement link count if applicable. */
			if (nlinks > 0 && (p->fts_info == FTS_D ||
			    p->fts_info == FTS_DC || p->fts_info == FTS_DOT))
				--nlinks;
		}

		/* We walk in directory order so "ls -f" doesn't get upset. */
		p->fts_link = NULL;
		if (head == NULL)
			head = tail = p;
		else {
			tail->fts_link = p;
			tail = p;
		}
		++nitems;
	}
	if (dirp)
		(void)closedir(dirp);

	/*
	 * If realloc() changed the address of the path, adjust the
	 * addresses for the rest of the tree and the dir list.
	 */
	if (doadjust)
		fts_padjust(sp, head);

	/*
	 * If not changing directories, reset the path back to original
	 * state.
	 */
	if (ISSET(FTS_NOCHDIR)) {
		if (len == sp->fts_pathlen || nitems == 0)
			--cp;
		*cp = '\0';
	}

	/*
	 * If descended after called from fts_children or after called from
	 * fts_read and nothing found, get back.  At the root level we use
	 * the saved fd; if one of fts_open()'s arguments is a relative path
	 * to an empty directory, we wind up here with no other way back.  If
	 * can't get back, we're done.
	 */
	if (descend && (type == BCHILD || !nitems) &&
	    (cur->fts_level == FTS_ROOTLEVEL ?
	     FCHDIR(sp, sp->fts_rfd) :
	     fts_safe_changedir(sp, cur->fts_parent, -1, ".."))) {
		cur->fts_info = FTS_ERR;
		SET(FTS_STOP);
		return (NULL);
	}

	/* If didn't find anything, return NULL. */
	if (!nitems) {
		if (type == BREAD)
			cur->fts_info = FTS_DP;
		return (NULL);
	}

	/* Sort the entries. */
	if (sp->fts_compar && nitems > 1)
		head = fts_sort(sp, head, nitems);
	return (head);
}

static u_short
fts_stat(sp, p, follow)
	FTS *sp;
	register FTSENT *p;
	int follow;
{
	register FTSENT *t;
	register dev_t dev;
#ifdef __MINGW32__
	register long long ino;
#else
	register ino_t ino;
#endif
	struct xar_stat *sbp, sb;
	int saved_errno;

	/* If user needs stat info, stat buffer already allocated. */
	sbp = ISSET(FTS_NOSTAT) ? &sb : p->fts_statp;

#if defined FTS_WHITEOUT && 0
	/* check for whiteout */
	if (p->fts_flags & FTS_ISW) {
		if (sbp != &sb) {
			memset(sbp, '\0', sizeof (*sbp));
			sbp->st_mode = S_IFWHT;
		}
		return (FTS_W);
       }
#endif

	/*
	 * If doing a logical walk, or application requested FTS_FOLLOW, do
	 * a stat(2).  If that fails, check for a non-existent symlink.  If
	 * fail, set the errno from the stat call.
	 */
	if (ISSET(FTS_LOGICAL) || follow) {
		if (stat(p->fts_accpath, sbp)) {
			saved_errno = errno;
#ifdef S_IFLNK
			if (!lstat(p->fts_accpath, sbp)) {
				errno = 0;
				return (FTS_SLNONE);
			}
#endif
			p->fts_errno = saved_errno;
			goto err;
		}
	} else if (lstat(p->fts_accpath, sbp)) {
		p->fts_errno = errno;
err:		memset(sbp, 0, sizeof(struct xar_stat));
		return (FTS_NS);
	}

	if (S_ISDIR(sbp->st_mode)) {
		/*
		 * Set the device/inode.  Used to find cycles and check for
		 * crossing mount points.  Also remember the link count, used
		 * in fts_build to limit the number of stat calls.  It is
		 * understood that these fields are only referenced if fts_info
		 * is set to FTS_D.
		 */
		dev = p->fts_dev = sbp->st_dev;
		ino = p->fts_ino = sbp->st_ino;
		p->fts_nlink = sbp->st_nlink;

		if (ISDOT(p->fts_name))
			return (FTS_DOT);

		/*
		 * Cycle detection is done by brute force when the directory
		 * is first encountered.  If the tree gets deep enough or the
		 * number of symbolic links to directories is high enough,
		 * something faster might be worthwhile.
		 */
		for (t = p->fts_parent;
		    t->fts_level >= FTS_ROOTLEVEL; t = t->fts_parent)
			if (ino == t->fts_ino && dev == t->fts_dev) {
				p->fts_cycle = t;
				return (FTS_DC);
			}
		return (FTS_D);
	}
	if (S_ISLNK(sbp->st_mode))
		return (FTS_SL);
	if (S_ISREG(sbp->st_mode))
		return (FTS_F);
	return (FTS_DEFAULT);
}

static FTSENT *
fts_sort(sp, head, nitems)
	FTS *sp;
	FTSENT *head;
	register int nitems;
{
	register FTSENT **ap, *p;

	/*
	 * Construct an array of pointers to the structures and call qsort(3).
	 * Reassemble the array in the order returned by qsort.  If unable to
	 * sort for memory reasons, return the directory entries in their
	 * current order.  Allocate enough space for the current needs plus
	 * 40 so don't realloc one entry at a time.
	 */
	if (nitems > sp->fts_nitems) {
		struct _ftsent **a;

		sp->fts_nitems = nitems + 40;
		if ((a = realloc(sp->fts_array,
 		    (size_t)(sp->fts_nitems * sizeof(FTSENT *)))) == NULL) {
			free(sp->fts_array);
			sp->fts_array = NULL;
			sp->fts_nitems = 0;
			return (head);
		}
		sp->fts_array = a;
	}
	for (ap = sp->fts_array, p = head; p; p = p->fts_link)
		*ap++ = p;
	qsort((void *)sp->fts_array, nitems, sizeof(FTSENT *), sp->fts_compar);
	for (head = *(ap = sp->fts_array); --nitems; ++ap)
		ap[0]->fts_link = ap[1];
	ap[0]->fts_link = NULL;
	return (head);
}

static FTSENT *
fts_alloc(sp, name, namelen)
	FTS *sp;
	const char *name;
	register int namelen;
{
	register FTSENT *p;
	size_t len;

	/*
	 * The file name is a variable length array and no stat structure is
	 * necessary if the user has set the nostat bit.  Allocate the FTSENT
	 * structure, the file name and the stat structure in one chunk, but
	 * be careful that the stat structure is reasonably aligned.  Since the
	 * fts_name field is declared to be of size 1, the fts_name pointer is
	 * namelen + 2 before the first possible address of the stat structure.
	 */
	len = sizeof(FTSENT) + namelen;
	if (!ISSET(FTS_NOSTAT))
		len += sizeof(struct xar_stat) + ALIGNBYTES;
	if ((p = malloc(len)) == NULL)
		return (NULL);

	/* Copy the name and guarantee NUL termination. */
	memmove(p->fts_name, name, namelen);
	p->fts_name[namelen] = '\0';

	if (!ISSET(FTS_NOSTAT))
		p->fts_statp = (struct xar_stat *)ALIGN(p->fts_name + namelen + 2);
	p->fts_namelen = namelen;
	p->fts_path = sp->fts_path;
	p->fts_errno = 0;
	p->fts_flags = 0;
	p->fts_instr = FTS_NOINSTR;
	p->fts_number = 0;
	p->fts_pointer = NULL;
	return (p);
}

static void
fts_lfree(head)
	register FTSENT *head;
{
	register FTSENT *p;

	/* Free a linked list of structures. */
	while ((p = head)) {
		head = head->fts_link;
		free(p);
	}
}

/*
 * Allow essentially unlimited paths; find, rm, ls should all work on any tree.
 * Most systems will allow creation of paths much longer than MAXPATHLEN, even
 * though the kernel won't resolve them.  Add the size (not just what's needed)
 * plus 256 bytes so don't realloc the path 2 bytes at a time.
 */
static int
fts_palloc(sp, more)
	FTS *sp;
	size_t more;
{
	char *p;

	sp->fts_pathlen += more + 256;
	/*
	 * Check for possible wraparound.  In an FTS, fts_pathlen is
	 * a signed int but in an FTSENT it is an unsigned short.
	 * We limit fts_pathlen to USHRT_MAX to be safe in both cases.
	 */
	if (sp->fts_pathlen < 0 || sp->fts_pathlen >= USHRT_MAX) {
		if (sp->fts_path) {
			free(sp->fts_path);
			sp->fts_path = NULL;
		}
		sp->fts_path = NULL;
		errno = ENAMETOOLONG;
		return (1);
	}
	p = realloc(sp->fts_path, sp->fts_pathlen);
	if (p == NULL) {
		free(sp->fts_path);
		sp->fts_path = NULL;
		return 1;
	}
	sp->fts_path = p;
	return 0;
}

/*
 * When the path is realloc'd, have to fix all of the pointers in structures
 * already returned.
 */
static void
fts_padjust(sp, head)
	FTS *sp;
	FTSENT *head;
{
	FTSENT *p;
	char *addr = sp->fts_path;

#define	ADJUST(p) do {							\
	if ((p)->fts_accpath != (p)->fts_name) {			\
		(p)->fts_accpath =					\
		    (char *)addr + ((p)->fts_accpath - (p)->fts_path);	\
	}								\
	(p)->fts_path = addr;						\
} while (0)
	/* Adjust the current set of children. */
	for (p = sp->fts_child; p; p = p->fts_link)
		ADJUST(p);

	/* Adjust the rest of the tree, including the current level. */
	for (p = head; p->fts_level >= FTS_ROOTLEVEL;) {
		ADJUST(p);
		p = p->fts_link ? p->fts_link : p->fts_parent;
	}
}

static size_t
fts_maxarglen(argv)
	char * const *argv;
{
	size_t len, max;

	for (max = 0; *argv; ++argv)
		if ((len = strlen(*argv)) > max)
			max = len;
	return (max + 1);
}

/*
 * Change to dir specified by fd or p->fts_accpath without getting
 * tricked by someone changing the world out from underneath us.
 * Assumes p->fts_dev and p->fts_ino are filled in.
 */
static int
fts_safe_changedir(sp, p, fd, path)
	FTS *sp;
	FTSENT *p;
	int fd;
	const char *path;
{
	int ret, oerrno, newfd;
	struct stat sb;

	newfd = fd;
	if (ISSET(FTS_NOCHDIR))
		return (0);
	if (fd < 0 && (newfd = open(path, O_RDONLY, 0)) < 0)
		return (-1);
	if (fstat(newfd, &sb)) {
		ret = -1;
		goto bail;
	}
	if (p->fts_dev != sb.st_dev || p->fts_ino != sb.st_ino) {
		errno = ENOENT;		/* disinformation */
		ret = -1;
		goto bail;
	}
	ret = fchdir(newfd);
bail:
	oerrno = errno;
	if (fd < 0)
		(void)close(newfd);
	errno = oerrno;
	return (ret);
}

/* chown for w32, assumes NTFS security features and Windows XP or later.  */

/* Enabling privileges and restoring previous privileges.  */
static BOOL
set_privilege (HANDLE h_token, LPCTSTR priv_name, BOOL enable_p,
	       TOKEN_PRIVILEGES *old_priv)
{
  TOKEN_PRIVILEGES priv;
  DWORD priv_size = sizeof (priv);
  DWORD opriv_size = sizeof (*old_priv);
  BOOL ret_val = FALSE;

  priv.PrivilegeCount = 1;
  priv.Privileges[0].Attributes = enable_p ? SE_PRIVILEGE_ENABLED : 0;
  LookupPrivilegeValue (NULL, priv_name, &priv.Privileges[0].Luid);
  if (AdjustTokenPrivileges (h_token, FALSE, &priv, priv_size,
			     old_priv, &opriv_size)
      && GetLastError () != ERROR_NOT_ALL_ASSIGNED)
    ret_val = TRUE;

  return ret_val;
}

static BOOL
restore_privileges (HANDLE h_token, TOKEN_PRIVILEGES *privs)
{
  DWORD priv_size = sizeof (*privs);
  BOOL ret_val = FALSE;

  if (AdjustTokenPrivileges (h_token, FALSE, privs, priv_size, NULL, NULL)
      && GetLastError () != ERROR_NOT_ALL_ASSIGNED)
    ret_val = TRUE;

  return ret_val;
}

/* Given the SIDs for the new owner and group, change the file's owner
   and primary group.  */
static int
change_owner_and_group (const char *file, PSID owner_sid, PSID group_sid)
{
  HANDLE h_token = NULL, h_thread = GetCurrentThread ();
  TOKEN_PRIVILEGES old1, old2;
  DWORD err;
  BOOL res;
  int st = 0, retval = -1;
  SECURITY_INFORMATION flags = 0;

  if (owner_sid)
    flags |= OWNER_SECURITY_INFORMATION;
  if (group_sid)
    flags |= GROUP_SECURITY_INFORMATION;
  if (!flags)
    return 0;

  res = OpenThreadToken (h_thread,
			 TOKEN_QUERY | TOKEN_ADJUST_PRIVILEGES,
			 FALSE, &h_token);
  if (!res && GetLastError () == ERROR_NO_TOKEN)
    {
      if (ImpersonateSelf (SecurityImpersonation))
	  res = OpenThreadToken (h_thread,
				 TOKEN_QUERY | TOKEN_ADJUST_PRIVILEGES,
				 FALSE, &h_token);
    }
  if (res)
    {
      st = set_privilege (h_token, SE_TAKE_OWNERSHIP_NAME, TRUE, &old1);
      st += set_privilege (h_token, SE_RESTORE_NAME, TRUE, &old2);
    }

  err = SetNamedSecurityInfo ((char *)file, SE_FILE_OBJECT, flags,
				  owner_sid, group_sid, NULL, NULL);
  if (st >= 2)
    restore_privileges (h_token, &old2);
  if (st >= 1)
    restore_privileges (h_token, &old1);

  if (err == ERROR_SUCCESS)
    retval = 0;

  if (h_token)
    CloseHandle (h_token);

  return retval;
}

static PSID
sid_of_local_domain (SID_NAME_USE *dom_type)
{
  static PSID  psid = NULL;
  static DWORD dom_size = SECURITY_MAX_SID_SIZE;
  static SID_NAME_USE dtype = SidTypeInvalid;
  char         uname[UNLEN+1], domain[1025], domain2[1025];
  DWORD        ulength = sizeof (uname), dlength = sizeof (domain), needed;
  DWORD        dlength2 = sizeof (domain2);
  HANDLE       token = NULL;
  SID_NAME_USE user_type;
  unsigned char *buf = NULL;
  DWORD        blen = 0;
  TOKEN_USER   user_token;
  BOOL         result;

  if (!psid)
    {
      result = OpenProcessToken (GetCurrentProcess (), TOKEN_QUERY, &token);
      if (result)
	{
	  result = GetTokenInformation (token, TokenUser, NULL, 0, &blen);
	  if (!result && GetLastError () == ERROR_INSUFFICIENT_BUFFER)
	    {
	      buf = malloc (blen);
	      if (!buf)
		return NULL;
	      result = GetTokenInformation (token, TokenUser,
					    (LPVOID)buf, blen, &needed);
	      if (result)
		{
		  memcpy (&user_token, buf, sizeof (user_token));
		  result = LookupAccountSid (NULL, user_token.User.Sid,
					     uname, &ulength,
					     domain, &dlength, &user_type);
		  if (result)
		    {
		      psid = LocalAlloc (LMEM_FIXED, dom_size);
		      result = LookupAccountName (NULL, domain, psid, &dom_size,
						  domain2, &dlength2, &dtype);
			/* Sigh.  Windows lies to us regarding the domain
			   type value it returns from LookupAccountName
			   when we run on a standalone workstation: we get
			   SidTypeDomain, even though there's no domain
			   controller anywhere in sight.  So we cheat in
			   response: we compare the domain SID to the SID
			   of the local computer, and if they are identical,
			   we override the domain type with SidTypeComputer,
			   to signal to the caller that it should apply the
			   rules of a standalone workstation.  */
			{
			  TCHAR name_buf[32767];
			  DWORD name_buf_len = sizeof(name_buf);

			  if (GetComputerName (name_buf, &name_buf_len))
			    {
			      PSID psid2;

			      dlength2 = sizeof(domain2);
			      dom_size = SECURITY_MAX_SID_SIZE;
			      psid2 = LocalAlloc (LMEM_FIXED, dom_size);
			      if (LookupAccountName (NULL, name_buf,
						     psid2, &dom_size,
						     domain2, &dlength2, &dtype)
				  && EqualSid (psid, psid2))
				dtype = SidTypeComputer;
			      if (psid2)
				LocalFree (psid2);
			    }
			}
		    }
		}
	    }
	  else
	    result = FALSE;
	}

      if (!result)
	{
	  if (psid)
	    LocalFree (psid);
	  psid = NULL;
	}
    }

  *dom_type = dtype;
  return psid;
}

static unsigned
get_file_owner_or_group (const char *file, int what)
{
  PSECURITY_DESCRIPTOR psd = NULL;
  DWORD sd_len, err;
  SECURITY_INFORMATION sinfo =
    (what == 1) ? OWNER_SECURITY_INFORMATION : GROUP_SECURITY_INFORMATION;
  PSID sid;
  BOOL dflt;
  DWORD n_subauthorities, result = 0;

  if (!GetFileSecurity (file, sinfo, psd, 0, &sd_len))
    {
      err = GetLastError ();
      if (err != ERROR_INSUFFICIENT_BUFFER)
	return 0;
    }

  psd = malloc (sd_len);
  if (!psd)
    return 0;
  if (!GetFileSecurity (file, sinfo, psd, sd_len, &sd_len))
    {
      free (psd);
      return 0;
    }

  if (((what == 1)
       ? GetSecurityDescriptorOwner (psd, &sid, &dflt)
       : GetSecurityDescriptorGroup (psd, &sid, &dflt))
      && IsValidSid (sid))
    {
      free (psd);
      n_subauthorities = *GetSidSubAuthorityCount (sid);
      if (n_subauthorities > SID_MAX_SUB_AUTHORITIES)
	result = 0;
      result = *GetSidSubAuthority (sid, n_subauthorities - 1);
    }
  else
    err = GetLastError ();

  return result;
}

/* Return the RID of the FILE's owner.  */
static unsigned
get_file_owner (const char *file)
{
  return get_file_owner_or_group (file, 1);
}

/* Return the RID of the FILE's primary group.  */
static unsigned
get_file_primary_group (const char *file)
{
  return get_file_owner_or_group (file, 2);
}

struct rid_to_sid {
  unsigned rid;
  const char *sid_str;
};

/* Known SID values whose first n-1 components are not the local
   domain SID.  Any RID not in the list will have its n-1
   sub-authorities taken from the local domain SID.  */
static struct rid_to_sid rid_to_user_sid_db[] = {
  { 99999, NULL }
};

/* This was compiled from KB243330.  */
static struct rid_to_sid rid_to_group_sid_db[] = {
  {     9, "S-1-5-9"     },	/* Enterprise Domain Controllers */
  {    14, "S-1-5-14"    },	/* Remote Interactive Login */
  {    18, "S-1-5-18"    },	/* Local System */
  {   544, "S-1-5-32-544"},	/* Local Administrators */
  {   545, "S-1-5-32-545"},	/* Local Users */
  {   546, "S-1-5-32-546"},	/* Guests */
  {   547, "S-1-5-32-547"},	/* Power Users */
  {   548, "S-1-5-32-548"},	/* Account Operators */
  {   549, "S-1-5-32-549"},	/* Server Operators */
  {   550, "S-1-5-32-550"},	/* Print Operators */
  {   551, "S-1-5-32-551"},	/* Backup Operators */
  {   573, "S-1-5-32-573"},	/* Event Log Readers */
  { 99999, NULL          }
};

/* Guess the n-1 subauthorities of a SID to which RID belongs.  WHAT
   is 1 if RID is for a user, 2 if it is for a group.  If any
   non-recoverable error occurs, value is NULL.  */
static PSID
intuit_sid_from_rid (unsigned rid, int what)
{
  SID_IDENTIFIER_AUTHORITY auth_nt = {SECURITY_NT_AUTHORITY};
  PSID pdom_sid, sid;
  struct rid_to_sid *pdb;
  DWORD n_sub;
  DWORD sub_auth[SID_MAX_SUB_AUTHORITIES];
  DWORD i;
  SID_NAME_USE type;

  if (what != 1 && what != 2)
    {
      errno = EINVAL;
      return NULL;
    }

  pdb = (what == 1) ? rid_to_user_sid_db : rid_to_group_sid_db;
  for ( ; pdb->sid_str; pdb++)
    if (pdb->rid == rid)
      {
	if (ConvertStringSidToSid ((char *)pdb->sid_str, &sid))
	  {
	    /* Sigh... The SID returned by ConvertStringSidToSid needs
	       to be freed by a call to LocalFree, whereas the SID
	       returned by AllocateAndInitializeSid should be freed by
	       FreeSid.  Since we don't want out caller to know how we
	       created the SID, we call AllocateAndInitializeSid for
	       the SID converted from string, and free the latter.  */
	    n_sub = *GetSidSubAuthorityCount (sid);
	    memset (sub_auth, 0, sizeof (sub_auth));
	    for (i = 0; i < SID_MAX_SUB_AUTHORITIES && i < n_sub; i++)
	      sub_auth[i] = *GetSidSubAuthority (sid, i);

	    LocalFree (sid);
	    if (AllocateAndInitializeSid (&auth_nt, n_sub,
					  sub_auth[0], sub_auth[1],
					  sub_auth[2], sub_auth[3],
					  sub_auth[4], sub_auth[5],
					  sub_auth[6], sub_auth[7],
					  &sid))
	      {
		return sid;
	      }
	  }
	break;
      }

  pdom_sid = sid_of_local_domain (&type);
  if (!pdom_sid)
    return NULL;
  n_sub = *GetSidSubAuthorityCount (pdom_sid);
  memset (sub_auth, 0, sizeof (sub_auth));
  for (i = 0; i < SID_MAX_SUB_AUTHORITIES && i < n_sub; i++)
    sub_auth[i] = *GetSidSubAuthority (pdom_sid, i);
  if (i >= 8)
    i = 7;
  sub_auth[i] = rid;
  if (AllocateAndInitializeSid (&auth_nt, i + 1,
				sub_auth[0], sub_auth[1],
				sub_auth[2], sub_auth[3],
				sub_auth[4], sub_auth[5],
				sub_auth[6], sub_auth[7],
				&sid))
    {
      return sid;
    }

  return NULL;
}

int
chown (const char *fn, unsigned uid, unsigned gid)
{
  PSID owner_sid = NULL, group_sid = NULL;

  /* Get the file's current owner and primary group. */
  DWORD current_owner = get_file_owner (fn);
  DWORD current_group = get_file_primary_group (fn);
  int retval;
  SID_NAME_USE domain_type;

  sid_of_local_domain (&domain_type);
  /* Map UID of zero to the local Administrator and GID of zero to
     built-in Administrators.  On NTFS, a RID of zero means Everyone
     (SID=S-1-1-0), and it doesn't make sense to set the owner or the
     group to that.  It is much more plausible that the zero came from
     a Posix system, where it means a superuser (root).  */
  if (uid == 0)
    {
      if (domain_type == SidTypeDomain)
	uid = DOMAIN_GROUP_RID_ADMINS;
      else
	uid = DOMAIN_USER_RID_ADMIN;
    }
  if (gid == 0)
    gid = DOMAIN_ALIAS_RID_ADMINS;

  /* Do we have anything non-trivial to do?  */
  if (current_owner == uid && current_group == gid)
    return 0;

  /* Intuit the SID of the new owner and the new primary group from
     their RIDs.  */
  if (current_owner != uid)
    {
      owner_sid = intuit_sid_from_rid (uid, 1);
    }
  if (current_group != gid)
    {
      group_sid = intuit_sid_from_rid (gid, 2);
    }

  /* Finally, make the change.  */
  if ((owner_sid && !IsValidSid (owner_sid))
      || (group_sid && !IsValidSid (group_sid)))
    {
      errno = EINVAL;
      return -1;
    }

  retval = change_owner_and_group (fn, owner_sid, group_sid);
  if (owner_sid)
    FreeSid (owner_sid);
  if (group_sid)
    FreeSid (group_sid);

  return retval;
}

/* Emulation of Posix ACL functions for MS-Windows.  Only a limited
   subset of functions is supported: get/set file's ACL, get an ACL
   entry, and convert an ACL to or from textual representation.  The
   textual representation is the SDDL string, not the Posix multi-line
   format.  */

#define SDDL_REVISION_1	1
WINADVAPI BOOL WINAPI
ConvertStringSecurityDescriptorToSecurityDescriptorA(
    LPCSTR StringSecurityDescriptor,
    DWORD StringSDRevision,
    PSECURITY_DESCRIPTOR  *SecurityDescriptor,
    PULONG  SecurityDescriptorSize
    );
WINADVAPI BOOL WINAPI
ConvertStringSecurityDescriptorToSecurityDescriptorW(
    LPCWSTR StringSecurityDescriptor,
    DWORD StringSDRevision,
    PSECURITY_DESCRIPTOR  *SecurityDescriptor,
    PULONG  SecurityDescriptorSize
    );
WINADVAPI BOOL WINAPI
ConvertSecurityDescriptorToStringSecurityDescriptorA(
    PSECURITY_DESCRIPTOR  SecurityDescriptor,
    DWORD RequestedStringSDRevision,
    SECURITY_INFORMATION SecurityInformation,
    LPSTR  *StringSecurityDescriptor,
    PULONG StringSecurityDescriptorLen
    );
WINADVAPI BOOL WINAPI
ConvertSecurityDescriptorToStringSecurityDescriptorW(
    PSECURITY_DESCRIPTOR  SecurityDescriptor,
    DWORD RequestedStringSDRevision,
    SECURITY_INFORMATION SecurityInformation,
    LPWSTR  *StringSecurityDescriptor,
    PULONG StringSecurityDescriptorLen
    );
#ifdef UNICODE
#define ConvertStringSecurityDescriptorToSecurityDescriptor  ConvertStringSecurityDescriptorToSecurityDescriptorW
#define ConvertSecurityDescriptorToStringSecurityDescriptor  ConvertSecurityDescriptorToStringSecurityDescriptorW
#else
#define ConvertStringSecurityDescriptorToSecurityDescriptor  ConvertStringSecurityDescriptorToSecurityDescriptorA
#define ConvertSecurityDescriptorToStringSecurityDescriptor  ConvertSecurityDescriptorToStringSecurityDescriptorA
#endif

#include <sys/acl.h>

int
acl_valid (acl_t acl)
{
  return IsValidSecurityDescriptor ((PSECURITY_DESCRIPTOR)acl) ? 0 : -1;
}

char *
acl_to_text (acl_t acl, ssize_t *size)
{
  LPTSTR str_acl;
  SECURITY_INFORMATION flags =
    OWNER_SECURITY_INFORMATION |
    GROUP_SECURITY_INFORMATION |
    DACL_SECURITY_INFORMATION;
  char *retval = NULL;
  ssize_t local_size;

  if (ConvertSecurityDescriptorToStringSecurityDescriptor ((PSECURITY_DESCRIPTOR)acl, SDDL_REVISION_1, flags, &str_acl, &local_size))
    {
      retval = strdup (str_acl);
      if (!retval)
	errno = ENOMEM;
      else if (size)
	*size = local_size;
      LocalFree (str_acl);
    }
  else
    errno = EINVAL;

  return retval;
}

acl_t
acl_from_text (const char *acl_str)
{
  PSECURITY_DESCRIPTOR psd, retval = NULL;
  ULONG sd_size;

  if (ConvertStringSecurityDescriptorToSecurityDescriptor (acl_str, SDDL_REVISION_1, &psd, &sd_size))
    {
      retval = malloc (sd_size);
      if (retval)
	memcpy (retval, psd, sd_size);
      else
	errno = ENOMEM;
      LocalFree (psd);
    }
  else
    errno = EINVAL;

  return retval;
}

int
acl_free (void *ptr)
{
  free (ptr);
  return 0;
}

acl_t
acl_get_file (const char *fname, acl_type_t type)
{
  PSECURITY_DESCRIPTOR psd = NULL;

  if (type == ACL_TYPE_ACCESS)
    {
      DWORD sd_len, err;
      SECURITY_INFORMATION si =
	OWNER_SECURITY_INFORMATION |
	GROUP_SECURITY_INFORMATION |
	DACL_SECURITY_INFORMATION ;

      if (!GetFileSecurity (fname, si, psd, 0, &sd_len))
	{
	  err = GetLastError ();
	  if (err == ERROR_INSUFFICIENT_BUFFER)
	    {
	      psd = malloc (sd_len);
	      if (!psd)
		errno = ENOMEM;
	      else if (!GetFileSecurity (fname, si, psd, sd_len, &sd_len))
		{
		  free (psd);
		  errno = EIO;
		  psd = NULL;
		}
	    }
	  else
	    errno = EIO;
	}
    }
  else if (type != ACL_TYPE_DEFAULT)
    errno = EINVAL;

  return psd;
}

int
acl_set_file (const char *fname, acl_type_t type, acl_t acl)
{
  HANDLE h_token = NULL, h_thread = GetCurrentThread ();
  TOKEN_PRIVILEGES old1, old2;
  DWORD err;
  BOOL res;
  int st = 0, retval = -1;
  SECURITY_INFORMATION flags = 0;
  PSID psid;
  PACL pacl;
  BOOL dflt;
  BOOL dacl_present;

  if (acl_valid (acl) != 0
      || (type != ACL_TYPE_DEFAULT && type != ACL_TYPE_ACCESS))
    {
      errno = EINVAL;
      return -1;
    }

  if (type == ACL_TYPE_DEFAULT)
    {
      errno = ENOSYS;
      return -1;
    }

  if (GetSecurityDescriptorOwner ((PSECURITY_DESCRIPTOR)acl, &psid, &dflt)
      && psid)
    flags |= OWNER_SECURITY_INFORMATION;
  if (GetSecurityDescriptorGroup ((PSECURITY_DESCRIPTOR)acl, &psid, &dflt)
      && psid)
    flags |= GROUP_SECURITY_INFORMATION;
  if (GetSecurityDescriptorDacl ((PSECURITY_DESCRIPTOR)acl, &dacl_present,
				 &pacl, &dflt)
      && dacl_present)
    flags |= DACL_SECURITY_INFORMATION;
  if (!flags)
    return 0;

  res = OpenThreadToken (h_thread,
			 TOKEN_QUERY | TOKEN_ADJUST_PRIVILEGES,
			 FALSE, &h_token);
  if (!res && GetLastError () == ERROR_NO_TOKEN)
    {
      if (ImpersonateSelf (SecurityImpersonation))
	  res = OpenThreadToken (h_thread,
				 TOKEN_QUERY | TOKEN_ADJUST_PRIVILEGES,
				 FALSE, &h_token);
    }
  /* According to KB-245153, setting the owner will succeed if either:
     (1) the caller is the user who will be the new owner, and has the
         SE_TAKE_OWNERSHIP privilege, or
     (2) the caller has the SE_RESTORE privilege, in which case she can
         set any valid user or group as the owner

     We request below both SE_TAKE_OWNERSHIP and SE_RESTORE
     privileges, and disregard any failures in obtaining them.  If
     these privileges cannot be obtained, and do not already exist in
     the calling thread's security token, this function could fail
     with EPERM.  */
  if (res)
    {
      st = set_privilege (h_token, SE_TAKE_OWNERSHIP_NAME, TRUE, &old1);
      st += set_privilege (h_token, SE_RESTORE_NAME, TRUE, &old2);
    }

  SetFileSecurity ((char *)fname, flags, (PSECURITY_DESCRIPTOR)acl);
  err = GetLastError ();
  if (st >= 2)
    restore_privileges (h_token, &old2);
  if (st >= 1)
    restore_privileges (h_token, &old1);

  if (err == ERROR_SUCCESS)
    retval = 0;
  else if (err == ERROR_INVALID_OWNER)
    errno = EPERM;

  if (h_token)
    CloseHandle (h_token);

  return retval;
}

int
acl_get_entry (acl_t acl, int entry_id, acl_entry_t *entry)
{
  static int last_entry = -1;
  static DWORD ace_count = 0;
  ACL_SIZE_INFORMATION acl_size_info;
  PACL pacl;
  BOOL dflt;
  BOOL dacl_present;
  LPVOID tem;

  if (acl_valid (acl) != 0
      || (entry_id != ACL_FIRST_ENTRY && entry_id != ACL_NEXT_ENTRY)
      || (last_entry == -1 && entry_id != ACL_FIRST_ENTRY))
    {
      errno = EINVAL;
      return -1;
    }

  if (!GetSecurityDescriptorDacl ((PSECURITY_DESCRIPTOR)acl, &dacl_present,
				  &pacl, &dflt))
    {
      errno = EIO;
      return -1;
    }
  if (!dacl_present)
    return 0;

  if (entry_id == ACL_FIRST_ENTRY)
    {
      memset (&acl_size_info, 0, sizeof (acl_size_info));
      acl_size_info.AclBytesInUse = sizeof(ACL);
      if (!GetAclInformation (pacl, &acl_size_info, sizeof(acl_size_info),
			      AclSizeInformation))
	{
	  errno = EIO;
	  return -1;
	}
      ace_count = acl_size_info.AceCount;
      last_entry = 0;
    }
  else if (last_entry >= ace_count)
    return 0;
  else
    last_entry++;

  if (!GetAce (pacl, last_entry, &tem))
    {
      errno = EIO;
      return -1;
    }
  *entry = tem;
  return 1;
}
