#!/usr/bin/python
'gnews.py - google news clone'

from config import config as config
import renderer
import fetcher
import os.path
import urllib
from logger import log
import configloader

sources = configloader.load()

def main():
    "gnews's main function"
# TODO: argv check

    # fetch RSS feed
    entries = []
    for feed in sources:
        f = fetcher.FeedFetcher(feed)
        e = f.get_entries()
        entries.extend(e)

    # aggregate tags
    tags = {}
    for entry in entries:
        for tag in entry['tags']:
            if tag in tags:
                tags[tag]['entry'].append(entry)
            else:
                tags[tag] = {}
                tags[tag]['entry'] = [entry,]
                tags[tag]['quoted_name'] = urllib.quote(tag.encode('utf-8'))

    # count stories for each tags
    for tag in tags:
        tags[tag]['count'] = len(tags[tag]['entry'])

    # sort tag by count
    sorted_tags = tags.keys()
    sorted_tags.sort(lambda x,y: cmp(tags[y]['count'], tags[x]['count']))

    # sort by date
    cmp_entries = (lambda x,y: 1 if (x["date"] < y["date"]) else -1)
    entries.sort(cmp_entries)

    for e in entries:
        log(e["date"])

    call_plugin('pre_render', entries)

    # do rendering
    params = {
        'tags':tags,
        'page':{},
        'sorted_tags':sorted_tags
        }

    # render index page
    do_rendering('index', 'index%s.html', entries, params)

    # render tag page
    for tag in tags:
        subentries = tags[tag]['entry']
        do_rendering('tags', tag + '%s.html', subentries, params)


def call_plugin(function_name, entries):
    "call plugin"
    for plugin in config['plugins']:
        mod = _get_plugin(plugin)
        f = mod.__getattribute__(function_name)
        f(entries)

class PluginError(Exception):
    def __init__(self, value):
        self.value = value
    def __str__(self):
        return 'plugin "' + self.value + '" is not found.'

def _get_plugin(plugin_name):
    'load plugin by config settings'

    # fallback when filter isn't defined
    if plugin_name is None:
        return lambda x:x
        
    # import module
    mods = __import__(config['plugin_directory'], 
                      globals(),
                      locals(),
                      [plugin_name,])
    try:
        mod = mods.__getattribute__(plugin_name)
    except AttributeError:
        raise PluginError(plugin_name)

    return mod


def do_rendering(page_type, filename, entries, params):
    "rendering page"

    r = renderer.Renderer(sources)
    tmpl = config[page_type]['template']
    output_dir = config[page_type]['output_directory']

    # do pagination
    pageunit = config['pagination_unit']
    total_page = 1 + (len(entries) - 1) / pageunit
    params['page']['total'] = total_page
    params['page']['filename'] = filename

    # rendering each page
    for page in range(1, total_page + 1):
        params['page']['current'] = page
        start = pageunit * (page - 1)
        end = pageunit * page
        if page == 1:
            output_fullpath = os.path.join(output_dir, filename % '')
        else:
            output_fullpath = os.path.join(output_dir, filename % page)

        log('generate ' + output_fullpath + '...')
        f = open(output_fullpath, "w")
        html = r.render(tmpl, entries[start:end], params)
        f.write(html)
        f.close()
    

if __name__ == '__main__':
    main()
