;;; -*- Mode: Lisp; Package: EDITOR -*-
;;;
;;; This file is part of xyzzy.
;;;

(provide "calendar")

(in-package "editor")

(export '(calendar calendar-mode *calendar-mode-hook* *calendar-mode-map*
	  calendar-next-page-or-year calendar-previous-page-or-year
	  calendar-mouse-popup calendar-popup))

(defvar *calendar-gregorian-reform-year* 1582)
(defvar *calendar-gregorian-reform-month* 10)
(defvar *calendar-gregorian-reform-day* 15)

(defvar *calendar-equinox-parameter-list*
  '((1850)
    (1899 1983 19.8277 22.2588)
    (1979 1983 20.8357 23.2588)
    (2099 1980 20.8431 23.2488)
    (2150 1980 21.8510 24.2488)))

(defvar *calendar-last-day-of-month*
  #(0 31 28 31 30 31 30 31 31 30 31 30 31))

(defvar *calendar-japanese-holiday-list*
  '(
    ;1873(6)N1015 z344
    (1874 1912 1 3 "n")
    (1874 1912 1 5 "VN")
    (1874 1912 1 30 "FVc")
    (1874 1912 2 11 "I")
    (1874 1912 4 3 "_Vc")
    (1874 1878 9 17 "_")
    (1873 1911 11 3 "V")
    (1873 1911 11 23 "V")
    ;1878(11)N65 z23
    (1879 1912 3 calendar-vernal-equinox "tGcˍ")
    (1878 1911 9 calendar-autumnal-equinox "HGcˍ")
    ;1879(12)N75 z27
    (1879 1911 10 17 "_")
    ;1912(吳)N93 19 ({s)
    (1913 1927 1 3 "n")
    (1913 1927 1 5 "VN")
    (1913 1927 2 11 "I")
    (1913 1926 3 calendar-vernal-equinox "tGcˍ")
    (1913 1926 4 3 "_Vc")
    (1913 1926 7 30 "Vc")
    (1913 1926 8 31 "V")
    (1912 1926 9 calendar-autumnal-equinox "HGcˍ")
    (1912 1926 10 17 "_")
    (1912 1926 11 23 "V")
    ;1913(吳2)N716 259
    (1913 1926 10 31 "Vߏj")
    ;1915(吳4)N921 161
    (1915 1915 11 10 "ʃmX")
    (1915 1915 11 14 "另")
    (1915 1915 11 16 "Xy另Ռ")
    ;1927(a2)N33 25 ({s)
    (1928 1948 1 3 "n")
    (1928 1948 1 5 "VN")
    (1928 1948 2 11 "I")
    (1927 1948 3 calendar-vernal-equinox "tGcˍ")
    (1927 1948 4 3 "_Vc")
    (1927 1948 4 29 "V")
    (1927 1947 9 calendar-autumnal-equinox "HGcˍ")
    (1927 1947 10 17 "_")
    (1927 1947 11 3 "")
    (1927 1947 11 23 "V")
    (1927 1947 12 25 "吳Vc")
    ;1928(a3)N98 226 
    (1928 1928 11 10 "ʃmX")
    (1928 1928 11 14 "另")
    (1928 1928 11 16 "Xy另Ռ")
    ;1948(a23)N720 @178 ({s)
    (1949 nil 1 1 "")
    (1949 1999 1 15 "l̓")
    (1949 nil 3 calendar-vernal-equinox "t̓")
    (1949 1988 4 29 "Vca")
    (1949 nil 5 3 "@LO")
    (1949 nil 5 5 "ǂ̓")
    (1948 nil 9 calendar-autumnal-equinox "H̓")
    (1948 nil 11 3 "̓")
    (1948 nil 11 23 "ΘJӂ̓")
    ;1959(a34)N317 @16 ({s)
    (1959 1959 4 10 "cqmě̋V")
    ;1966(a41)N625 @86 ({s)
    (1967 nil 2 11 "LO̓")
    (1966 2002 9 15 "hV̓")
    (1966 1999 10 10 "̈̓")
    ;1989()N217 @4 ({s)
    (1989 1989 2 24 "aVc̑r̗")
    ;1989()N217 @5 ({s)
    (1989 nil 4 29 "݂ǂ̓")
    (1989 nil 12 23 "Vca")
    ;1990(2)N61 @24 ({s)
    (1990 1990 11 12 "ʗ琳a̋V")
    ;1993(5)N430 @32 ({s)
    (1993 1993 6 9 "cqmě̋V")
    ;1995(7)N38 @22 (1996(8)N11{s)
    (1996 2002 7 20 "C̓")
    ;1998(10)N1021 @141 (2000(12)N11{s)
    (2000 nil 1 (1 . 1) "l̓")
    (2000 nil 10 (1 . 1) "̈̓")
    ;2001(13)N622 @59 (2003(15)N11{s)
    (2003 nil 7 (2 . 1) "C̓")
    (2003 nil 9 (2 . 1) "hV̓")
    ))

(defun calendar-calc-equinox (year f)
  (let ((x (cdr (assoc year *calendar-equinox-parameter-list* :test #'<=))))
    (and x (truncate (- (+ (funcall f x) (* 0.242194 (- year 1980)))
			(truncate (- year (car x)) 4))))))

(defun calendar-vernal-equinox (year)
  (calendar-calc-equinox year #'cadr))

(defun calendar-autumnal-equinox (year)
  (calendar-calc-equinox year #'caddr))

(defun calendar-gregorian-p (year month day)
  (or (> year *calendar-gregorian-reform-year*)
      (and (= year *calendar-gregorian-reform-year*)
	   (or (> month *calendar-gregorian-reform-month*)
	       (and (= month *calendar-gregorian-reform-month*)
		    (>= day *calendar-gregorian-reform-day*))))))

(defun calendar-gregorian-leap-year-p (year)
  (or (zerop (rem year 400))
      (and (zerop (rem year 4))
	   (not (zerop (rem year 100))))))

(defun calendar-julian-leap-year-p (year)
  (zerop (rem year 4)))

(defun calendar-gregorian-last-day-of-month (year month)
  (if (and (= month 2) (calendar-gregorian-leap-year-p year))
      29
    (svref *calendar-last-day-of-month* month)))

(defun calendar-julian-last-day-of-month (year month)
  (if (and (= month 2) (calendar-julian-leap-year-p year))
      29
    (svref *calendar-last-day-of-month* month)))

(defun calendar-jd-from-ut (year month day &optional (gregorian-p
						      (calendar-gregorian-p
						       year month day)))
  (when (<= month 2)
    (decf year)
    (incf month 12))
  (+ (truncate (* 365.25 (+ year 4716)))
     (truncate (* 30.6001 (+ month 1)))
     day (if gregorian-p
	     (let ((a (truncate year 100)))
	       (- (truncate a 4) a -2))
	   0)
     -1524))

(defun calendar-ut-day (year month day)
  (rem (+ 1 (calendar-jd-from-ut year month day)) 7))

(defun calendar-japanese-holiday (year month v)
  (let ((sun (- 1 (calendar-ut-day year month 1)))
	;; 1985(a60)N1227 @103 ({s) - ̋x
	(nh (and (> year 1985) "̋x"))
	;; 1973(a48)N412 @10 ({s) - U֋x
	(ex (or (> year 1973) (and (= year 1973) (>= month 4)))))
    (dotimes (i (length v))
      (setf (svref v i) nil))
    (when (<= year *calendar-gregorian-reform-year*)
      (return-from calendar-japanese-holiday))
    (dolist (x *calendar-japanese-holiday-list*)
      (when (and (>= year (pop x))
		 (let ((e (pop x)))
		   (or (null e) (<= year e)))
		 (= month (pop x)))
	(let ((day (let ((d (pop x)))
		     (cond ((numberp d) d)
			   ((consp d)
			    (let ((n (+ (cdr d) sun)))
			      (+ (if (plusp n) n (+ n 7))
				 (* 7 (car d)))))
			   (t
			    (funcall d year))))))
	  (when day
	    (setf (svref v day) (car x))))))
    (when nh
      (do ((d 3 (+ d 1)))
	  ((>= d (length v)))
	(when (and (svref v d)
		   (not (svref v (- d 1)))
		   (svref v (- d 2)))
	  (setf (svref v (- d 1)) nh))))
    (do ((d (if (plusp sun) sun (+ sun 7)) (+ d 7)))
	((>= d (- (length v) 1)))
      (cond ((or (null (svref v d))
		 (eq (svref v d) nh))
	     (setf (svref v d) t))
	    ((and ex
		  (not (eq (svref v d) nh))
		  (or (eq (svref v (+ d 1)) nh)
		      (null (svref v (+ d 1)))))
	     (setf (svref v (+ d 1)) "U֋x"))))))

(defun calendar-forward-line (&optional (n 1))
  (dotimes (i n)
    (or (forward-line 1)
	(progn
	  (goto-eol)
	  (insert #\LFD)))))

(defun calendar-goto-column (column)
  (let ((indent-tabs-mode nil))
    (declare (special indent-tabs-mode))
    (goto-eol)
    (indent-to column)))

(defun calendar-title-string (year month)
  (if (plusp year)
      (format nil " ~DN~D" year month)
    (format nil " BC~DN~D" (- 1 year) month)))

(defun calendar-print-month (year month &optional today (column 0) (v (make-vector 33)))
  (let* ((dow (calendar-ut-day year month 1))
	 (generator (cond ((calendar-gregorian-p year month 1)
			   (let ((last (calendar-gregorian-last-day-of-month year month))
				 (day 0))
			     #'(lambda ()
				 (if (>= day last)
				     nil
				   (incf day)))))
			  ((if (= month 12)
			       (calendar-gregorian-p (+ year 1) 1 1)
			     (calendar-gregorian-p year (+ month 1) 1))
			   (let ((jd (calendar-jd-from-ut year month 1))
				 (reform-jd (calendar-jd-from-ut
					     *calendar-gregorian-reform-year*
					     *calendar-gregorian-reform-month*
					     *calendar-gregorian-reform-day*))
				 (last (calendar-gregorian-last-day-of-month year month))
				 (day 0))
			     #'(lambda ()
				 (cond ((< jd reform-jd)
					(incf jd)
					(incf day))
				       ((= jd reform-jd)
					(incf jd)
					(and (< day *calendar-gregorian-reform-day*)
					     (setq day *calendar-gregorian-reform-day*)))
				       ((= day last)
					nil)
				       (t
					(incf day))))))
			  (t
			   (let ((last (calendar-julian-last-day-of-month year month))
				 (day 0))
			     #'(lambda ()
				 (if (= day last)
				     nil
				   (incf day))))))))
    (calendar-japanese-holiday year month v)
    (calendar-goto-column column)
    (insert (calendar-title-string year month))
    (calendar-forward-line 1)
    (calendar-goto-column column)
    (dotimes (i 7)
      (let ((o (+ (point) 1)))
	(insert #\SPC (svref "ΐ؋y" i))
	(case i
	  (0 (set-text-attribute o (point) 'calendar :foreground 1))
	  (6 (set-text-attribute o (point) 'calendar :foreground 4)))))
    (calendar-forward-line 1)
    (calendar-goto-column (+ column (* dow 3)))
    (let (d)
      (while (setq d (funcall generator))
	(when (= dow 7)
	  (setq dow 0)
	  (calendar-forward-line 1)
	  (calendar-goto-column column))
	(let ((o (+ (point) 1)))
	  (insert (format nil " ~2D" d))
	  (cond ((or (svref v d)
		     (zerop dow))
		 (set-text-attribute o (point) (cons 'calendar (svref v d))
				     :foreground 1 :underline (eql d today)))
		((= dow 6)
		 (set-text-attribute o (point) 'calendar
				     :foreground 4 :underline (eql d today)))
		((eql d today)
		 (set-text-attribute o (point) 'calendar :underline t)))
	  (incf dow))))))

(defun calendar-print (year month nmonths y m d)
  (with-output-to-temp-buffer ("*Calendar*")
    (let ((v (make-vector 33))
	  (column 0)
	  (start (point)))
      (dotimes (i nmonths)
	(goto-char start)
	(calendar-print-month year month
			      (and (= year y) (= month m) d)
			      column v)
	(cond ((= column 48)
	       (goto-char start)
	       (calendar-forward-line 9)
	       (goto-bol)
	       (setq start (point))
	       (setq column 0))
	      (t
	       (incf column 24)))
	(cond ((= month 12)
	       (setq month 1)
	       (incf year))
	      (t
	       (incf month))))))
  (goto-char (point-min))
  (setq buffer-read-only t))

(defun calendar-popup ()
  (interactive)
  (setq calendar-range-begin nil)
  (multiple-value-bind (p1 p2 tag)
      (find-text-attribute-point (point))
    (when (and (consp tag)
	       (eq (car tag) 'calendar)
	       (stringp (cdr tag)))
      (setq calendar-range-begin p1)
      (setq calendar-range-end p2)
      (popup-string (cdr tag) p2))))

(defun calendar-mouse-popup ()
  (interactive)
  (save-excursion
    (goto-last-mouse-point)
    (cond ((and calendar-range-begin
		(>= (point) calendar-range-begin)
		(< (point) calendar-range-end))
	   (continue-popup))
	  (t
	   (calendar-popup)))))

(defun calendar-next-page-or-year (&optional (arg 1))
  (interactive)
  (unless (next-page arg)
    (incf calendar-current-year arg)
    (multiple-value-bind (s im h d m y)
	(decode-universal-time (get-universal-time))
      (calendar-print calendar-current-year 1 12 y m d)
      (when (minusp arg)
	(goto-char (point-max)))))
  t)

(defun calendar-previous-page-or-year (&optional (arg 1))
  (interactive)
  (calendar-next-page-or-year (- arg)))

(defun calendar (&optional year)
  (interactive "p")
  (let (month nmonths)
    (multiple-value-bind (s im h d m y)
	(decode-universal-time (get-universal-time))
      (cond (year
	     (setq month 1 nmonths 12))
	    (t
	     (if (<= m 6)
		 (setq month 7 year (- y 1))
	       (setq month 1 year y))
	     (setq nmonths 18)))
      (calendar-print year month nmonths y m d)
      (calendar-mode year)
      (scan-buffer (calendar-title-string y m))
      t)))

(defvar *calendar-mode-hook* nil)
(defvar *calendar-mode-map* nil)

(unless *calendar-mode-map*
  (setq *calendar-mode-map* (make-sparse-keymap))
  (define-key *calendar-mode-map* #\? 'calendar-popup)
  (define-key *calendar-mode-map* #\MouseMove 'calendar-mouse-popup)
  (define-key *calendar-mode-map* #\C-v 'calendar-next-page-or-year)
  (define-key *calendar-mode-map* #\M-v 'calendar-previous-page-or-year)
  (define-key *calendar-mode-map* #\C-z 'calendar-previous-page-or-year)
  (define-key *calendar-mode-map* #\q 'kill-selected-buffer)
  (define-key *calendar-mode-map* #\PageUp 'calendar-previous-page-or-year)
  (define-key *calendar-mode-map* #\PageDown 'calendar-next-page-or-year))

(defun calendar-mode (year)
  (kill-all-local-variables)
  (setq mode-name "Calendar")
  (setq buffer-mode 'calendar-mode)
  (use-keymap *calendar-mode-map*)
  (make-local-variable 'calendar-range-begin)
  (setq calendar-range-begin nil)
  (make-local-variable 'calendar-range-end)
  (setq calendar-range-end nil)
  (make-local-variable 'calendar-current-year)
  (setq calendar-current-year year)
  (run-hooks '*calendar-mode-hook*))
