/*
 * Copyright (C) 2006 uguu@users.sourceforge.jp, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.expression_computer;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;

/**
 * <p>
 * コンパイルされたコマンドのリストです。
 * </p>
 * 
 * @author uguu@users.sourceforge.jp
 */
public final class CommandList implements List {

    private List commandList = new ArrayList();

    /**
     * {@inheritDoc}
     */
    public int size() {
        return this.commandList.size();
    }

    /**
     * {@inheritDoc}
     */
    public boolean isEmpty() {
        return this.commandList.isEmpty();
    }

    /**
     * {@inheritDoc}
     */
    public boolean contains(Object o) {
        return this.commandList.contains(o);
    }

    /**
     * {@inheritDoc}
     */
    public Iterator iterator() {
        return this.commandList.iterator();
    }

    /**
     * {@inheritDoc}
     */
    public Object[] toArray() {
        return this.commandList.toArray();
    }

    /**
     * {@inheritDoc}
     */
    public Object[] toArray(Object[] a) {
        return this.commandList.toArray(a);
    }

    /**
     * <p>
     * 格納されているコマンドを配列で返します。
     * </p>
     * 
     * @return コマンドの配列。
     */
    public Command[] toCommandArray() {
        return (Command[]) this.commandList.toArray(new Command[0]);
    }

    /**
     * {@inheritDoc}
     */
    public boolean add(Object o) {
        return this.addCommand((Command) o);
    }

    /**
     * <p>
     * コマンドを追加します。
     * </p>
     * 
     * @param command
     *            追加するコマンド。<br>
     *            nullの場合、{@link NullPointerException}例外をスローします。
     * @return 追加に成功した場合はtrue。
     */
    public boolean addCommand(Command command) {
        if (command == null) {
            throw new NullPointerException("commandがnullです。");
        }
        return this.commandList.add(command);
    }

    /**
     * {@inheritDoc}
     */
    public boolean remove(Object o) {
        return this.commandList.remove(o);
    }

    /**
     * {@inheritDoc}
     */
    public boolean containsAll(Collection c) {
        return this.commandList.containsAll(c);
    }

    /**
     * {@inheritDoc}
     */
    public boolean addAll(Collection c) {
        if (c == null) {
            throw new NullPointerException("cがnullです。");
        }
        for (Iterator i = c.iterator(); i.hasNext();) {
            Object o = i.next();
            this.add(o);
        }
        return true;
    }

    /**
     * {@inheritDoc}
     */
    public boolean addAll(int index, Collection c) {
        if (c == null) {
            throw new NullPointerException("cがnullです。");
        }
        for (Iterator i = c.iterator(); i.hasNext();) {
            Object o = i.next();
            this.add(index, o);
            index++;
        }
        return true;
    }

    /**
     * {@inheritDoc}
     */
    public boolean removeAll(Collection c) {
        return this.commandList.removeAll(c);
    }

    /**
     * {@inheritDoc}
     */
    public boolean retainAll(Collection c) {
        return this.commandList.retainAll(c);
    }

    /**
     * {@inheritDoc}
     */
    public void clear() {
        this.commandList.clear();
    }

    /**
     * {@inheritDoc}
     */
    public Object get(int index) {
        return this.getCommand(index);
    }

    /**
     * <p>
     * コマンドを返します。
     * </p>
     * 
     * @param index
     *            取得するコマンドの位置を表すインデックス番号。<br>
     *            インデックス番号が範囲外の場合、{@link IndexOutOfBoundsException}例外をスローします。
     * @return 指定した位置のコマンド。
     */
    public Command getCommand(int index) {
        return (Command) this.commandList.get(index);
    }

    /**
     * {@inheritDoc}
     */
    public Object set(int index, Object element) {
        return this.setCommand(index, (Command) element);
    }

    /**
     * <p>
     * 指定した位置のコマンドを指定したコマンドで置き換えます。
     * </p>
     * 
     * @param index
     *            置き換えるコマンドの位置を表すインデックス番号。<br>
     *            インデックス番号が範囲外の場合、{@link IndexOutOfBoundsException}例外をスローします。
     * @param command
     *            格納するコマンド。<br>
     *            nullの場合、{@link NullPointerException}例外をスローします。
     * @return 指定した位置に以前あったコマンド。
     */
    public Command setCommand(int index, Command command) {
        if (command == null) {
            throw new NullPointerException("commandがnullです。");
        }
        return (Command) this.commandList.set(index, command);
    }

    /**
     * {@inheritDoc}
     */
    public void add(int index, Object element) {
        this.addCommand(index, (Command) element);
    }

    /**
     * <p>
     * 指定した位置にコマンドを挿入します。
     * </p>
     * 
     * @param index
     *            コマンドを挿入する位置を表すインデックス番号。<br>
     *            インデックス番号が範囲外の場合、{@link IndexOutOfBoundsException}例外をスローします。
     * @param command
     *            挿入するコマンド。<br>
     *            nullの場合、{@link NullPointerException}例外をスローします。
     */
    public void addCommand(int index, Command command) {
        if (command == null) {
            throw new NullPointerException("commandがnullです。");
        }
        this.commandList.add(index, command);
    }

    /**
     * {@inheritDoc}
     */
    public Object remove(int index) {
        return this.commandList.remove(index);
    }

    /**
     * {@inheritDoc}
     */
    public int indexOf(Object o) {
        return this.commandList.indexOf(o);
    }

    /**
     * {@inheritDoc}
     */
    public int lastIndexOf(Object o) {
        return this.commandList.lastIndexOf(o);
    }

    /**
     * {@inheritDoc}
     */
    public ListIterator listIterator() {
        return this.commandList.listIterator();
    }

    /**
     * {@inheritDoc}
     */
    public ListIterator listIterator(int index) {
        return this.commandList.listIterator(index);
    }

    /**
     * {@inheritDoc}
     */
    public List subList(int fromIndex, int toIndex) {
        return this.commandList.subList(fromIndex, toIndex);
    }

    /**
     * {@inheritDoc}
     */
    public String toString() {
        String str = "";
        for (Iterator i = this.iterator(); i.hasNext();) {
            Command c = (Command) i.next();
            str += c.toString() + "\n";
        }
        return str;
    }

}
