/*
 * Copyright (c) 2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.ta.data.model;

import java.io.Serializable;
import java.util.Date;

import jp.sf.orangesignal.ta.data.CompressType;
import jp.sf.orangesignal.ta.data.annotation.AnnotationParser;
import jp.sf.orangesignal.ta.data.annotation.DateItem;
import jp.sf.orangesignal.ta.data.annotation.NumberCompressorType;
import jp.sf.orangesignal.ta.data.annotation.NumberItem;
import jp.sf.orangesignal.ta.util.Assert;

/**
 * 価格データセットを提供します。
 * 
 * @author 杉澤 浩二
 * @since 2.2
 */
@NumberCompressorType(CompressType.SUM)
public class PriceDataset implements Serializable {

	private static final long serialVersionUID = 3074771444092709164L;

	/**
	 * 日時データを保持します。
	 */
	@DateItem
	protected Date[] date;

	/**
	 * 始値データを保持します。
	 */
	@NumberItem
	@NumberCompressorType(CompressType.FIRST)
	protected Number[] open;

	/**
	 * 高値データを保持します。
	 */
	@NumberItem
	@NumberCompressorType(CompressType.HIGHEST)
	protected Number[] high;

	/**
	 * 安値データを保持します。
	 */
	@NumberItem
	@NumberCompressorType(CompressType.LOWEST)
	protected Number[] low;

	/**
	 * 終値データを保持します。
	 */
	@NumberItem
	@NumberCompressorType(CompressType.LAST)
	protected Number[] close;

	/**
	 * 出来高データを保持します。
	 */
	@NumberItem
	@NumberCompressorType(CompressType.SUM)
	protected Number[] volume;

	/**
	 * デフォルトコンストラクタです。
	 */
	public PriceDataset() {}

	/**
	 * 指定された価格データから価格データセットの新しいインスタンスを構築して返します。
	 * 
	 * @param prices 価格データ
	 * @return 価格データセット
	 * @throws IllegalArgumentException <code>prices</code> が <code>null</code> または要素が存在しない場合
	 */
	public static PriceDataset newInstance(final Price[] prices) {
		Assert.notEmpty(prices, "Prices must not be empty");
		return AnnotationParser.parse(prices).build(PriceDataset.class).execute();
	}

	/**
	 * 日時データを返します。
	 * 
	 * @return 日時データ
	 */
	public Date[] getDate() { return date; }

	/**
	 * 日時データを設定します。
	 * 
	 * @param date 日時データ
	 */
	public void setDate(final Date[] date) { this.date = date; }

	/**
	 * 始値データを返します。
	 * 
	 * @return 始値データ
	 */
	public Number[] getOpen() { return open; }

	/**
	 * 始値データを設定します。
	 * 
	 * @param open 始値データ
	 */
	public void setOpen(final Number[] open) { this.open = open; }

	/**
	 * 高値データを返します。
	 * 
	 * @return 高値データ
	 */
	public Number[] getHigh() { return high; }

	/**
	 * 高値データを設定します。
	 * 
	 * @param high 高値データ
	 */
	public void setHigh(final Number[] high) { this.high = high; }

	/**
	 * 安値データを返します。
	 * 
	 * @return 安値データ
	 */
	public Number[] getLow() { return low; }

	/**
	 * 安値データを設定します。
	 * 
	 * @param low 安値データ
	 */
	public void setLow(final Number[] low) { this.low = low; }

	/**
	 * 終値データを返します。
	 * 
	 * @return 終値データ
	 */
	public Number[] getClose() { return close; }

	/**
	 * 終値データを設定します。
	 * 
	 * @param close 終値データ
	 */
	public void setClose(final Number[] close) { this.close = close; }

	/**
	 * 出来高データを返します。
	 * 
	 * @return 出来高データ。出来高データが存在しない場合は <code>null</code>
	 */
	public Number[] getVolume() { return volume; }

	/**
	 * 出来高データを設定します。
	 * 
	 * @param volume 出来高データ
	 */
	public void setVolume(final Number[] volume) { this.volume = volume; }

}
