/*
 * Copyright (c) 2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.ta.data;

import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.zip.ZipFile;

import jp.gr.java_conf.dangan.util.lha.LhaFile;
import jp.sf.orangesignal.csv.Csv;
import jp.sf.orangesignal.csv.CsvListHandler;
import jp.sf.orangesignal.ta.util.Assert;

/**
 * CSV データロード機能を提供します。
 * 
 * @param <T> データの型
 * @author 杉澤 浩二
 * @since 2.2
 */
public abstract class CsvDataLoader<T> {

	/**
	 * CSV データ設定情報を保持します。
	 */
	private CsvDataConfig config;

	/**
	 * CSV データアクセスハンドラを保持します。
	 */
	private CsvListHandler<T> handler;

	/**
	 * デフォルトコンストラクタです。
	 */
	public CsvDataLoader() {}

	/**
	 * CSV データ設定情報を設定します。
	 * 
	 * @param config CSV データ設定情報
	 */
	public void setConfig(final CsvDataConfig config) { this.config = config; }

	/**
	 * CSV データアクセスハンドラを設定します。
	 * 
	 * @param handler CSV データアクセスハンドラ
	 */
	public void setHandler(final CsvListHandler<T> handler) { this.handler = handler; }

/*
	public boolean listed(final String symbol) {
		final File file = getFile(symbol);
		return file.isFile() && file.exists();
	}
*/

	/**
	 * 指定されたシンボルの CSV ファイルから CSV データを読込んで返します。
	 * 
	 * @param symbol シンボル
	 * @return CSV データのリスト
	 * @throws IOException 入出力例外が発生した場合
	 */
	public List<T> load(final String symbol) throws IOException {
		final File file = getFile(symbol);
		final List<T> list;
		switch (config.getType()) {
			case ZIP:
				final ZipFile zipFile = new ZipFile(file);
				try {
					list = Csv.load(zipFile, config.getEncoding(), config.getCsvConfig(), handler);
				} finally {
					zipFile.close();
				}
				break;
			case LHA:
				final LhaFile lhaFile = new LhaFile(file);
				try {
					list = Csv.load(lhaFile, config.getEncoding(), config.getCsvConfig(), handler);
				} finally {
					lhaFile.close();
				}
				break;
			default:
				list = Csv.load(file, config.getEncoding(), config.getCsvConfig(), handler);
		}
		if (config.isReverseOrder()) {
			Collections.sort(list, Collections.reverseOrder());
		}
		return list;
	}

	private File getFile(final String symbol) {
		Assert.notEmpty(symbol, "Symbol must not be empty");
		return new File(config.getPath(), new StringBuilder().append(symbol).append('.').append(config.getType().toString().toLowerCase()).toString());
	}

}
