/*
 * Copyright (c) 2008-2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.ta.candle;

import java.util.Date;

/**
 * ローソク足情報のインターフェースです。
 * 
 * @author 杉澤 浩二
 */
public interface Candlestick {

	/**
	 * トレンドの種類を返します。
	 * 
	 * @return トレンドの種類または <code>null</code>
	 */
	public TrendType getTrend();

	/**
	 * 陰陽線の種類を返します。
	 * 
	 * @return 陰陽線の種類
	 */
	public CandlestickColor getColor();

	/**
	 * 胴体の種類を返します。
	 * 
	 * @return 胴体の種類
	 */
	public RealBodyType getBodyType();

	/**
	 * 上影(上ヒゲ)の種類を返します。
	 * 
	 * @return 上影(上ヒゲ)の種類
	 */
	public ShadowType getUpperShadowType();

	/**
	 * 下影(下ヒゲ)の種類を返します。
	 * 
	 * @return 下影(下ヒゲ)の種類
	 */
	public ShadowType getLowerShadowType();

	/**
	 * 日時を返します。
	 * 
	 * @return 日時
	 */
	public Date getDate();

	/**
	 * 始値を返します。
	 * 
	 * @return 始値
	 */
	public double getOpen();

	/**
	 * 高値を返します。
	 * 
	 * @return 高値
	 */
	public double getHigh();

	/**
	 * 安値を返します。
	 * 
	 * @return 安値
	 */
	public double getLow();

	/**
	 * 終値を返します。
	 * 
	 * @return 終値
	 */
	public double getClose();

	/**
	 * ローソクの長さ(高値－安値)を返します。
	 * 
	 * @return ローソクの長さ
	 */
	public double length();

	/**
	 * ローソク中央の値を返します。
	 * 
	 * @return ローソク中央の値
	 * @since 1.1
	 */
	public double median();

	/**
	 * 胴体の長さ(終値－始値)を返します。
	 * 
	 * @return 実体の長さ
	 */
	public double body();

	/**
	 * 胴体上部の値(始値または終値の大きい方)を返します。
	 * 
	 * @return 実体上部の値
	 */
	public double upperBody();

	/**
	 * 胴体下部の値(始値または終値の小さい方)を返します。
	 * 
	 * @return 実体下部の値
	 */
	public double lowerBody();

	/**
	 * 胴体中央の値を返します。
	 * 
	 * @return 実体中央の値
	 */
	public double midpoint();

	/**
	 * 影(ヒゲ)全体の長さ(ローソクの長さ－実体の長さ)を返します。
	 * 
	 * @return 影(ヒゲ)全体の長さ
	 */
	public double shadow();

	/**
	 * 上影(上ヒゲ)の長さを返します。
	 * 
	 * @return 上影(上ヒゲ)の長さ
	 */
	public double upperShadow();

	/**
	 * 下影(下ヒゲ)の長さを返します。
	 * 
	 * @return 下影(下ヒゲ)の長さ
	 */
	public double lowerShadow();

	/**
	 * このローソク足の形が大陽線かどうかを返します。
	 * 
	 * @return このローソク足が大陽線の場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isLongWhite();

	/**
	 * このローソク足の形が大陰線かどうかを返します。
	 * 
	 * @return このローソク足が大陰線の場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isLongBlack();

	/**
	 * このローソク足の形が小陽線かどうかを返します。
	 * 
	 * @return このローソク足が小陽線の場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isShortWhite();

	/**
	 * このローソク足の形が小陰線かどうかを返します。
	 * 
	 * @return このローソク足が小陰線の場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isShortBlack();

	/**
	 * このローソク足の形が丸坊主かどうかを返します。
	 * 
	 * @return このローソク足が丸坊主の場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isMarubozu();

	/**
	 * このローソク足の形が寄付坊主かどうかを返します。
	 * 
	 * @return このローソク足の形が寄付坊主の場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isOpeningMarubozu();

	/**
	 * このローソク足の形が大引坊主かどうかを返します。
	 * 
	 * @return このローソク足の形が大引坊主の場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isClosingMarubozu();

	/**
	 * このローソク足の形が小さい丸坊主かどうかを返します。
	 * 
	 * @return このローソク足が小さい丸坊主の場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isSmallMarubozu();

	/**
	 * このローソク足の形が小さい寄付坊主かどうかを返します。
	 * 
	 * @return このローソク足の形が小さい寄付坊主の場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isSmallOpeningMarubozu();

	/**
	 * このローソク足の形が小さい大引坊主かどうかを返します。
	 * 
	 * @return このローソク足の形が小さい大引坊主の場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isSmallClosingMarubozu();

	/**
	 * このローソク足の形が独楽(コマ)かどうかを返します。
	 * 
	 * @return このローソク足の形が独楽(コマ)の場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isSpinningTop();

	/**
	 * このローソク足の形が足長独楽(コマ)かどうかを返します。
	 * 
	 * @return このローソク足の形が足長独楽(コマ)の場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isHighWave();

	/**
	 * このローソク足の形が唐傘(カラカサ)又は首吊りかどうかを返します。
	 * 
	 * @return このローソク足の形が唐傘(カラカサ)又は首吊りの場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isHangingMan();

	/**
	 * このローソク足の形が金槌(カナヅチ/トンカチ)かどうかを返します。
	 * 
	 * @return このローソク足の形が金槌(カナヅチ/トンカチ)の場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isShootingStar();

	/**
	 * このローソク足の形が足長同時かどうかを返します。
	 * 
	 * @return このローソク足の形が足長同時の場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isLongLeggedDoji();

	/**
	 * このローソク足の形が塔婆(トウバ)かどうかを返します。
	 * 
	 * @return このローソク足の形が塔婆(トウバ)の場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isGravestoneDoji();

	/**
	 * このローソク足の形が蜻蛉(トンボ)かどうかを返します。
	 * 
	 * @return このローソク足の形が蜻蛉(トンボ)の場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isDragonflyDoji();

	/**
	 * このローソク足の形が四値同時かどうかを返します。
	 * 
	 * @return このローソク足の形が四値同時の場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isFourPriceDoji();

	/**
	 * 指定された値が、このローソク足に包括されるかどうかを返します。
	 * 
	 * @param a 値
	 * @return 包括される場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean contains(final double a);

	/**
	 * 指定された値が、このローソク足に包括されるかどうかを返します。
	 * 
	 * @param a 値1
	 * @param b 値2
	 * @return 包括される場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean contains(final double a, final double b);

	/**
	 * 指定されたローソク足が、このローソク足に包括されるかどうかを返します。
	 * 
	 * @param c ローソク足
	 * @return 包括される場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean contains(final Candlestick c);

	/**
	 * 指定されたローソク足の実体が、このローソク足に包括されるかどうかを返します。
	 * 
	 * @param c ローソク足
	 * @return 包括される場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean containsByBody(final Candlestick c);

	/**
	 * 指定された値が、このローソク足の実体に包括されるかどうかを返します。
	 * 
	 * @param a 値
	 * @return 包括される場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean containsBody(final double a);

	/**
	 * 指定された値が、このローソク足の実体に包括されるかどうかを返します。
	 * 
	 * @param a 値1
	 * @param b 値2
	 * @return 包括される場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean containsBody(final double a, final double b);

	/**
	 * 指定されたローソク足が、このローソク足の実体に包括されるかどうかを返します。
	 * 
	 * @param c ローソク足
	 * @return 包括される場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean containsBody(final Candlestick c);

	/**
	 * 指定されたローソク足の実体が、このローソク足の実体に包括されるかどうかを返します。
	 * 
	 * @param c ローソク足
	 * @return 包括される場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean containsBodyByBody(final Candlestick c);

	/**
	 * 指定された値が、このローソク足にオーバラップするかどうかを返します。
	 * 
	 * @param a 値1
	 * @param b 値2
	 * @return オーバラップする場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean overlaps(final double a, final double b);

	/**
	 * 指定されたローソク足が、このローソク足にオーバラップするかどうかを返します。
	 * 
	 * @param c ローソク足
	 * @return オーバラップする場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean overlaps(final Candlestick c);

	/**
	 * 指定されたローソク足の実体が、このローソク足にオーバラップするかどうかを返します。
	 * 
	 * @param c ローソク足
	 * @return オーバラップする場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean overlapsByBody(final Candlestick c);

	/**
	 * 指定された値が、このローソク足の実体にオーバラップするかどうかを返します。
	 * 
	 * @param a 値1
	 * @param b 値2
	 * @return オーバラップする場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean overlapsBody(final double a, final double b);

	/**
	 * 指定されたローソク足が、このローソク足の実体にオーバラップするかどうかを返します。
	 * 
	 * @param c ローソク足
	 * @return オーバラップする場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean overlapsBody(final Candlestick c);

	/**
	 * 指定されたローソク足の実体が、このローソク足の実体にオーバラップするかどうかを返します。
	 * 
	 * @param c ローソク足
	 * @return オーバラップする場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean overlapsBodyByBody(final Candlestick c);

	/**
	 * 指定された値とこのローソク足にギャップ(窓)があるかどうかを返します。
	 * 
	 * @param a 値1
	 * @param b 値2
	 * @return ギャップ(窓)がある場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean gap(final double a, final double b);

	/**
	 * 指定されたローソク足とこのローソク足にギャップ(窓)があるかどうかを返します。
	 * 
	 * @param c ローソク足
	 * @return ギャップ(窓)がある場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean gap(final Candlestick c);

	/**
	 * 指定されたローソク足の実体とこのローソク足にギャップ(窓)があるかどうかを返します。
	 * 
	 * @param c ローソク足
	 * @return ギャップ(窓)がある場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean gapByBody(final Candlestick c);

	/**
	 * 指定された値とこのローソク足の実体にギャップ(窓)があるかどうかを返します。
	 * 
	 * @param a 値1
	 * @param b 値2
	 * @return ギャップ(窓)がある場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean gapBody(final double a, final double b);

	/**
	 * 指定されたローソク足とこのローソク足の実体にギャップ(窓)があるかどうかを返します。
	 * 
	 * @param c ローソク足
	 * @return ギャップ(窓)がある場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean gapBody(final Candlestick c);

	/**
	 * 指定されたローソク足の実体とこのローソク足の実体にギャップ(窓)があるかどうかを返します。
	 * 
	 * @param c ローソク足
	 * @return ギャップ(窓)がある場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean gapBodyByBody(final Candlestick c);

	/**
	 * 指定された値とこのローソク足に上位のギャップ(窓)があるかどうかを返します。
	 * 
	 * @param a 値1
	 * @param b 値2
	 * @return ギャップ(窓)がある場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean gapUp(final double a, final double b);

	/**
	 * 指定されたローソク足とこのローソク足に上位のギャップ(窓)があるかどうかを返します。
	 * 
	 * @param c ローソク足
	 * @return ギャップ(窓)がある場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean gapUp(final Candlestick c);

	/**
	 * 指定されたローソク足の実体とこのローソク足に上位のギャップ(窓)があるかどうかを返します。
	 * 
	 * @param c ローソク足
	 * @return ギャップ(窓)がある場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean gapUpByBody(final Candlestick c);

	/**
	 * 指定された値とこのローソク足の実体に上位のギャップ(窓)があるかどうかを返します。
	 * 
	 * @param a 値1
	 * @param b 値2
	 * @return ギャップ(窓)がある場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean gapUpBody(final double a, final double b);

	/**
	 * 指定されたローソク足とこのローソク足の実体に上位のギャップ(窓)があるかどうかを返します。
	 * 
	 * @param c ローソク足
	 * @return ギャップ(窓)がある場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean gapUpBody(final Candlestick c);

	/**
	 * 指定されたローソク足の実体とこのローソク足の実体に上位のギャップ(窓)があるかどうかを返します。
	 * 
	 * @param c ローソク足
	 * @return ギャップ(窓)がある場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean gapUpBodyByBody(final Candlestick c);

	/**
	 * 指定された値とこのローソク足に下位のギャップ(窓)があるかどうかを返します。
	 * 
	 * @param a 値1
	 * @param b 値2
	 * @return ギャップ(窓)がある場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean gapDown(final double a, final double b);

	/**
	 * 指定されたローソク足とこのローソク足に下位のギャップ(窓)があるかどうかを返します。
	 * 
	 * @param c ローソク足
	 * @return ギャップ(窓)がある場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean gapDown(final Candlestick c);

	/**
	 * 指定されたローソク足の実体とこのローソク足に下位のギャップ(窓)があるかどうかを返します。
	 * 
	 * @param c ローソク足
	 * @return ギャップ(窓)がある場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean gapDownByBody(final Candlestick c);

	/**
	 * 指定された値とこのローソク足の実体に下位のギャップ(窓)があるかどうかを返します。
	 * 
	 * @param a 値1
	 * @param b 値2
	 * @return ギャップ(窓)がある場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean gapDownBody(final double a, final double b);

	/**
	 * 指定されたローソク足とこのローソク足の実体に下位のギャップ(窓)があるかどうかを返します。
	 * 
	 * @param c ローソク足
	 * @return ギャップ(窓)がある場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean gapDownBody(final Candlestick c);

	/**
	 * 指定されたローソク足の実体とこのローソク足の実体に下位のギャップ(窓)があるかどうかを返します。
	 * 
	 * @param c ローソク足
	 * @return ギャップ(窓)がある場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean gapDownBodyByBody(final Candlestick c);

	/**
	 * 指定された値が始値の同距離かどうかを返します。
	 * 
	 * @param a 値
	 * @return 始値の同距離である場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isSameOpen(final double a);

	/**
	 * 指定された値が高値の同距離かどうかを返します。
	 * 
	 * @param a 値
	 * @return 高値の同距離である場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isSameHigh(final double a);

	/**
	 * 指定された値が安値の同距離かどうかを返します。
	 * 
	 * @param a 値
	 * @return 安値の同距離である場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isSameLow(final double a);

	/**
	 * 指定された値が終値の同距離かどうかを返します。
	 * 
	 * @param a 値
	 * @return 終値の同距離である場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isSameClose(final double a);

	/**
	 * このローソク足の同距離基準値を使用して、指定された値同士が同距離かどうかを返します。
	 * 
	 * @param a 値1
	 * @param b 値2
	 * @return 値同士が同距離である場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isSame(final double a, final double b);

	/**
	 * 指定された値が始値の近距離かどうかを返します。
	 * 
	 * @param a 値
	 * @return 始値の近距離である場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isNearOpen(final double a);

	/**
	 * 指定された値が高値の近距離かどうかを返します。
	 * 
	 * @param a 値
	 * @return 高値の近距離である場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isNearHigh(final double a);

	/**
	 * 指定された値が安値の近距離かどうかを返します。
	 * 
	 * @param a 値
	 * @return 安値の近距離である場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isNearLow(final double a);

	/**
	 * 指定された値が終値の近距離かどうかを返します。
	 * 
	 * @param a 値
	 * @return 終値の近距離である場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isNearClose(final double a);

	/**
	 * このローソク足の近距離基準値を使用して、指定された値同士が近距離かどうかを返します。
	 * 
	 * @param a 値1
	 * @param b 値2
	 * @return 値同士が近距離である場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isNear(final double a, final double b);

	/**
	 * 指定された値が始値の遠距離かどうかを返します。
	 * 
	 * @param a 値
	 * @return 始値の遠距離である場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isFarOpen(final double a);

	/**
	 * 指定された値が高値の遠距離かどうかを返します。
	 * 
	 * @param a 値
	 * @return 高値の遠距離である場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isFarHigh(final double a);

	/**
	 * 指定された値が安値の遠距離かどうかを返します。
	 * 
	 * @param a 値
	 * @return 安値の遠距離である場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isFarLow(final double a);

	/**
	 * 指定された値が終値の遠距離かどうかを返します。
	 * 
	 * @param a 値
	 * @return 終値の遠距離である場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isFarClose(final double a);

	/**
	 * このローソク足の遠距離基準値を使用して、指定された値同士が遠距離かどうかを返します。
	 * 
	 * @param a 値1
	 * @param b 値2
	 * @return 値同士が遠距離である場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public boolean isFar(final double a, final double b);

	/**
	 * 終値がこのローソク足の 75% 以上の位置にあるかどうかを返します。
	 * 
	 * @return 終値がこのローソク足の 75% 以上の位置にある場合は <code>true</code> それ以外の場合は <code>false</code>
	 * @since 1.1
	 */
	public boolean isCloseInTopQuarter();

	/**
	 * 終値がこのローソク足の 25% 以下の位置にあるかどうかを返します。
	 * 
	 * @return 終値がこのローソク足の 25% 以下の位置にある場合は <code>true</code> それ以外の場合は <code>false</code>
	 * @since 1.1
	 */
	public boolean isCloseInBottomQuarter();

}