/*
 * Copyright (c) 2009-2010 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.csv.handlers;

import java.io.IOException;
import java.lang.reflect.Field;
import java.text.Format;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

import jp.sf.orangesignal.csv.CsvReader;
import jp.sf.orangesignal.csv.CsvWriter;
import jp.sf.orangesignal.csv.filters.CsvValueFilter;
import jp.sf.orangesignal.csv.manager.CsvColumnPositionMappingBeanManager;

/**
 * 区切り文字形式データの項目位置を基準として Java プログラム要素のリストと区切り文字形式データアクセスを行うハンドラを提供します。
 * 
 * @author 杉澤 浩二
 * @see CsvColumnPositionMappingBeanManager
 */
public class ColumnPositionMappingBeanListHandler<T> extends BeanListHandlerSupport<T, ColumnPositionMappingBeanListHandler<T>> {

	/**
	 * 項目位置と Java プログラム要素のフィールド名のマップを保持します。
	 */
	private SortedMap<Integer, String> columnMapping = new TreeMap<Integer, String>();

	/**
	 * 区切り文字形式データフィルタを保持します。
	 */
	private CsvValueFilter valueFilter;

	/**
	 * ヘッダを使用するかどうかを保持します。
	 * 
	 * @since 1.2
	 */
	private boolean useHeader = true;

	// ------------------------------------------------------------------------

	/**
	 * コンストラクタです。
	 * 
	 * @param type Java プログラム要素の型
	 * @throws IllegalArgumentException <code>type</code> が <code>null</code> の場合
	 */
	public ColumnPositionMappingBeanListHandler(final Class<T> type) {
		super(type);
	}

	// ------------------------------------------------------------------------

	/**
	 * 指定された Java プログラム要素のフィールド名を現在の最後の項目位置としてマップへ追加します。
	 * 
	 * @param field Java プログラム要素のフィールド名
	 * @return このオブジェクトへの参照
	 */
	public ColumnPositionMappingBeanListHandler<T> addColumn(final String field) {
		return addColumn(field, null);
	}

	/**
	 * 指定された Java プログラム要素のフィールド名を現在の最後の項目位置としてマップへ追加します。
	 * 
	 * @param field Java プログラム要素のフィールド名
	 * @param format フィールドを処理するフォーマットオブジェクト (オプション)
	 * @return このオブジェクトへの参照
	 * @since 1.2
	 */
	public ColumnPositionMappingBeanListHandler<T> addColumn(final String field, final Format format) {
		return addColumn(getMaxColumnPosition() + 1, field, format);
	}

	/**
	 * 指定された項目位置と Java プログラム要素のフィールド名をマップへ追加します。
	 * 
	 * @param position 項目位置
	 * @param field Java プログラム要素のフィールド名
	 * @return このオブジェクトへの参照
	 */
	public ColumnPositionMappingBeanListHandler<T> addColumn(final int position, final String field) {
		return addColumn(position, field, null);
	}

	/**
	 * 指定された項目位置と Java プログラム要素のフィールド名をマップへ追加します。
	 * 
	 * @param position 項目名
	 * @param field Java プログラム要素のフィールド名
	 * @param format フィールドを処理するフォーマットオブジェクト (オプション)
	 * @return このオブジェクトへの参照
	 * @since 1.2
	 */
	public ColumnPositionMappingBeanListHandler<T> addColumn(final int position, final String field, final Format format) {
		columnMapping.put(position, field);
		if (format != null) {
			setValueParser(field, format);
			setValueFormatter(Integer.valueOf(position), format);
		}
		return this;
	}

	/**
	 * 項目位置と Java プログラム要素のフィールド名のマップを設定します。
	 * 
	 * @param columnMapping 項目位置と Java プログラム要素のフィールド名のマップ
	 * @throws IllegalArgumentException <code>columnMapping</code> が <code>null</code> の場合
	 * @since 1.2.4
	 */
	public void setColumnMapping(final Map<Integer, String> columnMapping) {
		if (columnMapping == null) {
			throw new IllegalArgumentException("Column mapping must not be null");
		}
		this.columnMapping = new TreeMap<Integer, String>(columnMapping);
	}

	/**
	 * 項目位置と Java プログラム要素のフィールド名のマップを設定します。
	 * 
	 * @param columnMapping 項目位置と Java プログラム要素のフィールド名のマップ
	 * @return このオブジェクトへの参照
	 * @throws IllegalArgumentException <code>columnMapping</code> が <code>null</code> の場合
	 */
	public ColumnPositionMappingBeanListHandler<T> columnMapping(final Map<Integer, String> columnMapping) {
		setColumnMapping(columnMapping);
		return this;
	}

	/**
	 * 区切り文字形式データフィルタを設定します。
	 * 
	 * @param filter 区切り文字形式データフィルタ
	 * @return このオブジェクトへの参照
	 * @since 1.2.3
	 */
	public ColumnPositionMappingBeanListHandler<T> filter(final CsvValueFilter filter) {
		this.valueFilter = filter;
		return this;
	}

	// ------------------------------------------------------------------------

	private int getMaxColumnPosition() {
		if (columnMapping.size() > 0) {
			return columnMapping.lastKey();
		}
		return -1;
	}

	// ------------------------------------------------------------------------

	@Override
	public List<T> load(final CsvReader reader, final boolean ignoreScalar) throws IOException {
		// TODO: 必要があればここに入力パラメータを検証するコードを記述します。

		// 項目位置とフィールド名のマップが指定されていない場合は、最初の行をヘッダとして読込んでマップを作成します。
		if (columnMapping.size() == 0) {
			final List<String> columnNames = reader.readValues();
			if (columnNames == null) {
				// ヘッダがない場合は例外をスローします。
				throw new IOException("No header is available");
			}
			for (final String name : columnNames) {
				addColumn(name);
			}
		}

		// データ部を処理します。
		final List<T> results = new ArrayList<T>();
		final Field[] fields = getType().getDeclaredFields();
		final Map<String, Object[]> fieldColumnsMap = createFieldAndColumnsMap(columnMapping);
		final boolean order = ignoreScalar || (orders != null && !orders.isEmpty());
		int offset = 0;
		List<String> values;
		while ((values = reader.readValues()) != null && (order || limit <= 0 || results.size() < limit)) {
			if (valueFilter != null && !valueFilter.accept(values)) {
				continue;
			}
			if (beanFilter == null && !order && offset < this.offset) {
				offset++;
				continue;
			}
			final T bean = createBean();
			for (final Field f : fields) {
				final Object[] columns = fieldColumnsMap.get(f.getName());
				final int count = columns == null ? 0 : columns.length;

				Object o = null;
				if (count == 1) {
					final int pos = (Integer) columns[0];
					if (pos >= 0) {
						o = stringToObject(f, values.get(pos));
					}
				} else if (count > 1) {
					final StringBuilder sb = new StringBuilder();
					for (final Object column : columns) {
						final int pos = (Integer) column;
						if (pos >= 0) {
							final String s = values.get(pos);
							if (s != null) {
								sb.append(s);
							}
						}
					}
					o = stringToObject(f, sb.toString());
				}
				if (o != null) {
					setFieldValue(bean, f, o);
				}
			}
			if (beanFilter != null) {
				if (!beanFilter.accept(bean)) {
					continue;
				}
				if (!order && offset < this.offset) {
					offset++;
					continue;
				}
			}
			results.add(bean);
		}

		if (ignoreScalar || !order) {
			return results;
		}
		return processScalar(results);
	}

	@Override
	public void save(final List<T> list, final CsvWriter writer) throws IOException {
		// TODO: 必要があればここに入力パラメータを検証するコードを記述します。

		// 項目位置とフィールド名のマップが指定されていない場合は、フィールドからマップを作成します。
		if (columnMapping.size() == 0) {
			for (final Field f : getType().getDeclaredFields()) {
				addColumn(f.getName());
			}
		}
		final int columnCount = getMaxColumnPosition() + 1;

		// ヘッダ部を処理します。
		if (useHeader) {
			String[] names = new String[columnCount];
			for (final Map.Entry<Integer, String> e : columnMapping.entrySet()) {
				names[e.getKey()] = e.getValue();
			}
			writer.writeValues(Arrays.asList(names));
		}

		// データ部を処理します。
		for (final T bean : list) {
			// 要素が null の場合は null 出力します。
			if (bean == null) {
				writer.writeValues(null);
				continue;
			} else if (beanFilter != null && !beanFilter.accept(bean)) {
				continue;
			}

			final Class<?> type = bean.getClass();
			final String[] values = new String[columnCount];
			for (final Map.Entry<Integer, String> e : columnMapping.entrySet()) {
				final int pos = e.getKey();
				if (pos == -1) {
					continue;
				}
				final Field f = getField(type, e.getValue());
				values[pos] = objectToString(Integer.valueOf(pos), getFieldValue(bean, f));
			}
			final List<String> _values = Arrays.asList(values);
			if (valueFilter != null && !valueFilter.accept(_values)) {
				continue;
			}
			writer.writeValues(_values);
		}
	}

}
