/*
 * Copyright (c) 2006-2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.chart.ui.screen;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ItemListener;

import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;

import jp.sf.orangesignal.chart.ChartSettings;
import jp.sf.orangesignal.chart.ChartSettings.AntiWatchSettings;
import jp.sf.orangesignal.chart.event.ChartEvent;
import jp.sf.orangesignal.chart.event.ChartScreenEvent;
import jp.sf.orangesignal.chart.ui.Icons;
import jp.sf.orangesignal.chart.ui.MAType;
import jp.sf.orangesignal.chart.ui.canvas.IndexCanvas;
import jp.sf.orangesignal.chart.util.StringManager;

/**
 * 逆ウォッチ曲線サイド画面を提供します。
 * 
 * @author 杉澤 浩二
 */
public class AntiWatchSideScreen extends SideScreen implements ItemListener {

	private static final long serialVersionUID = -6148675225780002493L;

	/**
	 * 指標キャンバスを保持します。
	 */
	private final IndexCanvas canvas;

	/**
	 * [範囲に0を含める]チェックボックスを保持します。
	 */
	private final JCheckBox fixed = new JCheckBox(StringManager.getString("fixed"));

	/**
	 * [シグナルマーカー]チェックボックスを保持します。
	 */
	private final JCheckBox marker = new JCheckBox(StringManager.getString("marker.signal"));

	/**
	 * 移動平均の種類コンボボックスを保持します。
	 */
	private final JComboBox maType = createMovingAverageJComboBox();

	/**
	 * 移動平均の期間ボックスを保持します。
	 */
	private final JSpinner maPeriod = new JSpinner(new SpinnerNumberModel(25, 1, 200, 1));

	/**
	 * コンストラクタです。
	 * 
	 * @param icons アイコン情報
	 */
	public AntiWatchSideScreen(final Icons icons) {
		// 各コントロールを初期化します。
		this.canvas = new IndexCanvas(icons);

		final GridBagLayout layout = new GridBagLayout();
		final GridBagConstraints c = new GridBagConstraints();
		setLayout(layout);

		int y = 0;

		addCanvas(layout, c, y++, this.canvas);
		addCheckBox(layout, c, y++, this.fixed);
		addCheckBox(layout, c, y++, this.marker);
		addSeparator(layout, c, y++);
		addPanel(layout, c, y, createMaPanel());
	}

	private JPanel createMaPanel() {
		final JPanel panel = new JPanel();
		final GridBagLayout layout = new GridBagLayout();
		final GridBagConstraints c = new GridBagConstraints();
		panel.setLayout(layout);

		int y = 0;

		addSubComboBox(panel, layout, c, y++, new JLabel(StringManager.getString("matype")), this.maType);
		addSubSpinner(panel, layout, c, y, new JLabel(StringManager.getString("period")), this.maPeriod);

		return panel;
	}

	@Override
	public void load(final ChartSettings settings) {
		setVisibleAdvancedOptions(settings.advancedOptions);
		final AntiWatchSettings s = settings.antiwatch;
		this.fixed.setSelected(s.fixed);
		this.marker.setSelected(s.signalMarker);
		this.maType.setSelectedItem(s.maType);
		this.maPeriod.setValue(s.maPeriod);
	}

	@Override
	public void save(final ChartSettings settings) {
		final AntiWatchSettings s = settings.antiwatch;
		s.fixed = this.fixed.isSelected();
		s.signalMarker = this.marker.isSelected();
		s.maType = (MAType) this.maType.getSelectedItem();
		s.maPeriod = (Integer) this.maPeriod.getValue();
	}

	/**
	 * 指定された設定情報とこの画面の設定情報を比較して、開始位置や期間を無視できるかどうかを返します。
	 * 
	 * @param settings 設定情報
	 * @return 開始位置や期間を無視できるかどうか
	 */
	public boolean isIgnoreStart(final ChartSettings settings) {
		final AntiWatchSettings s = settings.antiwatch;
		return s.maPeriod == ((Integer) this.maPeriod.getValue()).intValue();
	}

	@Override
	public void datasetSwitch(final ChartEvent e) {
		super.notify = false;
		final boolean enabled = e.getDataset() != null;

		this.fixed.setEnabled(enabled);
		this.marker.setEnabled(enabled);
		this.maType.setEnabled(enabled);
		this.maPeriod.setEnabled(enabled);

		if (enabled)
			load(e.getSettings());

		this.canvas.switchDataset(e);
		super.notify = true;
	}

	@Override
	public void periodSwitch(final ChartEvent e) {
		this.canvas.switchDataset(e);
	}

	@Override
	public void settingChanged(final ChartEvent e) {
		this.canvas.switchDataset(e);
	}

	@Override
	public void positionChanged(final ChartScreenEvent e) {
		this.canvas.setup(e);
	}

	@Override
	public void setVisibleAdvancedOptions(final boolean visible) {
		this.fixed.setVisible(visible);
	}

}
