package jp.sf.nikonikofw.util;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;

import jp.sf.nikonikofw.util.converter.DefaultConverter;
import jp.sf.nikonikofw.util.converter.IConverter;

/**
 * 文字列関係のユーティリティメソッドを提供します。
 * 
 * @author Naoki Takezoe
 */
public class StringUtils {
	
	private static List<IConverter> converters = new ArrayList<IConverter>();
	static {
		converters.add(new DefaultConverter());
	}
	
	/**
	 * 文字列がnullまたは空文字列であるかをテストします。
	 * 
	 * @param value 文字列
	 * @return 文字列がnullまたは空文字列の場合true、そうでない場合false
	 */
	public static boolean isEmpty(String value){
		return value == null || value.length() == 0;
	}
	
	/**
	 * 文字列がnullまたは空文字列でないかをテストします。
	 * 
	 * @param value 文字列
	 * @return 文字列がnullまたは空文字列でない場合true、そうでない場合false
	 */
	public static boolean isNotEmpty(String value){
		return !isEmpty(value);
	}
	
	/**
	 * 文字列がnullまたは空文字列（空白文字を含む）であるかをテストします。
	 * 
	 * @param value 文字列
	 * @return 文字列がnullまたは空文字列（空白文字を含む）である場合true、そうでない場合false
	 */
	public static boolean isBlank(String value){
		return value == null || value.trim().length() == 0;
	}
	
	/**
	 * 文字列がnullまたは空文字列（空白文字を含む）でないかをテストします。
	 * 
	 * @param value 文字列
	 * @return 文字列がnullまたは空文字列（空白文字を含む）でない場合true、そうでない場合false
	 */
	public static boolean isNotBlank(String value){
		return !isBlank(value);
	}
	
	/**
	 * {@link StringUtils#convert(String, Class)}で使用するコンバータを登録します。
	 * 
	 * @param converter コンバータ
	 */
	public static void registerConverter(IConverter converter){
		converters.add(converter);
	}
	
	/**
	 * <code>IConverter</code>の実装クラスを用いて文字列を指定した型に変換します。
	 * 
	 * @param value 文字列
	 * @param targetType 変換後の型
	 * @return 変換後の値
	 */
	public static Object convert(String value, Class<?> targetType){
		for(IConverter converter: converters){
			if(converter.isSupport(targetType)){
				return converter.convert(value, targetType);
			}
		}
		return null;
	}
	
	/**
	 * HTMLタグをエスケープします。
	 * 
	 * @param value 文字列
	 * @return HTMLタグをエスケープした文字列
	 */
	public static String escapeHtml(String value){
		if(StringUtils.isEmpty(value)){
			return "";
		}
		value = value.replace("&", "&amp;");
		value = value.replace("<", "&lt;");
		value = value.replace(">", "&gt;");
		value = value.replace("\"", "&quot;");
		return value;
	}
	
	/**
	 * 改行をbrタグに変換します。
	 * 
	 * @param value 文字列
	 * @return 改行をbrタグに変換した文字列
	 */
	public static String convertBr(String value){
		if(StringUtils.isEmpty(value)){
			return "";
		}
		value = value.replace("\r\n", "\n");
		value = value.replace("\r", "\n");
		value = value.replace("\n", "<br>");
		return value;
	}
	
	/**
	 * JavaScriptの文字列リテラルをエスケープします。
	 *
	 * @param value 文字列
	 * @return エスケープされた文字列
	 */
	public static String escapeJavaScript(String value){
		return value;
	}
	
	/**
	 * URLをリンクに変換します。
	 * 
	 * @param value 文字列
	 * @return URLをリンクに変換した文字列
	 */
	public static String convertLink(String value){
		return value.replaceAll(
				"(http|https)://[A-Za-z0-9\\._~/:\\-?&=%;]+", "<a href=\"$0\">$0</a>");
	}
	
	/**
	 * 例外のスタックトレースを文字列として取得します。
	 * 
	 * @param t 例外
	 * @return スタックトレース
	 */
	public static String getStackTrace(Throwable t){
		StringWriter writer = new StringWriter();
		t.printStackTrace(new PrintWriter(writer));
		return writer.toString();
	}
	
	/**
	 * 文字列が全て半角数字かどうかをチェックします。
	 * 
	 * @param str 文字列
	 * @return 全て半角数字の場合true、半角数字ではない場合false
	 */
	public static boolean isNumeric(String str){
		if(str == null){
			return true;
		}
		for(int i=0;i<str.length();i++){
			char c = str.charAt(i);
			if(c < '0' || c > '9'){
				return false;
			}
		}
		return true;
	}
	
}
