package jp.sf.nikonikofw.persistence.jdbc;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import jp.sf.nikonikofw.Config;
import jp.sf.nikonikofw.persistence.jdbc.annotation.Column;
import jp.sf.nikonikofw.persistence.jdbc.annotation.PrimaryKey;
import jp.sf.nikonikofw.persistence.jdbc.annotation.Table;
import jp.sf.nikonikofw.util.StringUtils;

/**
 * エンティティのメタ情報を表すクラスです。
 * 
 * @author Naoki Takezoe
 */
public class TableMeta {
	
	private String tableName;
	
	private List<ColumnMeta> columns = new ArrayList<ColumnMeta>();
	
	/**
	 * テーブル名を取得します。
	 * 
	 * @return テーブル名
	 */
	public String getTableName() {
		return tableName;
	}

	/**
	 * テーブル名を設定します。
	 * 
	 * @param tableName テーブル名
	 */
	public void setTableName(String tableName) {
		this.tableName = tableName;
	}

	/**
	 * カラムのメタ情報のリストを取得します。
	 * 
	 * @return カラムのメタ情報のリスト
	 */
	public List<ColumnMeta> getColumns() {
		return columns;
	}

	/**
	 * カラムのメタ情報のリストを設定します。
	 * 
	 * @param columns カラムのメタ情報のリスト
	 */
	public void setColumns(List<ColumnMeta> columns) {
		this.columns = columns;
	}
	
	/**
	 * TableMetaをキャッシュするためのマップ。
	 */
	private static Map<Class<?>, TableMeta> cache = new ConcurrentHashMap<Class<?>, TableMeta>();

	/**
	 * エンティティのクラスオブジェクトからメタ情報を取得します。
	 * 
	 * @param entityClass エンティティのクラスオブジェクト
	 * @return エンティティのメタ情報
	 */
	public static TableMeta getTableMeta(Class<?> entityClass){
		// キャッシュがある場合はキャッシュから返却
		TableMeta tableMeta = cache.get(entityClass);
		if(tableMeta != null){
			return tableMeta;
		}
		
		INaming naming = ((JdbcPersistenceManager) Config.getPersistenceManager()).getNaming();
		
		tableMeta = new TableMeta();
		
		Table tableAnn = entityClass.getAnnotation(Table.class);
		if(tableAnn != null && StringUtils.isNotEmpty(tableAnn.name())){
			tableMeta.setTableName(tableAnn.name());
		} else {
			tableMeta.setTableName(naming.toDatabaseName(entityClass.getSimpleName()));
		}
		
		for(Field field: entityClass.getDeclaredFields()){
			field.setAccessible(true);
			
			Column columnAnn = field.getAnnotation(Column.class);
			if(columnAnn != null){
				ColumnMeta column = new ColumnMeta();
				column.setField(field);
				column.setPk(field.getAnnotation(PrimaryKey.class) != null);
				
				column.setColumnName(naming.toDatabaseName(field.getName()));
				if(StringUtils.isNotEmpty(columnAnn.name())){
					column.setColumnName(columnAnn.name());
				}
				
				tableMeta.getColumns().add(column);
			}
			
		}
		
		// キャッシュに格納
		cache.put(entityClass, tableMeta);
		
		return tableMeta;
	}
	
}
