package net.sf.amateras.nikocale.action.admin;

import java.util.Calendar;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import jp.sf.nikonikofw.action.IAction;
import jp.sf.nikonikofw.annotation.Request;
import jp.sf.nikonikofw.util.DateUtils;
import jp.sf.nikonikofw.util.StringUtils;
import net.sf.amateras.nikocale.entity.Entry;
import net.sf.amateras.nikocale.entity.SystemInfo;
import net.sf.amateras.nikocale.service.EntryService;
import net.sf.amateras.nikocale.service.SystemInfoService;

import org.apache.commons.mail.SimpleEmail;

/**
 * 昨日更新されたニコカレをメールで送信するアクションです。
 * <p>
 * SYSTEM_INFOテーブルのMAIL_ADDRESSカラムに設定されたメールアドレスに送信します。
 * このアクションを呼び出す際にクエリ文字列でaddressパラメータを指定することで、
 * 指定したアドレスに送信することもできます。
 * <p>
 * また、SYSTEM_INFOテーブルのSMTP_SERVER、SMTP_USER、SMTP_PASSWORDカラムに設定された
 * 情報を使用してメールの送信を行います。SMTPサーバの接続情報や送信先のメールアドレスが
 * 設定されていない場合、このアクションを呼び出してもメールの送信は行いません。
 * <p>
 * メールの本文にはニコカレへのリンクが出力されます。このリンクはこのアクションが
 * 呼び出された際のURLから自動的に生成されますが、クエリ文字列でurlパラメータを
 * 指定することで任意のURLに置換することが可能です。
 * 
 * @author Naoki Takezoe
 */
public class SendmailAction implements IAction {
	
	@Request
	public String url;
	
	@Request
	public String address;
	
	@Override
	public String execute(HttpServletRequest request,
			HttpServletResponse response) throws Exception {
		
		if("usage".equals(request.getQueryString())){
			request.setAttribute("title", "メール通知");
			if(StringUtils.isEmpty(url)){
				url = request.getRequestURL().toString().replaceFirst("/sendmail\\.do$", "");
			}
			request.setAttribute("url", url);
			return "sendmail.jsp";
		}
		
		SystemInfo info = SystemInfoService.getSystemInfo();
		
		// SMTPサーバが設定されていない場合はなにもしない
		if(StringUtils.isEmpty(info.getSmtpServer())){
			return null;
		}
		// 送信先を決定
		if(StringUtils.isEmpty(address)){
			address = info.getMailAddress();
			if(StringUtils.isEmpty(address)){
				return null;
			}
		}
		// 送信元を決定
		if(StringUtils.isEmpty(info.getMailFrom())){
			return null;
		}
		// 置換するURLを決定
		if(StringUtils.isEmpty(url)){
			url = request.getRequestURL().toString().replaceFirst("/admin/sendmail\\.do$", "");
		}
		
		List<Entry> entries = EntryService.getYesterdayEntries();
		
		// 昨日の日付を取得
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, -1);
		String yesterday = DateUtils.formatDate(cal.getTime());
		
		StringBuilder sb = new StringBuilder();
		if(entries.isEmpty()){
			sb.append("昨日（").append(yesterday).append("）更新されたニコカレはありませんでした。\n\n");
			
		} else {
			sb.append("昨日（").append(yesterday).append("）更新されたニコカレです。\n\n");
			for(Entry entry: entries){
				sb.append("■").append(entry.getMemberName()).append("さん - ")
					.append(DateUtils.formatDateTime(entry.getUpdateDate()))
					.append("\n");
				if(entry.getStatus() == 0){
					sb.append("(ニコニコ)");
				} else if(entry.getStatus() == 1){
					sb.append("(普通)");
				} else if(entry.getStatus() == 2){
					sb.append("(ダメダメ)");
				}
				sb.append("『").append(entry.getMessage()).append("』\n");
				if(entry.getCommentCount() != 0){
					sb.append("※").append(entry.getCommentCount()).append("件のコメントがあります。\n");
				}
				sb.append(String.format(
						"%s/entry.do?memberId=%d&year=%d&month=%d&day=%d\n\n", 
						url, entry.getUserId(), entry.getYear(), entry.getMonth(), entry.getDay()));
			}
		}
		
		sb.append("--\n");
		sb.append("ニコカレには以下のURLからアクセスできます。\n");
		sb.append(String.format("%s/\n", url));
		
		String smtpServer = info.getSmtpServer();
		
		// SMTPサーバのポート番号が指定されていた場合
		int port = -1;
		int index = smtpServer.indexOf(':');
		if(index > 0){
			smtpServer = smtpServer.substring(0, index - 1);
			port = Integer.parseInt(smtpServer.substring(index + 1));
		}
		
		SimpleEmail mail = new SimpleEmail();
		mail.setHostName(smtpServer);
		if(port >= 0){
			mail.setSmtpPort(port);
		}
		if(StringUtils.isNotEmpty(info.getSmtpUser()) && StringUtils.isNotEmpty(info.getSmtpPassword())){
			mail.setAuthentication(info.getSmtpUser(), info.getSmtpPassword());
		}
		mail.addTo(address);
		mail.setFrom(info.getMailFrom());
		mail.setSubject("【ニコカレ】" +yesterday + "のニコカレ");
		mail.setContent(sb.toString(), "text/plain; charset=ISO-2022-JP");
		mail.setCharset("ISO-2022-JP");
		
		
		mail.send();
		
		// System.out.println(sb.toString());
		
		return null;
	}

}
