﻿/*
[momiji music component library]
---------------------------------------------------------------------
Momiji.Sequencer.Midi.Smf.cpp
	stream component of standard midi file.
---------------------------------------------------------------------
Copyright (C) 2011 tyiki badwell {miria@users.sourceforge.jp}.

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/gpl-3.0.html>.
---------------------------------------------------------------------
*/
#include "StdAfx.h"

#include "Momiji.Sequencer.Midi.Smf.h"
#include "Momiji.Core.Midi.Data.h"
#include "Momiji.Util.h"

namespace Momiji{
namespace Sequencer {
namespace Midi {
namespace Smf {

	SmfStream::SmfStream(System::String^ path)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		this->Open(path);
		this->CheckSMF();
		this->Rewind();
	}

	SmfStream::~SmfStream()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		this->!SmfStream();
	}

	SmfStream::!SmfStream()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		
		if (this->_tracks != nullptr)
		{
			delete this->_tracks;
			this->_tracks = nullptr;
		}

		if (this->_view != nullptr)
		{
			delete this->_view;
			this->_view = nullptr;
		}
		if (this->_mmap != nullptr)
		{
			delete this->_mmap;
			this->_mmap = nullptr;
		}
	}

	void SmfStream::Open(System::String^ path)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] {1}",__FUNCTION__, path);
		#endif

		auto info = gcnew System::IO::FileInfo(path);

		this->_mmap = 
			System::IO::MemoryMappedFiles::MemoryMappedFile::CreateFromFile(
				info->FullName,
				System::IO::FileMode::Open,
				"momiji.smf",
				info->Length,
				System::IO::MemoryMappedFiles::MemoryMappedFileAccess::Read
			);

		this->_view =
			this->_mmap->CreateViewAccessor(
				0,
				0,
				System::IO::MemoryMappedFiles::MemoryMappedFileAccess::Read
			);
	}

	void SmfStream::CheckSMF()
	{
		System::UInt64 offset = 0;

		this->_view->Read<SMFHEADER_CHUNK>(offset, this->_smfHeader);

		this->_smfHeader.dwType		= Util::ToHostOrder(this->_smfHeader.dwType);
		this->_smfHeader.dwLength	= Util::ToHostOrder(this->_smfHeader.dwLength);
		this->_smfHeader.wFormat	= Util::ToHostOrder(this->_smfHeader.wFormat);
		this->_smfHeader.wNtrks		= Util::ToHostOrder(this->_smfHeader.wNtrks);
		this->_smfHeader.wDivition	= Util::ToHostOrder(this->_smfHeader.wDivition);

		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1,4:X}]",__FUNCTION__, this->_smfHeader.dwType);
			System::Console::WriteLine("[{0}] [{1}]",__FUNCTION__, this->_smfHeader.dwLength);
			System::Console::WriteLine("[{0}] [{1}]",__FUNCTION__, this->_smfHeader.wFormat);
			System::Console::WriteLine("[{0}] [{1}]",__FUNCTION__, this->_smfHeader.wNtrks);
			System::Console::WriteLine("[{0}] [{1}]",__FUNCTION__, this->_smfHeader.wDivition);
		#endif

		safe_cast<TYPE>(this->_smfHeader.dwType);
		safe_cast<FORMAT>(this->_smfHeader.wFormat);

		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1}]",__FUNCTION__, safe_cast<TYPE>(this->_smfHeader.dwType));
			System::Console::WriteLine("[{0}] [{1}]",__FUNCTION__, safe_cast<FORMAT>(this->_smfHeader.wFormat));
		#endif

		offset += this->_smfHeader.dwLength + 8;

		this->_tracks = gcnew array<SmfTrack^>(this->_smfHeader.wNtrks);
		for(System::UInt16 idx = 0; idx < this->_smfHeader.wNtrks; idx++)
		{
			SMFTRACK_CHUNK smfTrack;
			this->_view->Read<SMFTRACK_CHUNK>(offset, smfTrack);

			smfTrack.dwType		= Util::ToHostOrder(smfTrack.dwType);
			smfTrack.dwLength	= Util::ToHostOrder(smfTrack.dwLength);

			#ifdef _DEBUG
				System::Console::WriteLine("[{0}] [{1,4:X}]",__FUNCTION__, smfTrack.dwType);
				System::Console::WriteLine("[{0}] [{1}]",__FUNCTION__, smfTrack.dwLength);
			#endif

			safe_cast<TYPE>(smfTrack.dwType);

			#ifdef _DEBUG
				System::Console::WriteLine("[{0}] [{1}]",__FUNCTION__, safe_cast<TYPE>(smfTrack.dwType));
			#endif

			this->_tracks[idx] =
				gcnew SmfTrack(
					this,
					(offset + 8),
					(offset + 8 + smfTrack.dwLength - 1)
				);

			offset += smfTrack.dwLength + 8;
		}
	}

	//TODO: packetにIComparerを実装してもSortで反応してくれなかったので、仕方なくコレで。
	int IStreamPacketCompare(Core::IStreamPacket^ a, Core::IStreamPacket^ b) 
	{
		//TODO: RPN・NRPNの塊は維持したままソートしないとダメだった
		return a->tick.CompareTo(b->tick);
	}

	array<Core::IStreamPacket^>^ SmfStream::GetStreamPacket(System::Double deltaTime)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		auto result = gcnew System::Collections::Generic::List<Core::IStreamPacket^>;

		for each(auto track in this->_tracks)
		{
			auto data = track->GetStreamPacket(deltaTime);
			result->AddRange(data);
		}

	//	result->Sort(gcnew System::Comparison<Core::IStreamPacket^>(IStreamPacketCompare));

		return result->ToArray();
	}

	void SmfStream::Rewind()
	{
		for each (auto track in this->_tracks)
		{
			track->Rewind();
		}

		this->_speed = 100.0;
	}

	SmfStream::SmfTrack::SmfTrack(
		SmfStream^ stream,
		System::UInt64 headOffset, 
		System::UInt64 tailOffset
	):	_stream(stream),
		_headOffset(headOffset), 
		_tailOffset(tailOffset),
		_metaData(gcnew System::Collections::Generic::Dictionary<MetaData::META_TYPE, MetaData^>)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		this->Rewind();
	}

	SmfStream::SmfTrack::~SmfTrack()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		this->!SmfTrack();
	}

	SmfStream::SmfTrack::!SmfTrack()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
	}

	void SmfStream::SmfTrack::Rewind()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		this->_nowTick = .0;

		this->_metaData[MetaData::META_TYPE::SET_TEMPO] = gcnew MetaSetTempo(600000);
		this->_metaData[MetaData::META_TYPE::PORT] = gcnew MetaPort(0);
		this->_metaData[MetaData::META_TYPE::TIME_SIGNATURE] = gcnew MetaTimeSignature(4, 4, 24, 8);
		this->_metaData[MetaData::META_TYPE::KEY_SIGNATURE] = gcnew MetaKeySignature(0, false);

		this->_headPacket =
			gcnew SmfPacket(
				this,
				this->_headOffset,
				0,
				0
			);

		this->_packet = this->_headPacket;
	}

	array<Core::IStreamPacket^>^ SmfStream::SmfTrack::GetStreamPacket(System::Double deltaTime)
	{
		auto result = gcnew System::Collections::Generic::List<Core::IStreamPacket^>;

		auto packet = this->_packet;

		#ifdef _DEBUG
			System::Console::WriteLine("[{0}][nowtick:{1}]",__FUNCTION__, this->_nowTick);
		#endif

		auto metaSetTempo = safe_cast<MetaSetTempo^>(this->_metaData[MetaData::META_TYPE::SET_TEMPO]);
		this->_nowTick += (deltaTime * ((this->_stream->_smfHeader.wDivition * this->_stream->_speed * 10.0) / metaSetTempo->usecPerQNote));

		while(packet->hasNext && (packet->tick <= this->_nowTick))
		{
			#ifdef _DEBUG
				System::Console::WriteLine("[{0}][tick:{1}]",__FUNCTION__, packet->tick);
			#endif

			result->Add(packet);
			packet = packet->next;
		}

		this->_packet = packet;
		return result->ToArray();
	}

	SmfStream::SmfTrack::SmfPacket::SmfPacket(
		SmfTrack^			track,
		System::UInt64		offset,
		System::UInt64		previousTick,
		System::Byte		previousStatus
	): _track(track), _offset(offset), _isTerminate(false)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		auto tempOffset = this->_offset;

		//デルタタイム読み込み
		this->_tick = previousTick + this->GetDeltaTime(tempOffset); 
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}][tick:{1}]",__FUNCTION__, this->_tick);
		#endif
		
		//次にイベントデータ読み込み
		this->_status = this->NextByte(tempOffset);
		System::Byte param1 = 0;

		if (this->_status < 0x80)
		{//ランニングステータス
			param1 = this->_status;
			this->_status = previousStatus;
		}
		else
		{
			param1 = this->NextByte(tempOffset);
		}

		Core::IData^ data = nullptr;

		#ifdef _DEBUG
			System::Console::WriteLine("[{0}][status:{1,2:X}]",__FUNCTION__, this->_status);
		#endif

		auto metaPort = safe_cast<MetaPort^>(this->_track->_metaData[MetaData::META_TYPE::PORT]);

		switch (safe_cast<Core::Midi::MidiData::TYPE>(this->_status & 0xF0))
		{
		case Core::Midi::MidiData::TYPE::NOTE_OFF				:// {8x 128..143  | nn vv |  n:note num; v:velocity}
		case Core::Midi::MidiData::TYPE::NOTE_ON				:// {9x 144..159  | nn vv |  n:note num; v:velocity}
		case Core::Midi::MidiData::TYPE::POLYHONIC_KEY_PRESSURE	:// {Ax 160..176  | nn vv |  n:note num; v:velocity}
		case Core::Midi::MidiData::TYPE::CONTROL_CHANGE			:// {Bx 176..191  | cc vv |  n:ctrl num; v:value}
		case Core::Midi::MidiData::TYPE::PITCH_WHILE_CHANGE		:// {Ex 224..239  | ll mm |  l:least sig; m:most sig}
			{
				data =
					gcnew Core::Midi::ShortData(
						metaPort->number,
						this->_status,
						param1,
						this->NextByte(tempOffset)
					);
				break;
			}
		case Core::Midi::MidiData::TYPE::PROGRAM_CHANGE			:// {Cx 192..207  | pp    |  p:prog num}
		case Core::Midi::MidiData::TYPE::CNANNEL_PRESSURE		:// {Dx 208..223  | cc    |  c:chan num}
			{
				data =
					gcnew Core::Midi::ShortData(
						metaPort->number,
						this->_status,
						param1,
						0
					);
				break;
			}
		case Core::Midi::MidiData::TYPE::SYSTEM_MESSAGE			://	{Fx 240..255  | ll dd..dd}
			{
				switch(safe_cast<Core::Midi::MidiData::TYPE>(this->_status))
				{
				case Core::Midi::MidiData::TYPE::SYSTEM_EXCLUSIVE	://	{F0 length ID data..data F7}
					{
						auto size = param1;
						auto longData = gcnew array<System::Byte>(size+1);
						longData[0] = this->_status;
						this->NextByte(longData, tempOffset, 1, size);

						data =
							gcnew Core::Midi::LongData(
								metaPort->number,
								longData,
								longData->Length
							);
						break;
					}
				case Core::Midi::MidiData::TYPE::META				://	{FF: type length bytes}
					{
						auto type = param1;
						auto size = this->NextByte(tempOffset);
						//TODO 読み取りに長さチェックを加える
						auto bytes = gcnew array<System::Byte>(size+1);
						if (size > 0)
						{
							this->NextByte(bytes, tempOffset, 0, size);
						}
						bytes[size] = 0;

						#ifdef _DEBUG
							System::Console::WriteLine("[{0}][type:{1,2:X}][size:{2}]]",__FUNCTION__, type, size);
						#endif

						auto metaType = safe_cast<MetaData::META_TYPE>(type);

						switch(metaType)
						{
						case MetaData::META_TYPE::SEQUENCE_NUMBER: //  {FF: '00' 02     ss ss}
							{
								data =
									gcnew MetaSequenceNumber(
										Util::ToHostOrder(System::BitConverter::ToUInt16(bytes,0))
									);
								break;
							}
						case MetaData::META_TYPE::TEXT_EVENT:	//  {FF: '01' length text}
							{
								data =
									gcnew MetaText(
										""
									);
								break;
							}
						case MetaData::META_TYPE::COPYRIGHT_NOTICE://  {FF: '02' length text}
							{
								data =
									gcnew MetaCopyrightNotice(
										""
									);
								break;
							}
						case MetaData::META_TYPE::TRACK_NAME: //  {FF: '03' length text}
							{
								data =
									gcnew MetaTrackName(
										""
									);
								break;
							}
						case MetaData::META_TYPE::INSTRUMENT_NAME: //  {FF: '04' length text}
							{
								data =
									gcnew MetaInstrumentName(
										""
									);
								break;
							}
						case MetaData::META_TYPE::LYRIC://  {FF: '05' length text}
							{
								data =
									gcnew MetaLyric(
										""
									);
								break;
							}
						case MetaData::META_TYPE::MARKER://  {FF: '06' length text}
							{
								data =
									gcnew MetaMarker(
										""
									);
								break;
							}
						case MetaData::META_TYPE::CUE_POINT://  {FF: '07' length text}
							{
								data =
									gcnew MetaCuePoint(
										""
									);
								break;
							}
						case MetaData::META_TYPE::PROGRAM_NAME://  {FF: '08' length text}
							{
								data =
									gcnew MetaProgramName(
										""
									);
								break;
							}
						case MetaData::META_TYPE::DEVICE_NAME: //  {FF: '09' length text}
							{
								data =
									gcnew MetaDeviceName(
										""
									);
								break;
							}
						case MetaData::META_TYPE::PORT: //  {FF: '21' length PORT ??}	< 2バイト目の役割が不明；
							{
								System::UInt16 port = bytes[0];

								data =
									gcnew MetaPort(
										port //ToHostOrder(System::BitConverter::ToUInt16(bytes,0))
									);
								break;
							}
						case MetaData::META_TYPE::END_OF_TRACK: //  {FF: '2F'  00}
							{
								data =
									gcnew MetaEndOfTrack();

								this->_isTerminate = true;
								break;
							}
						case MetaData::META_TYPE::SET_TEMPO: //  {FF: '51'  03    tt tt tt   = microceconds/quarter note( Tempo = (60×10^6)÷MetaTempo )}
							{
								auto tempo = gcnew array<System::Byte>(4);
								tempo[0] = 0;
								tempo[1] = bytes[0];
								tempo[2] = bytes[1];
								tempo[3] = bytes[2];

								data =
									gcnew MetaSetTempo(
										Util::ToHostOrder(System::BitConverter::ToUInt32(tempo, 0))
									);
						
								break;
							}
						case MetaData::META_TYPE::SMPTE_OFFSET: //  {FF: '54'  05    hr mn se fr ff}
							{
								data =
									gcnew MetaSMPTEOffset(
										bytes[0],
										bytes[1],
										bytes[2],
										bytes[3],
										bytes[4]
									);
								break;
							}
						case MetaData::META_TYPE::TIME_SIGNATURE: //  {FF: '58'  04    nn dd cc bb}
							{
								data =
									gcnew MetaTimeSignature(
										bytes[0],
										2 ^ bytes[1],
										bytes[2],
										bytes[3]
									);
								break;
							}
						case MetaData::META_TYPE::KEY_SIGNATURE: //  {FF: '59'  02    sf mi}
							{
								data =
									gcnew MetaKeySignature(
										bytes[0],
										System::BitConverter::ToBoolean(bytes,1)
									);
								break;
							}
						case MetaData::META_TYPE::SEQ_SP_METAEVENT: //  {FF: '7F'  length data  ユーザ定義メタ}
							{
								data = nullptr;
								break;
							}
						}

						auto metaData = safe_cast<MetaData^>(data);
						if (
								metaData->conductor 
							&&	(safe_cast<SmfStream::FORMAT>(this->_track->_stream->_smfHeader.wFormat) == SmfStream::FORMAT::MULTIPLE_SYNC)
						)
						{
							for each(SmfTrack^ track in this->_track->_stream->_tracks)
							{
								track->_metaData[metaType] = metaData;
							}
						}
						else
						{
							this->_track->_metaData[metaType] = metaData;
						}

						break;
					}
				default:
					{
						throw gcnew SmfException("ステータス[{0,2:X}]は、対応していない。");
					}
				}
				break;
			}
		default:
			{
				//ここに到達できない
				throw gcnew SmfException("ロジック不正");
			}
		}

		this->_data = data;
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}][{1}]",__FUNCTION__, data);
		#endif

		this->_nextOffset = tempOffset;
	}

	SmfStream::SmfTrack::SmfPacket::~SmfPacket()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		this->!SmfPacket();
	}
	SmfStream::SmfTrack::SmfPacket::!SmfPacket()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
	}

	Core::IStreamPacket^ SmfStream::SmfTrack::SmfPacket::next::get()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		if (!this->hasNext)
		{
			throw gcnew SmfException("次のデータはありません。");
		}

		return
			gcnew SmfPacket(
				this->_track,
				this->_nextOffset,
				this->_tick,
				this->_status
			);
	}

	System::UInt32 SmfStream::SmfTrack::SmfPacket::GetDeltaTime(System::UInt64% tempOffset)
	{
		System::UInt32 result = 0;

		for (auto idx = 0; idx < 4; idx++)
		{
			auto data = this->NextByte(tempOffset);
			result = ((result << 7) | (data & 0x7F));
			if ((data & 0x80) == 0) break;
		}

		return result;
	}

	System::Byte SmfStream::SmfTrack::SmfPacket::NextByte(System::UInt64% tempOffset)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		
		return this->_track->_stream->_view->ReadByte(tempOffset++);
	}

	void SmfStream::SmfTrack::SmfPacket::NextByte(
		array<System::Byte>^ dest, 
		System::UInt64% tempOffset, 
		System::Byte start, 
		System::Byte size
	)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		this->_track->_stream->_view->ReadArray<System::Byte>(tempOffset, dest, start, size);
		tempOffset += size;
	}


}
}
}
}
