﻿/*
[momiji music component library]
---------------------------------------------------------------------
Momiji.Core.Vst.h
	vst dll importer.
---------------------------------------------------------------------
Copyright (C) 2011 tyiki badwell {miria@users.sourceforge.jp}.

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/gpl-3.0.html>.
---------------------------------------------------------------------
*/
#pragma once

#using <mscorlib.dll>

using namespace System::Runtime;

namespace Momiji{
namespace Interop {
namespace Vst {

#define VST_DLL_DELEGATE(_charset_)			[InteropServices::UnmanagedFunctionPointerAttribute(InteropServices::CallingConvention::Cdecl, CharSet = InteropServices::CharSet::##_charset_##, SetLastError = false)]

	//-------------------------------------------------------------------------------------------------------
	/** AEffect flags */
	//-------------------------------------------------------------------------------------------------------
	[System::FlagsAttribute]
	public enum class VstAEffectFlags: System::Int32
	{
	//-------------------------------------------------------------------------------------------------------
		effFlagsHasEditor     = 1 << 0,			//< set if the plug-in provides a custom editor
		effFlagsCanReplacing  = 1 << 4,			//< supports replacing process mode (which should the default mode in VST 2.4)
		effFlagsProgramChunks = 1 << 5,			//< program data is handled in formatless chunks
		effFlagsIsSynth       = 1 << 8,			//< plug-in is a synth (VSTi), Host may assign mixer channels for its outputs
		effFlagsNoSoundInStop = 1 << 9,			//< plug-in does not produce sound when input is all silence
		effFlagsCanDoubleReplacing = 1 << 12,	//< plug-in supports double precision processing

		effFlagsHasClipDeprecated = 1 << 1,			//< \deprecated deprecated in VST 2.4
		effFlagsHasVuDeprecated   = 1 << 2,			//< \deprecated deprecated in VST 2.4
		effFlagsCanMonoDeprecated = 1 << 3,			//< \deprecated deprecated in VST 2.4
		effFlagsExtIsAsyncDeprecated   = 1 << 10,	//< \deprecated deprecated in VST 2.4
		effFlagsExtHasBufferDeprecated = 1 << 11	//< \deprecated deprecated in VST 2.4
	//-------------------------------------------------------------------------------------------------------
	};

	//-------------------------------------------------------------------------------------------------------
	/** String length limits (in characters excl. 0 byte) */
	//-------------------------------------------------------------------------------------------------------
	public enum class VstStringConstants: System::Int32
	{
	//-------------------------------------------------------------------------------------------------------
		kVstMaxProgNameLen   = 24,	//< used for #effGetProgramName, #effSetProgramName, #effGetProgramNameIndexed
		kVstMaxParamStrLen   = 8,	//< used for #effGetParamLabel, #effGetParamDisplay, #effGetParamName
		kVstMaxVendorStrLen  = 64,	//< used for #effGetVendorString, #audioMasterGetVendorString
		kVstMaxProductStrLen = 64,	//< used for #effGetProductString, #audioMasterGetProductString
		kVstMaxEffectNameLen = 32,	//< used for #effGetEffectName
		kVstMaxNameLen       = 64,	//< used for #MidiProgramName, #MidiProgramCategory, #MidiKeyName, #VstSpeakerProperties, #VstPinProperties
		kVstMaxLabelLen      = 64,	//< used for #VstParameterProperties->label, #VstPinProperties->label
		kVstMaxShortLabelLen = 8,	//< used for #VstParameterProperties->shortLabel, #VstPinProperties->shortLabel
		kVstMaxCategLabelLen = 24,	//< used for #VstParameterProperties->label
		kVstMaxFileNameLen   = 100	//< used for #VstAudioFile->name
	//-------------------------------------------------------------------------------------------------------
	};

	WIN32_DLL_STRUCTLAYOUT_ANSI public ref struct AEffect
	{
	public:
	//-------------------------------------------------------------------------------------------------------
		System::Int32			magic;			//< must be #kEffectMagic ('VstP')

		/** Host to Plug-in dispatcher @see AudioEffect::dispatcher */
		System::IntPtr	dispatcher;
		
			/** \deprecated Accumulating process mode is deprecated in VST 2.4! Use AEffect::processReplacing instead! */
			System::IntPtr	processDeprecated;
	
		/** Set new value of automatable parameter @see AudioEffect::setParameter */
		System::IntPtr	setParameter;

		/** Returns current value of automatable parameter @see AudioEffect::getParameter*/
		System::IntPtr	getParameter;

		System::Int32 numPrograms;  //< number of programs
		System::Int32 numParams;	//< all programs are assumed to have numParams parameters
		System::Int32 numInputs;	//< number of audio inputs
		System::Int32 numOutputs;	//< number of audio outputs

		VstAEffectFlags flags;		//< @see VstAEffectFlags
	
		System::IntPtr resvd1;		//< reserved for Host, must be 0
		System::IntPtr resvd2;		//< reserved for Host, must be 0
	
		System::Int32 initialDelay;	//< for algorithms which need input in the first place (Group delay or latency in Samples). This value should be initialized in a resume state.
	
			System::Int32 realQualitiesDeprecated;	//< \deprecated unused member
			System::Int32 offQualitiesDeprecated;	//< \deprecated unused member
			System::Single ioRatioDeprecated;		//< \deprecated unused member

		System::IntPtr object;			//< #AudioEffect class pointer
		System::IntPtr user;			//< user-defined pointer

		System::Int32 uniqueID;		//< registered unique identifier (register it at Steinberg 3rd party support Web). This is used to identify a plug-in during save+load of preset and project.
		System::Int32 version;		//< plug-in version (example 1100 for version 1.1.0.0)

		/** Process audio samples in replacing mode @see AudioEffect::processReplacing */
		System::IntPtr processReplacing;

		/** Process double-precision audio samples in replacing mode @see AudioEffect::processDoubleReplacing */
		System::IntPtr processDoubleReplacing;
	
		[InteropServices::MarshalAs(InteropServices::UnmanagedType::ByValArray, SizeConst=56)]
			array<System::Char>^ future;	//< reserved for future use (please zero)
	//-------------------------------------------------------------------------------------------------------
	};

	//-------------------------------------------------------------------------------------------------------
	/** Basic dispatcher Opcodes (Host to Plug-in) */
	//-------------------------------------------------------------------------------------------------------
	public enum class AEffectOpcodes: System::Int32
	{
		effOpen = 0,		///< no arguments  @see AudioEffect::open
		effClose,			///< no arguments  @see AudioEffect::close

		effSetProgram,		///< [value]: new program number  @see AudioEffect::setProgram
		effGetProgram,		///< [return value]: current program number  @see AudioEffect::getProgram
		effSetProgramName,	///< [ptr]: char* with new program name, limited to #kVstMaxProgNameLen  @see AudioEffect::setProgramName
		effGetProgramName,	///< [ptr]: char buffer for current program name, limited to #kVstMaxProgNameLen  @see AudioEffect::getProgramName
	
		effGetParamLabel,	///< [ptr]: char buffer for parameter label, limited to #kVstMaxParamStrLen  @see AudioEffect::getParameterLabel
		effGetParamDisplay,	///< [ptr]: char buffer for parameter display, limited to #kVstMaxParamStrLen  @see AudioEffect::getParameterDisplay
		effGetParamName,	///< [ptr]: char buffer for parameter name, limited to #kVstMaxParamStrLen  @see AudioEffect::getParameterName
	
			effGetVuDeprecated,	///< \deprecated deprecated in VST 2.4

		effSetSampleRate,	///< [opt]: new sample rate for audio processing  @see AudioEffect::setSampleRate
		effSetBlockSize,	///< [value]: new maximum block size for audio processing  @see AudioEffect::setBlockSize
		effMainsChanged,	///< [value]: 0 means "turn off", 1 means "turn on"  @see AudioEffect::suspend @see AudioEffect::resume

		effEditGetRect,		///< [ptr]: #ERect** receiving pointer to editor size  @see ERect @see AEffEditor::getRect
		effEditOpen,		///< [ptr]: system dependent Window pointer, e.g. HWND on Windows  @see AEffEditor::open
		effEditClose,		///< no arguments @see AEffEditor::close

			effEditDrawDeprecated,	///< \deprecated deprecated in VST 2.4
			effEditMouseDeprecated,	///< \deprecated deprecated in VST 2.4
			effEditKeyDeprecated,	///< \deprecated deprecated in VST 2.4

		effEditIdle,		///< no arguments @see AEffEditor::idle
	
			effEditTopDeprecated,	///< \deprecated deprecated in VST 2.4
			effEditSleepDeprecated,	///< \deprecated deprecated in VST 2.4
			effIdentifyDeprecated,	///< \deprecated deprecated in VST 2.4
	
		effGetChunk,		///< [ptr]: void** for chunk data address [index]: 0 for bank, 1 for program  @see AudioEffect::getChunk
		effSetChunk,		///< [ptr]: chunk data [value]: byte size [index]: 0 for bank, 1 for program  @see AudioEffect::setChunk
 
		effProcessEvents,	///< [ptr]: #VstEvents*  @see AudioEffectX::processEvents

		effCanBeAutomated,						///< [index]: parameter index [return value]: 1=true, 0=false  @see AudioEffectX::canParameterBeAutomated
		effString2Parameter,					///< [index]: parameter index [ptr]: parameter string [return value]: true for success  @see AudioEffectX::string2parameter

			effGetNumProgramCategoriesDeprecated,	///< \deprecated deprecated in VST 2.4

		effGetProgramNameIndexed,				///< [index]: program index [ptr]: buffer for program name, limited to #kVstMaxProgNameLen [return value]: true for success  @see AudioEffectX::getProgramNameIndexed
	
			effCopyProgramDeprecated,	///< \deprecated deprecated in VST 2.4
			effConnectInputDeprecated,	///< \deprecated deprecated in VST 2.4
			effConnectOutputDeprecated,	///< \deprecated deprecated in VST 2.4
	
		effGetInputProperties,					///< [index]: input index [ptr]: #VstPinProperties* [return value]: 1 if supported  @see AudioEffectX::getInputProperties
		effGetOutputProperties,				///< [index]: output index [ptr]: #VstPinProperties* [return value]: 1 if supported  @see AudioEffectX::getOutputProperties
		effGetPlugCategory,					///< [return value]: category  @see VstPlugCategory @see AudioEffectX::getPlugCategory

			effGetCurrentPositionDeprecated,	///< \deprecated deprecated in VST 2.4
			effGetDestinationBufferDeprecated,	///< \deprecated deprecated in VST 2.4

		effOfflineNotify,						///< [ptr]: #VstAudioFile array [value]: count [index]: start flag  @see AudioEffectX::offlineNotify
		effOfflinePrepare,						///< [ptr]: #VstOfflineTask array [value]: count  @see AudioEffectX::offlinePrepare
		effOfflineRun,							///< [ptr]: #VstOfflineTask array [value]: count  @see AudioEffectX::offlineRun

		effProcessVarIo,						///< [ptr]: #VstVariableIo*  @see AudioEffectX::processVariableIo
		effSetSpeakerArrangement,				///< [value]: input #VstSpeakerArrangement* [ptr]: output #VstSpeakerArrangement*  @see AudioEffectX::setSpeakerArrangement

			effSetBlockSizeAndSampleRateDeprecated,	///< \deprecated deprecated in VST 2.4

		effSetBypass,							///< [value]: 1 = bypass, 0 = no bypass  @see AudioEffectX::setBypass
		effGetEffectName,						///< [ptr]: buffer for effect name, limited to #kVstMaxEffectNameLen  @see AudioEffectX::getEffectName

			effGetErrorTextDeprecated,	///< \deprecated deprecated in VST 2.4

		effGetVendorString,					///< [ptr]: buffer for effect vendor string, limited to #kVstMaxVendorStrLen  @see AudioEffectX::getVendorString
		effGetProductString,					///< [ptr]: buffer for effect vendor string, limited to #kVstMaxProductStrLen  @see AudioEffectX::getProductString
		effGetVendorVersion,					///< [return value]: vendor-specific version  @see AudioEffectX::getVendorVersion
		effVendorSpecific,						///< no definition, vendor specific handling  @see AudioEffectX::vendorSpecific
		effCanDo,								///< [ptr]: "can do" string [return value]: 0: "don't know" -1: "no" 1: "yes"  @see AudioEffectX::canDo
		effGetTailSize,						///< [return value]: tail size (for example the reverb time of a reverb plug-in); 0 is default (return 1 for 'no tail')

			effIdleDeprecated,				///< \deprecated deprecated in VST 2.4
			effGetIconDeprecated,			///< \deprecated deprecated in VST 2.4
			effSetViewPositionDeprecated,	///< \deprecated deprecated in VST 2.4

		effGetParameterProperties,				///< [index]: parameter index [ptr]: #VstParameterProperties* [return value]: 1 if supported  @see AudioEffectX::getParameterProperties

			effKeysRequiredDeprecated,	///< \deprecated deprecated in VST 2.4

		effGetVstVersion,						///< [return value]: VST version  @see AudioEffectX::getVstVersion

		effEditKeyDown,						///< [index]: ASCII character [value]: virtual key [opt]: modifiers [return value]: 1 if key used  @see AEffEditor::onKeyDown
		effEditKeyUp,							///< [index]: ASCII character [value]: virtual key [opt]: modifiers [return value]: 1 if key used  @see AEffEditor::onKeyUp
		effSetEditKnobMode,					///< [value]: knob mode 0: circular, 1: circular relativ, 2: linear (CKnobMode in VSTGUI)  @see AEffEditor::setKnobMode

		effGetMidiProgramName,					///< [index]: MIDI channel [ptr]: #MidiProgramName* [return value]: number of used programs, 0 if unsupported  @see AudioEffectX::getMidiProgramName
		effGetCurrentMidiProgram,				///< [index]: MIDI channel [ptr]: #MidiProgramName* [return value]: index of current program  @see AudioEffectX::getCurrentMidiProgram
		effGetMidiProgramCategory,				///< [index]: MIDI channel [ptr]: #MidiProgramCategory* [return value]: number of used categories, 0 if unsupported  @see AudioEffectX::getMidiProgramCategory
		effHasMidiProgramsChanged,				///< [index]: MIDI channel [return value]: 1 if the #MidiProgramName(s) or #MidiKeyName(s) have changed  @see AudioEffectX::hasMidiProgramsChanged
		effGetMidiKeyName,						///< [index]: MIDI channel [ptr]: #MidiKeyName* [return value]: true if supported, false otherwise  @see AudioEffectX::getMidiKeyName
	
		effBeginSetProgram,					///< no arguments  @see AudioEffectX::beginSetProgram
		effEndSetProgram,						///< no arguments  @see AudioEffectX::endSetProgram

		effGetSpeakerArrangement,				///< [value]: input #VstSpeakerArrangement* [ptr]: output #VstSpeakerArrangement*  @see AudioEffectX::getSpeakerArrangement
		effShellGetNextPlugin,					///< [ptr]: buffer for plug-in name, limited to #kVstMaxProductStrLen [return value]: next plugin's uniqueID  @see AudioEffectX::getNextShellPlugin

		effStartProcess,						///< no arguments  @see AudioEffectX::startProcess
		effStopProcess,						///< no arguments  @see AudioEffectX::stopProcess
		effSetTotalSampleToProcess,		    ///< [value]: number of samples to process, offline only!  @see AudioEffectX::setTotalSampleToProcess
		effSetPanLaw,							///< [value]: pan law [opt]: gain  @see VstPanLawType @see AudioEffectX::setPanLaw
	
		effBeginLoadBank,						///< [ptr]: #VstPatchChunkInfo* [return value]: -1: bank can't be loaded, 1: bank can be loaded, 0: unsupported  @see AudioEffectX::beginLoadBank
		effBeginLoadProgram,					///< [ptr]: #VstPatchChunkInfo* [return value]: -1: prog can't be loaded, 1: prog can be loaded, 0: unsupported  @see AudioEffectX::beginLoadProgram

		effSetProcessPrecision,				///< [value]: @see VstProcessPrecision  @see AudioEffectX::setProcessPrecision
		effGetNumMidiInputChannels,			///< [return value]: number of used MIDI input channels (1-15)  @see AudioEffectX::getNumMidiInputChannels
		effGetNumMidiOutputChannels,			///< [return value]: number of used MIDI output channels (1-15)  @see AudioEffectX::getNumMidiOutputChannels
	};

	//-------------------------------------------------------------------------------------------------------
	/** Basic dispatcher Opcodes (Plug-in to Host) */
	//-------------------------------------------------------------------------------------------------------
	public enum class AudioMasterOpcodes: System::Int32
	{
	//-------------------------------------------------------------------------------------------------------
		audioMasterAutomate = 0,	///< [index]: parameter index [opt]: parameter value  @see AudioEffect::setParameterAutomated
		audioMasterVersion,			///< [return value]: Host VST version (for example 2400 for VST 2.4) @see AudioEffect::getMasterVersion
		audioMasterCurrentId,		///< [return value]: current unique identifier on shell plug-in  @see AudioEffect::getCurrentUniqueId
		audioMasterIdle,			///< no arguments  @see AudioEffect::masterIdle
			audioMasterPinConnectedDeprecated,		///< \deprecated deprecated in VST 2.4 r2
			
			audioMasterDeprecated,
			audioMasterWantMidiDeprecated,	///< \deprecated deprecated in VST 2.4

		audioMasterGetTime,				///< [return value]: #VstTimeInfo* or null if not supported [value]: request mask  @see VstTimeInfoFlags @see AudioEffectX::getTimeInfo
		audioMasterProcessEvents,		///< [ptr]: pointer to #VstEvents  @see VstEvents @see AudioEffectX::sendVstEventsToHost

			audioMasterSetTimeDeprecated,	///< \deprecated deprecated in VST 2.4
			audioMasterTempoAtDeprecated,	///< \deprecated deprecated in VST 2.4
			audioMasterGetNumAutomatableParametersDeprecated,	///< \deprecated deprecated in VST 2.4
			audioMasterGetParameterQuantizationDeprecated,		///< \deprecated deprecated in VST 2.4

		audioMasterIOChanged,			///< [return value]: 1 if supported  @see AudioEffectX::ioChanged

			audioMasterNeedIdleDeprecated,	///< \deprecated deprecated in VST 2.4
	
		audioMasterSizeWindow,			///< [index]: new width [value]: new height [return value]: 1 if supported  @see AudioEffectX::sizeWindow
		audioMasterGetSampleRate,		///< [return value]: current sample rate  @see AudioEffectX::updateSampleRate
		audioMasterGetBlockSize,		///< [return value]: current block size  @see AudioEffectX::updateBlockSize
		audioMasterGetInputLatency,		///< [return value]: input latency in audio samples  @see AudioEffectX::getInputLatency
		audioMasterGetOutputLatency,	///< [return value]: output latency in audio samples  @see AudioEffectX::getOutputLatency

			audioMasterGetPreviousPlugDeprecated,			///< \deprecated deprecated in VST 2.4
			audioMasterGetNextPlugDeprecated,				///< \deprecated deprecated in VST 2.4
			audioMasterWillReplaceOrAccumulateDeprecated,	///< \deprecated deprecated in VST 2.4

		audioMasterGetCurrentProcessLevel,	///< [return value]: current process level  @see VstProcessLevels
		audioMasterGetAutomationState,		///< [return value]: current automation state  @see VstAutomationStates

		audioMasterOfflineStart,			///< [index]: numNewAudioFiles [value]: numAudioFiles [ptr]: #VstAudioFile*  @see AudioEffectX::offlineStart
		audioMasterOfflineRead,				///< [index]: bool readSource [value]: #VstOfflineOption* @see VstOfflineOption [ptr]: #VstOfflineTask*  @see VstOfflineTask @see AudioEffectX::offlineRead
		audioMasterOfflineWrite,			///< @see audioMasterOfflineRead @see AudioEffectX::offlineRead
		audioMasterOfflineGetCurrentPass,	///< @see AudioEffectX::offlineGetCurrentPass
		audioMasterOfflineGetCurrentMetaPass,	///< @see AudioEffectX::offlineGetCurrentMetaPass

			audioMasterSetOutputSampleRateDeprecated,			///< \deprecated deprecated in VST 2.4
			audioMasterGetOutputSpeakerArrangementDeprecated,	///< \deprecated deprecated in VST 2.4

		audioMasterGetVendorString,			///< [ptr]: char buffer for vendor string, limited to #kVstMaxVendorStrLen  @see AudioEffectX::getHostVendorString
		audioMasterGetProductString,		///< [ptr]: char buffer for vendor string, limited to #kVstMaxProductStrLen  @see AudioEffectX::getHostProductString
		audioMasterGetVendorVersion,		///< [return value]: vendor-specific version  @see AudioEffectX::getHostVendorVersion
		audioMasterVendorSpecific,			///< no definition, vendor specific handling  @see AudioEffectX::hostVendorSpecific
	
			audioMasterSetIconDeprecated,		///< \deprecated deprecated in VST 2.4
	
		audioMasterCanDo,					///< [ptr]: "can do" string [return value]: 1 for supported
		audioMasterGetLanguage,				///< [return value]: language code  @see VstHostLanguage

			audioMasterOpenWindowDeprecated,		///< \deprecated deprecated in VST 2.4
			audioMasterCloseWindowDeprecated,	///< \deprecated deprecated in VST 2.4

		audioMasterGetDirectory,			///< [return value]: FSSpec on MAC, else char*  @see AudioEffectX::getDirectory
		audioMasterUpdateDisplay,			///< no arguments	
		audioMasterBeginEdit,               ///< [index]: parameter index  @see AudioEffectX::beginEdit
		audioMasterEndEdit,                 ///< [index]: parameter index  @see AudioEffectX::endEdit
		audioMasterOpenFileSelector,		///< [ptr]: VstFileSelect* [return value]: 1 if supported  @see AudioEffectX::openFileSelector
		audioMasterCloseFileSelector,		///< [ptr]: VstFileSelect*  @see AudioEffectX::closeFileSelector
	
			audioMasterEditFileDeprecated,		///< \deprecated deprecated in VST 2.4
	
			audioMasterGetChunkFileDeprecated,	///< \deprecated deprecated in VST 2.4 [ptr]: char[2048] or sizeof (FSSpec) [return value]: 1 if supported  @see AudioEffectX::getChunkFile

			audioMasterGetInputSpeakerArrangementDeprecated,	///< \deprecated deprecated in VST 2.4
		
	//-------------------------------------------------------------------------------------------------------
	};

	//-------------------------------------------------------------------------------------------------------
	/** Parameter Properties used in #effGetParameterProperties. */
	//-------------------------------------------------------------------------------------------------------
	WIN32_DLL_STRUCTLAYOUT_ANSI public value struct VstParameterProperties
	{
	public:
		[System::FlagsAttribute]
		enum class VstParameterFlags: System::Int32
		{
		//-------------------------------------------------------------------------------------------------------
			kVstParameterIsSwitch				 = 1 << 0,	///< parameter is a switch (on/off)
			kVstParameterUsesIntegerMinMax		 = 1 << 1,	///< minInteger, maxInteger valid
			kVstParameterUsesFloatStep			 = 1 << 2,	///< stepFloat, smallStepFloat, largeStepFloat valid
			kVstParameterUsesIntStep			 = 1 << 3,	///< stepInteger, largeStepInteger valid
			kVstParameterSupportsDisplayIndex 	 = 1 << 4,	///< displayIndex valid
			kVstParameterSupportsDisplayCategory = 1 << 5,	///< category, etc. valid
			kVstParameterCanRamp				 = 1 << 6	///< set if parameter value can ramp up/down
		//-------------------------------------------------------------------------------------------------------
		};

	public:
	//-------------------------------------------------------------------------------------------------------
		System::Single stepFloat;			//< System::Single step
		System::Single smallStepFloat;		//< small System::Single step
		System::Single largeStepFloat;		//< large System::Single step
		[InteropServices::MarshalAs(InteropServices::UnmanagedType::ByValTStr, SizeConst=safe_cast<System::Int32>(VstStringConstants::kVstMaxLabelLen))]
			System::String^		label;		//< parameter label
		VstParameterFlags flags;				//< @see VstParameterFlags
		System::Int32 minInteger;		//< integer minimum
		System::Int32 maxInteger;		//< integer maximum
		System::Int32 stepInteger;		//< integer step
		System::Int32 largeStepInteger;	//< large integer step
		[InteropServices::MarshalAs(InteropServices::UnmanagedType::ByValTStr, SizeConst=safe_cast<System::Int32>(VstStringConstants::kVstMaxShortLabelLen))]
			System::String^		shortLabel;		//< short label, recommended: 6 + delimiter

		// The following are for remote controller display purposes.
		// Note that the kVstParameterSupportsDisplayIndex flag must be set.
		// Host can scan all parameters, and find out in what order
		// to display them:

		System::Int16 displayIndex;		//< index where this parameter should be displayed (starting with 0)

		// Host can also possibly display the parameter group (category), such as...
		// ---------------------------
		// Osc 1
		// Wave  Detune  Octave  Mod
		// ---------------------------
		// ...if the plug-in supports it (flag #kVstParameterSupportsDisplayCategory)

		System::Int16 category;			//< 0: no category, else group index + 1
		System::Int16 numParametersInCategory;			//< number of parameters in category
		System::Int16 reserved;			//< zero
		[InteropServices::MarshalAs(InteropServices::UnmanagedType::ByValTStr, SizeConst=safe_cast<System::Int32>(VstStringConstants::kVstMaxCategLabelLen))]
			System::String^		categoryLabel;		//< category label, e.g. "Osc 1" 

		[InteropServices::MarshalAs(InteropServices::UnmanagedType::ByValArray, SizeConst=16)]
			array<System::Char>^ future;	//< reserved for future use
	//-------------------------------------------------------------------------------------------------------
	};


	//-------------------------------------------------------------------------------------------------------
	/** Speaker Arrangement Types*/
	//-------------------------------------------------------------------------------------------------------
	public enum class VstSpeakerArrangementType: System::Int32
	{
	//-------------------------------------------------------------------------------------------------------
		kSpeakerArrUserDefined = -2,//< user defined
		kSpeakerArrEmpty = -1,		//< empty arrangement
		kSpeakerArrMono  =  0,		//< M
		kSpeakerArrStereo,			//< L R
		kSpeakerArrStereoSurround,	//< Ls Rs
		kSpeakerArrStereoCenter,	//< Lc Rc
		kSpeakerArrStereoSide,		//< Sl Sr
		kSpeakerArrStereoCLfe,		//< C Lfe
		kSpeakerArr30Cine,			//< L R C
		kSpeakerArr30Music,			//< L R S
		kSpeakerArr31Cine,			//< L R C Lfe
		kSpeakerArr31Music,			//< L R Lfe S
		kSpeakerArr40Cine,			//< L R C   S (LCRS)
		kSpeakerArr40Music,			//< L R Ls  Rs (Quadro)
		kSpeakerArr41Cine,			//< L R C   Lfe S (LCRS+Lfe)
		kSpeakerArr41Music,			//< L R Lfe Ls Rs (Quadro+Lfe)
		kSpeakerArr50,				//< L R C Ls  Rs 
		kSpeakerArr51,				//< L R C Lfe Ls Rs
		kSpeakerArr60Cine,			//< L R C   Ls  Rs Cs
		kSpeakerArr60Music,			//< L R Ls  Rs  Sl Sr 
		kSpeakerArr61Cine,			//< L R C   Lfe Ls Rs Cs
		kSpeakerArr61Music,			//< L R Lfe Ls  Rs Sl Sr 
		kSpeakerArr70Cine,			//< L R C Ls  Rs Lc Rc 
		kSpeakerArr70Music,			//< L R C Ls  Rs Sl Sr
		kSpeakerArr71Cine,			//< L R C Lfe Ls Rs Lc Rc
		kSpeakerArr71Music,			//< L R C Lfe Ls Rs Sl Sr
		kSpeakerArr80Cine,			//< L R C Ls  Rs Lc Rc Cs
		kSpeakerArr80Music,			//< L R C Ls  Rs Cs Sl Sr
		kSpeakerArr81Cine,			//< L R C Lfe Ls Rs Lc Rc Cs
		kSpeakerArr81Music,			//< L R C Lfe Ls Rs Cs Sl Sr 
		kSpeakerArr102,				//< L R C Lfe Ls Rs Tfl Tfc Tfr Trl Trr Lfe2
		kNumSpeakerArr
	//-------------------------------------------------------------------------------------------------------
	};

	//-------------------------------------------------------------------------------------------------------
	/** Pin Properties used in #effGetInputProperties and #effGetOutputProperties. */
	//-------------------------------------------------------------------------------------------------------
	WIN32_DLL_STRUCTLAYOUT_ANSI public value struct VstPinProperties
	{
	public:
		[System::FlagsAttribute]
		enum class VstPinPropertiesFlags: System::Int32
		{
		//-------------------------------------------------------------------------------------------------------
			kVstPinIsActive   = 1 << 0,		//< pin is active, ignored by Host
			kVstPinIsStereo   = 1 << 1,		//< pin is first of a stereo pair
			kVstPinUseSpeaker = 1 << 2		//< #VstPinProperties::arrangementType is valid and can be used to get the wanted arrangement
		//-------------------------------------------------------------------------------------------------------
		};
	
	//-------------------------------------------------------------------------------------------------------
	public:
		[InteropServices::MarshalAs(InteropServices::UnmanagedType::ByValTStr, SizeConst=safe_cast<System::Int32>(VstStringConstants::kVstMaxLabelLen))]
			System::String^			label;				//< pin name
		VstPinPropertiesFlags		flags;				//< @see VstPinPropertiesFlags
		VstSpeakerArrangementType	arrangementType;	//< @see VstSpeakerArrangementType
		[InteropServices::MarshalAs(InteropServices::UnmanagedType::ByValTStr, SizeConst=safe_cast<System::Int32>(VstStringConstants::kVstMaxShortLabelLen))]
			System::String^			shortLabel;			//< short name (recommended: 6 + delimiter)

		[InteropServices::MarshalAs(InteropServices::UnmanagedType::ByValArray, SizeConst=48)]
			array<System::Char>^	future;				//< reserved for future use
	//-------------------------------------------------------------------------------------------------------
	};

	//-------------------------------------------------------------------------------------------------------
	// VstEvent
	//-------------------------------------------------------------------------------------------------------
	//-------------------------------------------------------------------------------------------------------
	/** A generic timestamped event. */
	//-------------------------------------------------------------------------------------------------------
	WIN32_DLL_STRUCTLAYOUT_UNION_ANSI
	//WIN32_DLL_STRUCTLAYOUT_ANSI
	public value struct VstEvent
	{
	public:
		//-------------------------------------------------------------------------------------------------------
		/** VstEvent Types used by #VstEvent. */
		//-------------------------------------------------------------------------------------------------------
		enum class VstEventTypes: System::Int32
		{
		//-------------------------------------------------------------------------------------------------------
			kVstMidiType = 1,					//< MIDI event  @see VstMidiEvent
				kVstAudioTypeDeprecated,		//< \deprecated unused event type
				kVstVideoTypeDeprecated,		//< \deprecated unused event type
				kVstParameterTypeDeprecated,	//< \deprecated unused event type
				kVstTriggerTypeDeprecated,		//< \deprecated unused event type
			kVstSysExType						//< MIDI system exclusive  @see VstMidiSysexEvent
		//-------------------------------------------------------------------------------------------------------
		};

		//-------------------------------------------------------------------------------------------------------
		/** Flags used in #VstMidiEvent. */
		//-------------------------------------------------------------------------------------------------------
		enum class VstMidiEventFlags: System::Int32
		{
		//-------------------------------------------------------------------------------------------------------
			kVstMidiEventIsRealtime = 1 << 0	//< means that this event is played life (not in playback from a sequencer track).\n This allows the Plug-In to handle these flagged events with higher priority, especially when the Plug-In has a big latency (AEffect::initialDelay)
		//-------------------------------------------------------------------------------------------------------
		};

	//-------------------------------------------------------------------------------------------------------
	public:
/*		VstEvent()
		{
			this->byteSize = 
				InteropServices::Marshal::SizeOf(this)
				- (
					4 //InteropServices::Marshal::SizeOf(this->type)
					+ 4 //InteropServices::Marshal::SizeOf(this->byteSize)
				);
		}
*/
	public:
		[InteropServices::FieldOffset(0)]	VstEventTypes type;			//< @see VstEventTypes
		[InteropServices::FieldOffset(4)]	System::Int32 byteSize;		//< size of this event, excl. type and byteSize
		[InteropServices::FieldOffset(8)]	System::Int32 deltaFrames;	//< sample frames related to the current block start sample position
		[InteropServices::FieldOffset(12)]	System::Int32 _flags;		//< generic flags, none defined yet

		//** VstMidiEvent **
		[InteropServices::FieldOffset(12)]	VstMidiEventFlags	flags;		//< @see VstMidiEventFlags
		[InteropServices::FieldOffset(16)]	System::Int32	noteLength;		//< (in sample frames) of entire note, if available, else 0
		[InteropServices::FieldOffset(20)]	System::Int32	noteOffset;		//< offset (in sample frames) into note from note start if available, else 0
		[InteropServices::FieldOffset(24)]	System::Int32	midiData;		//< 1 to 3 MIDI bytes; midiData[3] is reserved (zero)
		[InteropServices::FieldOffset(24)]	System::Byte	midiData0;		//< 1 to 3 MIDI bytes; midiData[3] is reserved (zero)
		[InteropServices::FieldOffset(25)]	System::Byte	midiData1;
		[InteropServices::FieldOffset(26)]	System::Byte	midiData2;
		[InteropServices::FieldOffset(27)]	System::Byte	midiData3;
		[InteropServices::FieldOffset(28)]	System::Byte	detune;			//< -64 to +63 cents; for scales other than 'well-tempered' ('microtuning')
		[InteropServices::FieldOffset(29)]	System::Byte	noteOffVelocity;//< Note Off Velocity [0, 127]
		[InteropServices::FieldOffset(30)]	System::Byte	reserved1;		//< zero (Reserved for future use)
		[InteropServices::FieldOffset(31)]	System::Byte	reserved2;		//< zero (Reserved for future use)

		//** VstMidiSysexEvent ** //TODO: 64bitに対応してない
		[InteropServices::FieldOffset(16)]	System::Int32	dumpBytes;	//< byte size of sysexDump
		[InteropServices::FieldOffset(20)]	System::IntPtr	resvd1;		//< zero (Reserved for future use)
		[InteropServices::FieldOffset(24)]	System::IntPtr	sysexDump;	//< sysex dump
		[InteropServices::FieldOffset(28)]	System::IntPtr	resvd2;		//< zero (Reserved for future use)
	//-------------------------------------------------------------------------------------------------------

		virtual System::String^ ToString() override
		{
			if (type == VstEventTypes::kVstMidiType) {
				return 
					"type["+type.ToString("F")+"] byteSize["+byteSize+"] deltaFrames["+deltaFrames+"] flags["+flags.ToString("F")+"] noteLength["+noteLength+" noteOffset["+noteOffset+"] midiData["+midiData.ToString("X8")+":"+midiData0.ToString("X2")+":"+midiData1.ToString("X2")+":"+midiData2.ToString("X2")+":"+midiData3.ToString("X2")+"] detune["+detune+"] noteOffVelocity["+noteOffVelocity+"]";
			} else if (type == VstEventTypes::kVstSysExType) {
				return 
					"type["+type.ToString("F")+"] byteSize["+byteSize+"] deltaFrames["+deltaFrames+"] _flags["+_flags+"] dumpBytes["+dumpBytes+" sysexDump["+sysexDump+"]";
			} else {
				return 
					"type["+type.ToString("F")+"] byteSize["+byteSize+"] deltaFrames["+deltaFrames+"] _flags["+_flags+"]";
			}
		}
	};


	//-------------------------------------------------------------------------------------------------------
	/** A block of events for the current processed audio block. */
	//-------------------------------------------------------------------------------------------------------
	WIN32_DLL_STRUCTLAYOUT_ANSI public value struct VstEvents
	{
	//-------------------------------------------------------------------------------------------------------
	public:
		System::Int32	numEvents;		//< number of Events in array
		System::IntPtr	reserved;		//< zero (Reserved for future use)
		//System::IntPtr	events;			//VstEvent* events[2];	///< event pointer array, variable size
	//-------------------------------------------------------------------------------------------------------
	};

	//-------------------------------------------------------------------------------------------------------
	// VstTimeInfo
	//-------------------------------------------------------------------------------------------------------
	//-------------------------------------------------------------------------------------------------------
	/** VstTimeInfo requested via #audioMasterGetTime.  @see AudioEffectX::getTimeInfo 

	\note VstTimeInfo::samplePos :Current Position. It must always be valid, and should not cost a lot to ask for. The sample position is ahead of the time displayed to the user. In sequencer stop mode, its value does not change. A 32 bit integer is too small for sample positions, and it's a double to make it easier to convert between ppq and samples.
	\note VstTimeInfo::ppqPos : At tempo 120, 1 quarter makes 1/2 second, so 2.0 ppq translates to 48000 samples at 48kHz sample rate.
	.25 ppq is one sixteenth note then. if you need something like 480ppq, you simply multiply ppq by that scaler.
	\note VstTimeInfo::barStartPos : Say we're at bars/beats readout 3.3.3. That's 2 bars + 2 q + 2 sixteenth, makes 2 * 4 + 2 + .25 = 10.25 ppq. at tempo 120, that's 10.25 * .5 = 5.125 seconds, times 48000 = 246000 samples (if my calculator servers me well :-). 
	\note VstTimeInfo::samplesToNextClock : MIDI Clock Resolution (24 per Quarter Note), can be negative the distance to the next midi clock (24 ppq, pulses per quarter) in samples. unless samplePos falls precicely on a midi clock, this will either be negative such that the previous MIDI clock is addressed, or positive when referencing the following (future) MIDI clock.
	*/
	//-------------------------------------------------------------------------------------------------------
	WIN32_DLL_STRUCTLAYOUT_ANSI public value struct VstTimeInfo
	{
	public:
		//-------------------------------------------------------------------------------------------------------
		/** Flags used in #VstTimeInfo. */
		//-------------------------------------------------------------------------------------------------------
		[System::FlagsAttribute]
		enum class VstTimeInfoFlags: System::Int32
		{
		//-------------------------------------------------------------------------------------------------------
			kVstTransportChanged     = 1,		///< indicates that play, cycle or record state has changed
			kVstTransportPlaying     = 1 << 1,	///< set if Host sequencer is currently playing
			kVstTransportCycleActive = 1 << 2,	///< set if Host sequencer is in cycle mode
			kVstTransportRecording   = 1 << 3,	///< set if Host sequencer is in record mode
			kVstAutomationWriting    = 1 << 6,	///< set if automation write mode active (record parameter changes)
			kVstAutomationReading    = 1 << 7,	///< set if automation read mode active (play parameter changes)
			kVstNanosValid           = 1 << 8,	///< VstTimeInfo::nanoSeconds valid
			kVstPpqPosValid          = 1 << 9,	///< VstTimeInfo::ppqPos valid
			kVstTempoValid           = 1 << 10,	///< VstTimeInfo::tempo valid
			kVstBarsValid            = 1 << 11,	///< VstTimeInfo::barStartPos valid
			kVstCyclePosValid        = 1 << 12,	///< VstTimeInfo::cycleStartPos and VstTimeInfo::cycleEndPos valid
			kVstTimeSigValid         = 1 << 13,	///< VstTimeInfo::timeSigNumerator and VstTimeInfo::timeSigDenominator valid
			kVstSmpteValid           = 1 << 14,	///< VstTimeInfo::smpteOffset and VstTimeInfo::smpteFrameRate valid
			kVstClockValid           = 1 << 15	///< VstTimeInfo::samplesToNextClock valid
		//-------------------------------------------------------------------------------------------------------
		};

		//-------------------------------------------------------------------------------------------------------
		/** SMPTE Frame Rates. */
		//-------------------------------------------------------------------------------------------------------
		enum class VstSmpteFrameRate: System::Int32
		{
		//-------------------------------------------------------------------------------------------------------
			kVstSmpte24fps    = 0,		///< 24 fps
			kVstSmpte25fps    = 1,		///< 25 fps
			kVstSmpte2997fps  = 2,		///< 29.97 fps
			kVstSmpte30fps    = 3,		///< 30 fps
			kVstSmpte2997dfps = 4,		///< 29.97 drop
			kVstSmpte30dfps   = 5,		///< 30 drop

			kVstSmpteFilm16mm = 6, 		///< Film 16mm
			kVstSmpteFilm35mm = 7, 		///< Film 35mm
			kVstSmpte239fps   = 10,		///< HDTV: 23.976 fps
			kVstSmpte249fps   = 11,		///< HDTV: 24.976 fps
			kVstSmpte599fps   = 12,		///< HDTV: 59.94 fps
			kVstSmpte60fps    = 13		///< HDTV: 60 fps
		//-------------------------------------------------------------------------------------------------------
		};

	//-------------------------------------------------------------------------------------------------------
	public:
		System::Double samplePos;				///< current Position in audio samples (always valid)
		System::Double sampleRate;				///< current Sample Rate in Herz (always valid)
		System::Double nanoSeconds;				///< System Time in nanoseconds (10^-9 second)
		System::Double ppqPos;					///< Musical Position, in Quarter Note (1.0 equals 1 Quarter Note)
		System::Double tempo;					///< current Tempo in BPM (Beats Per Minute)
		System::Double barStartPos;				///< last Bar Start Position, in Quarter Note
		System::Double cycleStartPos;			///< Cycle Start (left locator), in Quarter Note
		System::Double cycleEndPos;				///< Cycle End (right locator), in Quarter Note
		System::Int32 timeSigNumerator;		///< Time Signature Numerator (e.g. 3 for 3/4)
		System::Int32 timeSigDenominator;	///< Time Signature Denominator (e.g. 4 for 3/4)
		System::Int32 smpteOffset;			///< SMPTE offset (in SMPTE subframes (bits; 1/80 of a frame)). The current SMPTE position can be calculated using #samplePos, #sampleRate, and #smpteFrameRate.
		System::Int32 smpteFrameRate;		///< @see VstSmpteFrameRate
		System::Int32 samplesToNextClock;	///< MIDI Clock Resolution (24 Per Quarter Note), can be negative (nearest clock)
		VstTimeInfoFlags flags;					///< @see VstTimeInfoFlags
	//-------------------------------------------------------------------------------------------------------
	};


	//-------------------------------------------------------------------------------------------------------
	/** Variable IO for Offline Processing. */
	//-------------------------------------------------------------------------------------------------------
	WIN32_DLL_STRUCTLAYOUT_ANSI public value struct VstVariableIo
	{
	//-------------------------------------------------------------------------------------------------------
		System::IntPtr	inputs;						//< float** input audio buffers
		System::IntPtr	outputs;					//< float** output audio buffers
		System::Int32	numSamplesInput;			//< number of incoming samples
		System::Int32	numSamplesOutput;			//< number of outgoing samples
		System::IntPtr	numSamplesInputProcessed;	//< System::Int32* number of samples actually processed of input
		System::IntPtr	numSamplesOutputProcessed;	//< System::Int32* number of samples actually processed of output
	//-------------------------------------------------------------------------------------------------------
	};

	public value class AudioMasterCallBack
	{
	public:
		ref class AudioMasterEventArgs: System::EventArgs
		{
		public:
			System::IntPtr/*AEffect^*/		effect;
			AudioMasterOpcodes	opcode;
			System::Int32		index;
			System::IntPtr		value;
			System::IntPtr		ptr;
			System::Single		opt;
			System::IntPtr		returnValue;
		};

		VST_DLL_DELEGATE(Ansi) delegate System::IntPtr Delegate(
			[InteropServices::In]System::IntPtr/*AEffect^*/		effect,
			[InteropServices::In]AudioMasterOpcodes	opcode,
			[InteropServices::In]System::Int32		index,
			[InteropServices::In]System::IntPtr		value,
			[InteropServices::In]System::IntPtr		ptr,
			[InteropServices::In]System::Single		opt
		);
	};


	VST_DLL_DELEGATE(Ansi) public delegate System::IntPtr AEffectDispatcherProc(
		[InteropServices::In]System::IntPtr/*AEffect^*/		effect,
		[InteropServices::In]AEffectOpcodes	opcode,
		[InteropServices::In]System::Int32	index,
		[InteropServices::In]System::IntPtr	value,
		[InteropServices::In]System::IntPtr	ptr,
		[InteropServices::In]System::Single	opt
	);

	VST_DLL_DELEGATE(Ansi) public delegate void AEffectProcessProc(
		[InteropServices::In]System::IntPtr/*AEffect^*/		effect,
		[InteropServices::In]System::IntPtr	inputs,
		[InteropServices::In]System::IntPtr	outputs,
		[InteropServices::In]System::Int32	sampleFrames
	);

	VST_DLL_DELEGATE(Ansi) public delegate void AEffectProcessDoubleProc(
		[InteropServices::In]System::IntPtr/*AEffect^*/		effect,
		[InteropServices::In]System::IntPtr	inputs,
		[InteropServices::In]System::IntPtr	outputs,
		[InteropServices::In]System::Int32	sampleFrames
	);

	VST_DLL_DELEGATE(Ansi) public delegate void AEffectSetParameterProc(
		[InteropServices::In]System::IntPtr/*AEffect^*/		effect,
		[InteropServices::In]System::Int32	index,
		[InteropServices::In]System::Single	parameter
	);

	VST_DLL_DELEGATE(Ansi) public delegate System::Single AEffectGetParameterProc(
		[InteropServices::In]System::IntPtr/*AEffect^*/		effect,
		[InteropServices::In]System::Int32	index
	);

	VST_DLL_DELEGATE(Ansi) public delegate System::IntPtr VSTPluginMain(
		[InteropServices::In][InteropServices::MarshalAs(InteropServices::UnmanagedType::FunctionPtr)]
			AudioMasterCallBack::Delegate^	audioMaster
	);


}
}
}
