﻿/*
[momiji music component library]
---------------------------------------------------------------------
Momiji.Core.Midi.In.cpp
	midi input component.
---------------------------------------------------------------------
Copyright (C) 2011 tyiki badwell {miria@users.sourceforge.jp}.

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/gpl-3.0.html>.
---------------------------------------------------------------------
*/
#include "StdAfx.h"

#include "Momiji.Interop.Winmm.h"
#include "Momiji.Core.Midi.In.h"
#include "Momiji.Core.Midi.Data.h"

namespace Momiji {
namespace Core {
namespace Midi {
namespace In {

	void Device::OnEventHandler(
		System::Object^ sender,
		Interop::Winmm::DriverCallBack::DriverEventArgs^ args
	)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1}] start", __FUNCTION__, System::Threading::Thread::CurrentThread->GetHashCode());
		#endif

		switch (args->uMsg)
		{
		case Interop::Winmm::DriverCallBack::MM_EXT_WINDOW_MESSAGE::MIM_OPEN:
			{
				this->DoOpen(args->dw1, args->dw2);
				break;
			}
		case Interop::Winmm::DriverCallBack::MM_EXT_WINDOW_MESSAGE::MIM_CLOSE:
			{
				this->DoClose(args->dw1, args->dw2);
				break;
			}
		case Interop::Winmm::DriverCallBack::MM_EXT_WINDOW_MESSAGE::MIM_DATA:
			{
				this->DoData(args->dw1, args->dw2);
				break;
			}
		case Interop::Winmm::DriverCallBack::MM_EXT_WINDOW_MESSAGE::MIM_LONGDATA:
			{
				this->DoLongData(args->dw1, args->dw2);
				break;
			}
		case Interop::Winmm::DriverCallBack::MM_EXT_WINDOW_MESSAGE::MIM_ERROR:
			{
				this->DoError(args->dw1, args->dw2);
				break;
			}
		case Interop::Winmm::DriverCallBack::MM_EXT_WINDOW_MESSAGE::MIM_LONGERROR:
			{
				this->DoLongError(args->dw1, args->dw2);
				break;
			}
		case Interop::Winmm::DriverCallBack::MM_EXT_WINDOW_MESSAGE::MIM_MOREDATA:
			{
				this->DoMoreData(args->dw1, args->dw2);
				break;
			}
		}
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1}] end", __FUNCTION__, System::Threading::Thread::CurrentThread->GetHashCode());
		#endif
	}

	void Device::DoOpen(System::IntPtr dwParam1, System::IntPtr dwParam2)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1,8:X}][{2,8:X}]", __FUNCTION__, dwParam1, dwParam2);
		#endif
		this->OnOpen(this, System::EventArgs::Empty);
	}

	void Device::DoClose(System::IntPtr dwParam1, System::IntPtr dwParam2)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1,8:X}][{2,8:X}]", __FUNCTION__, dwParam1, dwParam2);
		#endif
		this->OnClose(this, System::EventArgs::Empty);
	}

	///<summary>～</summary>
	///<remarks>～</remarks>
	///<param>～</param>
	///
	void Device::DoData(System::IntPtr dwParam1, System::IntPtr dwParam2)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1,8:X}][{2,8:X}]", __FUNCTION__, dwParam1, dwParam2);
		#endif
		auto data = dwParam1.ToInt32();
		auto time = dwParam2.ToInt32();
		this->OnData(data, time);
	}

	void Device::DoLongData(System::IntPtr dwParam1, System::IntPtr dwParam2)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1,8:X}][{2,8:X}]", __FUNCTION__, dwParam1, dwParam2);
		#endif

		this->Unprepare(dwParam1);

		auto header = safe_cast<Interop::Winmm::MidiHeader^>(InteropServices::Marshal::PtrToStructure(dwParam1, Interop::Winmm::MidiHeader::typeid));

		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] bytesRecorded {1}", __FUNCTION__, header->bytesRecorded);
		#endif

		if (header->bytesRecorded == 0)
		{
			#ifdef _DEBUG
			System::Console::WriteLine("[{0}] resetされたと判断して、再準備はしない", __FUNCTION__);
			#endif
		}
		else
		{
			auto data = gcnew array<System::Byte>(header->bytesRecorded);

			#ifdef _DEBUG
				System::Console::Write("[{0}] ", __FUNCTION__);
				for (System::UInt32 i = 0; i < header->bytesRecorded; i++)
				{
					System::Console::Write("[{0,2:X}]", InteropServices::Marshal::ReadByte(header->data,i));
				}
				System::Console::WriteLine();
			#endif

			InteropServices::Marshal::Copy( header->data, data, 0, data->Length );
			auto time = dwParam2.ToInt32();
			this->OnLongData(data, time);
			this->Prepare();
		}
	}

	void Device::DoError(System::IntPtr dwParam1, System::IntPtr dwParam2)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1,8:X}][{2,8:X}]", __FUNCTION__, dwParam1, dwParam2);
		#endif
	}

	void Device::DoLongError(System::IntPtr dwParam1, System::IntPtr dwParam2)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1,8:X}][{2,8:X}]", __FUNCTION__, dwParam1, dwParam2);
		#endif

		this->Unprepare(dwParam1);
	}

	void Device::DoMoreData(System::IntPtr dwParam1, System::IntPtr dwParam2)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1,8:X}][{2,8:X}]", __FUNCTION__, dwParam1, dwParam2);
		#endif

		this->Unprepare(dwParam1);

		auto header = safe_cast<Interop::Winmm::MidiHeader^>(InteropServices::Marshal::PtrToStructure(dwParam1, Interop::Winmm::MidiHeader::typeid));

		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] bytesRecorded {1}", __FUNCTION__, header->bytesRecorded);
		#endif

		auto data = gcnew array<System::Byte>(header->bytesRecorded);

		#ifdef _DEBUG
			System::Console::Write("[{0}] ", __FUNCTION__);
			for (System::UInt32 i = 0; i < header->bytesRecorded; i++)
			{
				System::Console::Write("[{0,2:X}]", InteropServices::Marshal::ReadByte(header->data,i));
			}
			System::Console::WriteLine();
		#endif

		InteropServices::Marshal::Copy( header->data, data, 0, data->Length );

		auto time = dwParam2.ToInt32();
		this->OnMoreData(data, time);
		this->Prepare();
	}

	System::UInt32 Device::GetNumDevices()
	{
		return Interop::Winmm::Function::midiInGetNumDevs();
	}

	Interop::Winmm::MidiInCapabilities^ Device::GetCapabilities(
		const System::UInt32 deviceID
	)
	{
		auto caps = gcnew Interop::Winmm::MidiInCapabilities();
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] caps size {1}",__FUNCTION__, InteropServices::Marshal::SizeOf(caps));
		#endif
		auto mmResult = 
			Interop::Winmm::Function::midiInGetDevCaps(
				deviceID, 
				caps, 
				InteropServices::Marshal::SizeOf(caps)
			);
		if (mmResult != Interop::Winmm::MMRESULT::NOERROR)
		{
			throw gcnew MidiInException(mmResult);
		}
		return caps;
	}

	Device::Device(
		const System::UInt32 deviceID
	):	_deviceID(deviceID)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] deviceID {1}",__FUNCTION__, this->_deviceID);
		#endif
		this->Open();
	}

	Device::~Device()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		this->!Device();
	}

	Device::!Device()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		this->Close();
	}

	Interop::Winmm::MidiInCapabilities^ Device::GetCapabilities()
	{
		return Device::GetCapabilities(this->_deviceID);
	}

	void Device::Open()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		this->_callBack = gcnew Core::Winmm::DriverCallBack(false); //イベントは同期で動かす
		this->_callBack->OnEvent += gcnew System::EventHandler<Interop::Winmm::DriverCallBack::DriverEventArgs^>(this, &Device::OnEventHandler);

		this->_headerPool =
			gcnew Core::Buffer::BufferPool<Interop::Winmm::MidiHeader^>(
				2,		//２回分のバッファを用意
				gcnew Core::Buffer::BufferPool<Interop::Winmm::MidiHeader^>::Allocator(this, &Device::AllocateHeader) 
			);

		this->_bufferPool =
			gcnew Core::Buffer::BufferPool<array<System::Byte>^>(
				2,		//２回分のバッファを用意
				gcnew Core::Buffer::BufferPool<array<System::Byte>^>::Allocator(this, &Device::AllocateBuffer) 
			);

		{
			auto mmResult = 
				Interop::Winmm::Function::midiInOpen(
					this->_handle,
					this->_deviceID,
					this->_callBack->GetDriverCallBackProc(),
					System::IntPtr::Zero,
					Interop::Winmm::DriverCallBack::TYPE::FUNCTION
				);
			if (mmResult != Interop::Winmm::MMRESULT::NOERROR) {
				throw gcnew MidiInException(mmResult);
			}
		}

		this->Prepare();

		{
			auto mmResult = Interop::Winmm::Function::midiInStart(this->_handle);
			if (mmResult != Interop::Winmm::MMRESULT::NOERROR)
			{
				throw gcnew MidiInException(mmResult);
			}
		}

		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] Handle invalid:{1} closed:{2}",__FUNCTION__, this->_handle->IsInvalid, this->_handle->IsClosed);
		#endif
	}

	void Device::Close()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] Handle invalid:{1} closed:{2}",__FUNCTION__, this->_handle->IsInvalid, this->_handle->IsClosed);
		#endif
		if (
			!this->_handle->IsInvalid
		&&	!this->_handle->IsClosed
		)
		{
			{
				auto mmResult = Interop::Winmm::Function::midiInStop(this->_handle);
				if (mmResult != Interop::Winmm::MMRESULT::NOERROR)
				{
					throw gcnew MidiInException(mmResult);
				}
				#ifdef _DEBUG
					System::Console::WriteLine("[{0}] midiInStop OK",__FUNCTION__);
				#endif
			}

			{
				auto mmResult = Interop::Winmm::Function::midiInReset(this->_handle);
				if (mmResult != Interop::Winmm::MMRESULT::NOERROR)
				{
					throw gcnew MidiInException(mmResult);
				}
				#ifdef _DEBUG
					System::Console::WriteLine("[{0}] midiInReset OK",__FUNCTION__);
				#endif
			}

			//バッファの開放待ち
			while(this->_headerPool->IsBusy())
			{
				#ifdef _DEBUG
					System::Console::WriteLine("[{0}] wait for unprepare headers ...",__FUNCTION__);
				#endif
				System::Threading::Thread::Sleep(5);
			}

			this->_handle->Close();
		}
		else
		{
			#ifdef _DEBUG
				System::Console::WriteLine("[{0}] openしていない状態なので、無視します。", __FUNCTION__);
			#endif
		}

		if (this->_callBack != nullptr)
		{
			this->_callBack->OnEvent -= gcnew System::EventHandler<Interop::Winmm::DriverCallBack::DriverEventArgs^>(this, &Device::OnEventHandler);
			delete this->_callBack;
		}

		if (this->_headerPool != nullptr)
		{
			delete this->_headerPool;
		}

		if (this->_bufferPool != nullptr)
		{
			delete this->_bufferPool;
		}
	}

	System::IntPtr Device::Prepare()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		auto header = this->_headerPool->Get();
		auto buffer = this->_bufferPool->Get();

		{
			auto midiHeader = header->GetBuffer();
			midiHeader->bufferLength = buffer->GetBuffer()->Length;
			midiHeader->data = buffer->GetBufferIntPtr();

			#ifdef _DEBUG
				{
					System::Console::WriteLine("[{0}] midiInPrepareHeader before", __FUNCTION__);
					System::Console::WriteLine("[{0}] {1}", __FUNCTION__, midiHeader);
				}
			#endif
		}

		{
			auto mmResult = 
				Interop::Winmm::Function::midiInPrepareHeader(
					this->_handle,
					header->GetBufferIntPtr(),
					InteropServices::Marshal::SizeOf(Interop::Winmm::MidiHeader::typeid)
				);
			if (mmResult != Interop::Winmm::MMRESULT::NOERROR)
			{
				this->_headerPool->Release(header);
				this->_bufferPool->Release(buffer);

				throw gcnew MidiInException(mmResult);
			}
		}
		#ifdef _DEBUG
			{
				auto midiHeader = header->GetBuffer();
				System::Console::WriteLine("[{0}] midiInPrepareHeader after", __FUNCTION__);
				System::Console::WriteLine("[{0}] {1}", __FUNCTION__, midiHeader);
			}
		#endif

		{
			auto mmResult =
				Interop::Winmm::Function::midiInAddBuffer(
					this->_handle,
					header->GetBufferIntPtr(),
					InteropServices::Marshal::SizeOf(Interop::Winmm::MidiHeader::typeid)
				);
			if (mmResult != Interop::Winmm::MMRESULT::NOERROR)
			{
				System::Console::WriteLine((gcnew MidiInException(mmResult))->ToString());
			}
		}
		#ifdef _DEBUG
			{
				auto midiHeader = header->GetBuffer();
				System::Console::WriteLine("[{0}] midiInAddBuffer after", __FUNCTION__);
				System::Console::WriteLine("[{0}] {1}", __FUNCTION__, midiHeader);
			}
		#endif

		return header->GetBufferIntPtr();
	}

	void Device::Unprepare(System::IntPtr headerPtr)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		auto header = this->_headerPool->GetBusy(headerPtr);
		
		#ifdef _DEBUG
			{
				auto midiHeader = header->GetBuffer();
				System::Console::WriteLine("[{0}] midiInUnprepareHeader before", __FUNCTION__);
				System::Console::WriteLine("[{0}] {1}", __FUNCTION__, midiHeader);
			}
		#endif
			
		auto mmResult = 
			Interop::Winmm::Function::midiInUnprepareHeader(
				this->_handle,
				headerPtr,
				InteropServices::Marshal::SizeOf(Interop::Winmm::MidiHeader::typeid)
			);
		if (mmResult != Interop::Winmm::MMRESULT::NOERROR)
		{
			throw gcnew MidiInException(mmResult);
		}

		auto bufferPtr = header->GetBuffer()->data;
		this->_headerPool->Release(header);

		auto buffer = this->_bufferPool->GetBusy(bufferPtr);
		this->_bufferPool->Release(buffer);

		#ifdef _DEBUG
			{
				auto midiHeader = header->GetBuffer();
				System::Console::WriteLine("[{0}] midiInUnprepareHeader after", __FUNCTION__);
				System::Console::WriteLine("[{0}] {1}", __FUNCTION__, midiHeader);
			}
		#endif
	}

	Interop::Winmm::MidiHeader^ Device::AllocateHeader()
	{
		return gcnew Interop::Winmm::MidiHeader();
	}

	array<System::Byte>^ Device::AllocateBuffer()
	{
		return gcnew array<System::Byte>(256); //SMFとして最大長 + 1
	}

	System::String^ MidiInException::Initialize()
	{
		auto errorMessage = System::String::Empty;
		auto buf = gcnew System::Text::StringBuilder(256);

		auto r = 
			Interop::Winmm::Function::midiInGetErrorText(
				this->_mmResult, 
				buf, 
				buf->Capacity
			);
		if (r == Interop::Winmm::MMRESULT::NOERROR)
		{
			errorMessage = buf->ToString();
		}
		else
		{
			errorMessage = "不明なエラー";
		}

		return errorMessage + "[" + this->_mmResult.ToString("D") + "]";
	}

}
}
}
}
