#
# Cookbook Name:: jenkins-grid
# Recipe:: docker-compose
#
# Copyright 2016-2017, whitestar
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

doc_url = 'https://hub.docker.com/_/jenkins/'

include_recipe 'docker-grid::compose'

config = node['jenkins-grid']['docker-compose']['config']
override_config = node.override['jenkins-grid']['docker-compose']['config']
force_override_config = node.force_override['jenkins-grid']['docker-compose']['config']
app_dir = node['jenkins-grid']['docker-compose']['app_dir']
certs_dir = "#{app_dir}/certs"
groovy_dir = "#{app_dir}/ref/init.groovy.d"

envs = {}
java_opts = []
jenkins_opts = []
vols = config['services']['jenkins']['volumes'].to_a

[
  app_dir,
  certs_dir,
  groovy_dir,
].each {|dir|
  resources(directory: dir) rescue directory dir do
    owner 'root'
    group 'root'
    mode '0755'
    recursive true
  end
}

ports = config['services']['jenkins']['ports']
override_config['services']['jenkins']['ports'] = ['8080:8080', '50000:50000'] if ports.empty?

jenkins_owner = node['jenkins-grid']['docker-compose']['jenkins_home']['owner']
jenkins_home_path = node['jenkins-grid']['docker-compose']['jenkins_home']['path']
unless jenkins_home_path.nil?
  directory jenkins_home_path do
    owner jenkins_owner
    group 'root'
    mode '0755'
    recursive true
  end
  vols.push("#{jenkins_home_path}:/var/jenkins_home")

  template "#{jenkins_home_path}/log.properties" do
    source 'var/lib/jenkins_home/log.properties'
    owner 'root'
    group 'root'
    mode '0644'
  end
  java_opts.push('-Djava.util.logging.config.file=/var/jenkins_home/log.properties')
end

if node['jenkins-grid']['with_ssl_cert_cookbook']
  include_recipe 'ssl_cert::server_key_pairs'
  ::Chef::Recipe.send(:include, SSLCert::Helper)
  cn = node['jenkins-grid']['ssl_cert']['common_name']
  key_path = server_key_path(cn)

  bash 'copy_ssl_server_key' do
    code <<-EOH
      cp #{key_path} #{certs_dir}/server.key
      chown #{jenkins_owner} #{certs_dir}/server.key
      chmod 600 #{certs_dir}/server.key
    EOH
    sensitive true
    action :run
    not_if "cmp #{key_path} #{certs_dir}/server.key"
    #action :nothing
    #subscribes :run, "file[#{key_path}]"
  end

  vols.push("#{server_cert_path(cn)}:/var/lib/jenkins/server.crt:ro")
  vols.push("#{certs_dir}/server.key:/var/lib/jenkins/server.key:ro")
  jenkins_opts.push('--httpsCertificate=/var/lib/jenkins/server.crt')
  jenkins_opts.push('--httpsPrivateKey=/var/lib/jenkins/server.key')
end

executors_conf = 'ref/init.groovy.d/executors.groovy'
vols.push("#{app_dir}/#{executors_conf}:/usr/share/jenkins/#{executors_conf}:ro")

unless jenkins_opts.empty?
  if !config['services']['jenkins']['environment'].nil? \
    && !config['services']['jenkins']['environment']['JENKINS_OPTS'].nil?
    jenkins_opts.unshift(config['services']['jenkins']['environment']['JENKINS_OPTS'])
  end
  envs['JENKINS_OPTS'] = jenkins_opts.join(' ')
end

unless java_opts.empty?
  if !config['services']['jenkins']['environment'].nil? \
    && !config['services']['jenkins']['environment']['JAVA_OPTS'].nil?
    java_opts.unshift(config['services']['jenkins']['environment']['JAVA_OPTS'])
  end
  envs['JAVA_OPTS'] = java_opts.join(' ')
end

# force_override for merging JENKINS_OPTS and JAVA_OPTS attributes' value.
force_override_config['services']['jenkins']['environment'] = envs unless envs.empty?
override_config['services']['jenkins']['volumes'] = vols unless vols.empty?

[
  'docker-compose.yml',
  'ref/init.groovy.d/executors.groovy',
].each {|conf_file|
  template "#{app_dir}/#{conf_file}" do
    source  "opt/docker-compose/app/jenkins/#{conf_file}"
    owner 'root'
    group 'root'
    mode '0644'
  end
}

log <<-"EOM"
Note: You must execute the following command manually.
  See #{doc_url}
  - Start:
    $ cd #{app_dir}
    $ docker-compose up -d
  - Stop
    $ docker-compose down
EOM
