#
# Cookbook Name:: concourse-ci
# Recipe:: docker-compose
#
# Copyright 2017, whitestar
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

require 'securerandom'

doc_url = 'https://concourse.ci/docker-repository.html'

include_recipe 'docker-grid::compose'

app_dir = node['concourse-ci']['docker-compose']['app_dir']
pgdata_dir = node['concourse-ci']['docker-compose']['pgdata_dir']
web_keys_dir = node['concourse-ci']['docker-compose']['web_keys_dir']
worker_keys_dir = node['concourse-ci']['docker-compose']['worker_keys_dir']
#certs_dir = "#{app_dir}/certs"

[
  app_dir,
  pgdata_dir,
  web_keys_dir,
  worker_keys_dir,
  #certs_dir,
].each {|dir|
  resources(directory: dir) rescue directory dir do
    owner 'root'
    group 'root'
    mode '0755'
    recursive true
  end if !dir.nil? && !dir.empty?
}

bash 'ssh-keygen_keys' do
  code <<-"EOH"
  script commands
    ssh-keygen -t rsa -f #{web_keys_dir}/tsa_host_key -N ''
    ssh-keygen -t rsa -f #{web_keys_dir}/session_signing_key -N ''
    ssh-keygen -t rsa -f #{worker_keys_dir}/worker_key -N ''
    cp #{worker_keys_dir}/worker_key.pub #{web_keys_dir}/authorized_worker_keys
    cp #{web_keys_dir}/tsa_host_key.pub #{worker_keys_dir}
  EOH
  action :run
  not_if { File.exist?("#{web_keys_dir}/tsa_host_key") }
  not_if { File.exist?("#{web_keys_dir}/session_signing_key") }
  not_if { File.exist?("#{worker_keys_dir}/worker_key") }
end

config_format_version = node['concourse-ci']['docker-compose']['config_format_version']

# if config_format_version == '1'
config_srvs = node['concourse-ci']['docker-compose']['config']
override_config_srvs = node.override['concourse-ci']['docker-compose']['config']
force_override_config_srvs = node.force_override['concourse-ci']['docker-compose']['config']
if config_format_version == '2'
  config_srvs = config_srvs['services']
  override_config_srvs = override_config_srvs['services']
  force_override_config_srvs = force_override_config_srvs['services']
end

# Database
db_envs_org = config_srvs['concourse-db']['environment']
db_envs = {}
db_vols = []

db_passwd = db_envs_org['POSTGRES_PASSWORD']
if db_passwd.nil? || db_passwd.empty?
  db_passwd = SecureRandom.hex  # or urlsafe_base64
  db_envs['POSTGRES_PASSWORD'] = db_passwd
end

db_vols.push("#{pgdata_dir}:#{db_envs_org['PGDATA']}") if !pgdata_dir.nil? && !pgdata_dir.empty?

force_override_config_srvs['concourse-db']['environment'] = db_envs unless db_envs.empty?
override_config_srvs['concourse-db']['volumes'] = db_vols unless db_vols.empty?

# Web
web_envs_org = config_srvs['concourse-web']['environment']
web_envs = {}

web_ports = config_srvs['concourse-web']['ports']
override_config_srvs['concourse-web']['ports'] = ['8080:8080'] if web_ports.empty?

basic_auth_passwd = web_envs_org['CONCOURSE_BASIC_AUTH_PASSWORD']
if basic_auth_passwd.nil? || basic_auth_passwd.empty?
  basic_auth_passwd = SecureRandom.hex  # or urlsafe_base64
  web_envs['CONCOURSE_BASIC_AUTH_PASSWORD'] = basic_auth_passwd
end

external_url = web_envs_org['CONCOURSE_EXTERNAL_URL']
web_envs['CONCOURSE_EXTERNAL_URL'] = "http://#{node['ipaddress']}:8080" if external_url.nil?

data_source = web_envs_org['CONCOURSE_POSTGRES_DATA_SOURCE']
data_source.gsub!(/<POSTGRES_PASSWORD>/, db_passwd)
web_envs['CONCOURSE_POSTGRES_DATA_SOURCE'] = data_source

force_override_config_srvs['concourse-web']['environment'] = web_envs unless web_envs.empty?

=begin
if node['concourse-ci']['with_ssl_cert_cookbook']
  ::Chef::Recipe.send(:include, SSLCert::Helper)
  cn = node['concourse-ci']['ssl_cert']['common_name']
  key_path = server_key_path(cn)

  bash 'copy_ssl_server_key' do
    code <<-EOH
      cp #{key_path} #{certs_dir}/server.key
      chmod 600 #{certs_dir}/server.key
    EOH
    sensitive true
    action :nothing
    subscribes :run, "file[#{key_path}]"
  end

  # TODO: key path setup.
end
=end

template "#{app_dir}/docker-compose.yml" do
  source  'opt/docker-compose/app/concourse/docker-compose.yml'
  owner 'root'
  group 'root'
  mode '0600'
  sensitive true
end

log <<-"EOM"
Note: You must execute the following command manually.
  See #{doc_url}
  - Start:
    $ cd #{app_dir}
    $ sudo docker-compose up -d
  - Stop
    $ sudo docker-compose down
EOM
