#ifndef CRC16_H_
#define CRC16_H_
/*
 *      Author: alexrayne <alexraynepe196@gmail.com>
  ------------------------------------------------------------------------
    Copyright (c) alexrayne

   All rights reserved.
   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are met:
   - Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   - Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   - Neither the name of ARM nor the names of its contributors may be used
     to endorse or promote products derived from this software without
     specific prior written permission.
   *
   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDERS AND CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. *
 */

/**
 * \file
 *         Header file for the CRC16 calculcation
 * \author
 *         alexrayne <alexraynepe196@gmail.com>
 *
 */

/** \addtogroup lib
 * @{ */

/**
 * \defgroup crc16 Cyclic Redundancy Check 16 (CRC16) calculation
 *
 * The Cyclic Redundancy Check 16 is a hash function that produces a
 * checksum that is used to detect errors in transmissions. The CRC16
 * calculation module is an iterative CRC calculator that can be used
 * to cumulatively update a CRC checksum for every incoming byte.
 *
 * @{
 */

/**
 * \name CRC16 CITT realisation style
 *
 *             different  realisations uses different size of lookup table,
 *             so it can be choose between code size vs speed
 *      *use: declare CRC16_STYLE define in project HAL configuration <project-conf.h>
 * @{
 */

/**
 * \brief      crc16 function without lookup table.
 *
 *             it uses bit operations, with no lookup.
 *             smalest code memory, but slowest code, especialy for 8bit mcu
 */
#define CRC16_STYLE_NOLOOKUP   0
/**
 * \brief      crc16 function with lookup table*256 - byte wise operation.
 *
 *             most popular algorithm - byte opiented operation, with lokup
 *             table on 256 entryes. most speed code, even 8bit mcu.
 *             but may impact on data cache.
 */
#define CRC16_STYLE_LOOKUP8   1
/**
 * \brief      crc16 function with lookup table*16 - 4bit wise operation.
 *
 *             less impact on data cache, use 4bit shift.
 */
#define CRC16_STYLE_LOOKUP4   2
/**
 * \brief      crc16 function lookup from RAM may works pretty faster vs FLASH one
 */
#define CRC16_STYLE_LOOKUP_FAST      0x100


//* \brief CRC standards polinoms
#define CRC16_CCITT_POLY             0x1021
#define CRC16_CCITT_RPOLY            0x8408
#define CRC16_ANSI_POLY              0x8005
#define CRC16_ANSI_RPOLY             0xA001
#define CRC16_ANSI_MODBUS            CRC16_ANSI_POLY

/* @} */

/**
 * \brief      Update an accumulated CRC16 checksum with one byte.
 * \param b    The byte to be added to the checksum
 * \param crc  The accumulated CRC that is to be updated.
 * \return     The updated CRC checksum.
 *
 *             This function updates an accumulated CRC16 checksum
 *             with one byte. It can be used as a running checksum, or
 *             to checksum an entire data block.
 *
 *             \note The algorithm used in this implementation is
 *             tailored for a running checksum and does not perform as
 *             well as a table-driven algorithm when checksumming an
 *             entire data block.
 *
 */

// Poly  : 0x1021    x^16 + x^12 + x^5 + 1
unsigned short crc16ccitt_add(unsigned char b, unsigned short crc);
// Poly  : 0x1021    x^16 + x^15 + x^2 + 1
unsigned short crc16ansi_add(unsigned char b, unsigned short crc);

/**
 * \brief      Calculate the CRC16 over a data area
 * \param data Pointer to the data
 * \param datalen The length of the data
 * \param acc  The accumulated CRC that is to be updated (or zero).
 * \return     The CRC16 checksum.
 *
 *             This function calculates the CRC16 checksum of a data area.
 *
 *             \note The algorithm used in this implementation is
 *             tailored for a running checksum and does not perform as
 *             well as a table-driven algorithm when checksumming an
 *             entire data block.
 */
unsigned short crc16ccitt_data(const void* data, int datalen,
			  unsigned short acc);
unsigned short crc16ansi_data(const void* data, int datalen,
              unsigned short acc);

/** @} */
/** @} */

#endif /* CRC16_H_ */
