/*
 *      Author: alexrayne <alexraynepe196@gmail.com>
  ------------------------------------------------------------------------
    Copyright (c) alexrayne

   All rights reserved.
   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are met:
   - Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   - Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   - Neither the name of ARM nor the names of its contributors may be used
     to endorse or promote products derived from this software without
     specific prior written permission.
   *
   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDERS AND CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. *
*/

#ifndef CLI_ARGS_HPP
#define CLI_ARGS_HPP

#include "cli.hpp"



namespace cli {


// парсер аргумента вида a=x / a='x' / a="x"
struct ArgRef {
    const_line      cmd;
    const_line      val;
    int_fast16_t    wlen;
    // @sa BADCmdID - в случае особенонго аргумента указывает его тип
    int_fast16_t    vlen;

    enum BADCmdID{
        cmdMINUS = -1, // --
        cmdPLUS  = -2, // ++
    };

    bool is_valid() const {
        return (cmd != nullptr) && (wlen > 0)
                /*&& (val != nullptr) && (vlen > 0)*/
                ;
    }

    // в s разбирает части аргумента вида [-[-]]<cmd>=(<word>
    //                                                 |"<word> .. <word>"
    //                                                 |'<word> .. <word>')
    // @return - строка после аргумента
    //          null - аргумент не может быть опознан
    const char* pass(const char* s);
};


//-----------------------------------------------------------------------------
// this style denotes arg parser-validator, and mixes with id|style
enum CLIArgStyle{
    argSTYLE_Pos = 24,
    argSTYLE_Msk = 0xffu << argSTYLE_Pos,

    argLONG     = 'L'<<argSTYLE_Pos, 
    argULONG    = 'U'<<argSTYLE_Pos, 
    argFLOAT    = 'F'<<argSTYLE_Pos, 
    argSTRING   = 'S'<<argSTYLE_Pos, 
    argBOOL     = 'B'<<argSTYLE_Pos, 
    argONOFF    = 'O'<<argSTYLE_Pos,
};

struct CLIArgDef: public CLINameDef {
    
    typedef unsigned ArgID;
    // this ID passed to arg assigner with accepted value
    // @sa CLIArgStyle first char - is one of CLIArgStyle

    static ArgID        as_id(unsigned x){return x & ~argSTYLE_Msk;}
    static CLIArgStyle  as_style(int x){return (CLIArgStyle)(x & argSTYLE_Msk);}
};

typedef const CLIArgDef*  CLIArgsDef;

struct CLIArgsProcessor {
public:
    CLIArgsProcessor(const CLINameDef* args_set);

    int cli_process(CLI_shell* shell, const_line line);

public:

    typedef enum { srOK,
                   // Нужно чтобы сообщение о ошибке напечатало
                   srBAD,   //argument not accepted. continue parce
                   srSTOP,  //stop parsing silently and well. 
                   srBreak  //so bad, should break parse
                 } SeqResult;
    typedef unsigned  ArgID;

    SeqResult _on_uncknown_arg;
    SeqResult _on_bad_arg;

    // этот обработчик вызывается на неопознаный аргумент
    SeqResult on_uncknown_arg() const {
        return _on_uncknown_arg;
    }

    SeqResult on_bad_arg(CLI_shell* shell, ArgID id);

public:

    //this is user defined arg value processors. 
    // provide it to assign args to controled structure
    virtual SeqResult assign_bool(CLI_shell* shell, ArgID id, bool value);
    virtual SeqResult assign_int(CLI_shell* shell, ArgID id, long value);
    virtual SeqResult assign_float(CLI_shell* shell, ArgID id, double value);
    virtual SeqResult assign_str(CLI_shell* shell, ArgID id
                                    , const_line value, unsigned len);

    //last processed arg
    ArgRef      arg;

protected:
    CLIArgsDef  args;

    template <CLIArgStyle T>
    SeqResult   pass(ArgID id, CLI_shell* shell);
};

}; //namespace cli

#endif // CLI_ARGS_HPP
