/*
 * tim_device.hpp
 *
 *  Created on: 28 нояб. 2018 г.
 *      Author: alexrayne <alexraynepe196@gmail.com>
  ------------------------------------------------------------------------
    Copyright (c) alexrayne

   All rights reserved.
   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are met:
   - Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   - Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   - Neither the name of ARM nor the names of its contributors may be used
     to endorse or promote products derived from this software without
     specific prior written permission.
   *
   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDERS AND CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. *
 *-----------------------------------------------------------------
 * реализация менеджера таймера на ВЕ94Т
 */

#ifndef HAL_MDR32F9QX_TIM_DEVICE_HPP_
#define HAL_MDR32F9QX_TIM_DEVICE_HPP_

#include <mcu_tim.h>
#include <mcu_nvic.h>
#include <mcu_gpio.h>
#include <timer_hal.hpp>

class TIMER_Device
        : public TIM_Device
{
public:
    //typedef TIM_Device  inherited;
    typedef TIM_TypeDef io_t;
    struct DEV_INIT{
        io_t*               timer;
        NVIC_InitTypeDef    NVIC_Init;
    };

    TIMER_Device();


    DevResult   connect(const DEV_INIT& mode);
    io_t*       io() const {return iox;}

public:
    DevResult   init();

    DevResult enable(bool onoff){
        TIM_Cmd(iox, (onoff)?ENABLE:DISABLE );
        return DEV_OK;
    };
    bool is_enabled() const{
        return TIM_is_enabled(iox);
    }

    enum SourceID{
          srcPCLK   = 0
        , srcTI0 = 1
        , srcTI1 = 2
        , srcTI2 = 3
        , srcTI3 = 4

        , srcIN1EDGE
        , srcIN1
        , srcIN2
    };

    enum SlaveModeID{
          slmMASTER = 0
          // сбрасывает счетчик, тактируемый изнутри, сигналом мастера
        , slmRESET  = LL_TIM_SLAVEMODE_RESET
        // сетчик, тактируемый изнутри, считает пока разрешает мастер
        , slmGATED  = LL_TIM_SLAVEMODE_GATED
        // сетчик, тактируемый изнутри, включается и начинает счет по сигналу от мастера
        , slmTRIG   = LL_TIM_SLAVEMODE_TRIGGER
    };

    enum Limits{
          prescale_limit = 0x10000u
        , period_max    = 0xffffu
        , period_limit  = period_max+1
        , width         = 16
        , comp_chanels  = 4
    };

    //  тактирование от системного клока требует прескалера, и мы его используем
    DevResult   cfg_internalCLK(unsigned prescale);
    DevResult   cfg_source(SourceID x);
    DevResult   cfg_trigger(SourceID x);
    // режим реакции на внешний сигнал
    DevResult   cfg_slave(SlaveModeID x);
    DevResult   set_period(unsigned period, timer_mode_t mode);
    // содержимое счетчика
    unsigned    value() const{return iox->CNT;};
    void        value(unsigned x) {iox->CNT = x;};
    // содержимое регистра перезагрузки
    unsigned    period() const {return iox->ARR;};
    void        period(unsigned x) {iox->ARR = x;};

    enum /*CompareModeID*/{
          compMASK = 0x7
        , compUPDATE_PRELOAD= 8, compUPDATE_ZERO = compUPDATE_PRELOAD
        , compUPDATE_NOW    = 0
    };

    DevResult   set_compare(chanel_id chanel, unsigned value, chanel_mode_t mode);
    // содержимое компаратора
    unsigned    compare(chanel_id chanel) const;
    void        compare(chanel_id chanel, unsigned x);
    DevResult   compare_mode(chanel_id ch, chanel_mode_t mode);

    enum /*IRQId*/{
          irqNONE   = 0
        , irqZERO   = TIM_IE_ZERO
        , irqTRIG   = TIM_IE_TRIG
        , irqRELOAD = TIM_IE_RELOAD
        // тоже что и irqRELOAD, но реагирует только на переполнение счетчика
        , irqOVER   = TIM_IE_RELOAD | 0x1000

        // прерывания событий от компараторов
        , irqOC1    = TIM_IE_OC1
        , irqOC2    = TIM_IE_OC2
        , irqOC3    = TIM_IE_OC3
        , irqOC4    = TIM_IE_OC4
        , irqOC_Msk = TIM_IE_OCx, irqOC_Pos = TIM_IE_OCx_Pos
    };

public:
    enum ICSourceID{
          icsNone       = 0x0
          //использовать вход своего канала
        , icsCHSame     = 0x10
          //использовать вход альтернативного канала: ch1 <-> ch2, ch3 <-> ch4
        , icsCHAlter    = 0x20
        , icsTrigger    = 0x30
        , icsSel_Msk    = 0x30

        , icsRise       = LL_TIM_IC_POLARITY_RISING
        , icsFall       = LL_TIM_IC_POLARITY_FALLING
        , icsBoth       = LL_TIM_IC_POLARITY_BOTHEDGE
        , icsFront_Msk  = 0xf

        // эти константы использую чтобы задать явный номер канала
        , icsCH1        = 0x100
        , icsCH2        = 0x200
        , icsCH3        = 0x300
        , icsCH4        = 0x400
        , icsCH_Msk     = 0xf00

    };
    typedef unsigned input_mode_t;
    DevResult   input_select(chanel_id ch, input_mode_t mode);

public:
    // \arg mode - набор прерываний IRQId, зависит от локализации к железу
    inline
    DevResult    irq_enable(unsigned mode){
        TIM_IE_enable(iox, mode);
        if ( (mode & irqRELOAD) != 0) {
            if ((mode & irqOVER) == irqOVER)
                LL_TIM_SetUpdateSource(iox, LL_TIM_UPDATESOURCE_COUNTER);
            else
                LL_TIM_SetUpdateSource(iox, LL_TIM_UPDATESOURCE_REGULAR);
        }
        return DEV_OK;
    };

    inline
    DevResult    irq_disable(unsigned mode){
        TIM_IE_disable(iox, mode);
        return DEV_OK;
    }

    // разрешение/запрещение прерывания NVIC таймера
    DevResult    irq_enable(){
        NVIC_EnableIRQ((IRQn_Type)cfg->NVIC_Init.NVIC_IRQChannel);
        return DEV_OK;
    };

    DevResult    irq_disable(){
        NVIC_DisableIRQ((IRQn_Type)cfg->NVIC_Init.NVIC_IRQChannel);
        return DEV_OK;
    };

public:
    // обработчик прерывания. Должен вызываться из процедуры обрботки прерывнаия таймера:
    /*
        TIMER_Device mytimer;
        extern "C" void TIM1_IRQHandler(void)
        {
            mytimer.IRQ();
        }
     *
     */
    void IRQ();
protected:
    io_t*           iox;
    const DEV_INIT* cfg;
    unsigned eval_period_us(unsigned us);
    void     compare_mode_preload(chanel_id ch, bool onoff);

    // индекс таймера
    int             io_idx;
    // таймер умеет CenterAligned режимы
    bool            io_can_sweep;
};



//---------------------------------------------------------------------------
// input chanels on timers
extern const GPIOFUNC_INIT     TIM9_IN1_GPIOE;
extern const GPIOFUNC_INIT     TIM9_IN2_GPIOE;
extern const GPIOFUNC_INIT     TIM9_IN1_GPIOA;
extern const GPIOFUNC_INIT     TIM9_IN2_GPIOA;



#endif /* HAL_MDR32F9QX_TIM_DEVICE_HPP_ */
