// "License": Public Domain
// I, Mathias Panzenböck, place this file hereby into the public domain. Use it at your own risk for whatever you like.
// In case there are jurisdictions that don't support putting things in the public domain you can also consider it to
// be "dual licensed" under the BSD, MIT and Apache licenses, if you want to. This code is trivial anyway. Consider it
// an example on how to get the endian conversion functions on different platforms.

#ifndef PORTABLE_ENDIAN_H__
#define PORTABLE_ENDIAN_H__


#include <stdint.h>

typedef uint16_t    uint16_net_t;
typedef  int16_t     int16_net_t;
typedef uint32_t    uint32_net_t;
typedef  int32_t     int32_net_t;
typedef uint64_t    uint64_net_t;
typedef  int64_t     int64_net_t;


#if (defined(_WIN16) || defined(_WIN32) || defined(_WIN64)) && !defined(__WINDOWS__)

#	define __WINDOWS__

#endif

#if defined(__linux__) || defined(__CYGWIN__)

#	include <endian.h>

#elif defined(__APPLE__)

#	include <libkern/OSByteOrder.h>

#	define htobe16(x) OSSwapHostToBigInt16(x)
#	define htole16(x) OSSwapHostToLittleInt16(x)
#	define be16toh(x) OSSwapBigToHostInt16(x)
#	define le16toh(x) OSSwapLittleToHostInt16(x)
 
#	define htobe32(x) OSSwapHostToBigInt32(x)
#	define htole32(x) OSSwapHostToLittleInt32(x)
#	define be32toh(x) OSSwapBigToHostInt32(x)
#	define le32toh(x) OSSwapLittleToHostInt32(x)
 
#	define htobe64(x) OSSwapHostToBigInt64(x)
#	define htole64(x) OSSwapHostToLittleInt64(x)
#	define be64toh(x) OSSwapBigToHostInt64(x)
#	define le64toh(x) OSSwapLittleToHostInt64(x)

#	define __BYTE_ORDER    BYTE_ORDER
#	define __BIG_ENDIAN    BIG_ENDIAN
#	define __LITTLE_ENDIAN LITTLE_ENDIAN
#	define __PDP_ENDIAN    PDP_ENDIAN

#elif defined(__OpenBSD__)

#	include <sys/endian.h>

#elif defined(__NetBSD__) || defined(__FreeBSD__) || defined(__DragonFly__)

#	include <sys/endian.h>

#	define be16toh(x) betoh16(x)
#	define le16toh(x) letoh16(x)

#	define be32toh(x) betoh32(x)
#	define le32toh(x) letoh32(x)

#	define be64toh(x) betoh64(x)
#	define le64toh(x) letoh64(x)

#elif defined(__WINDOWS__)

#	include <winsock2.h>
#	include <sys/param.h>

#	if BYTE_ORDER == LITTLE_ENDIAN

#		define htobe16(x) htons(x)
#		define htole16(x) (x)
#		define be16toh(x) ntohs(x)
#		define le16toh(x) (x)
 
#		define htobe32(x) htonl(x)
#		define htole32(x) (x)
#		define be32toh(x) ntohl(x)
#		define le32toh(x) (x)
 
#		define htobe64(x) htonll(x)
#		define htole64(x) (x)
#		define be64toh(x) ntohll(x)
#		define le64toh(x) (x)

#	elif BYTE_ORDER == BIG_ENDIAN

		/* that would be xbox 360 */
#		define htobe16(x) (x)
#		define htole16(x) __builtin_bswap16(x)
#		define be16toh(x) (x)
#		define le16toh(x) __builtin_bswap16(x)
 
#		define htobe32(x) (x)
#		define htole32(x) __builtin_bswap32(x)
#		define be32toh(x) (x)
#		define le32toh(x) __builtin_bswap32(x)
 
#		define htobe64(x) (x)
#		define htole64(x) __builtin_bswap64(x)
#		define be64toh(x) (x)
#		define le64toh(x) __builtin_bswap64(x)

#	else

#		error byte order not supported

#	endif

#	define __BYTE_ORDER    BYTE_ORDER
#	define __BIG_ENDIAN    BIG_ENDIAN
#	define __LITTLE_ENDIAN LITTLE_ENDIAN
#	define __PDP_ENDIAN    PDP_ENDIAN

#elif defined(_NEWLIB_VERSION) || defined(__GNUC__)

	/*
	 * GNU ARM toolchain, and possibly other bare-metal toolchains
	 * built on newlib.  Tested with
	 * (GNU Tools for ARM Embedded Processors 6-2017-q2-update
	 */

#	include <machine/endian.h>

#	if BYTE_ORDER == LITTLE_ENDIAN

#		define htobe16(x) __bswap16(x)
#		define htole16(x) (x)
#		define be16toh(x) __bswap16(x)
#		define le16toh(x) (x)

#		define htobe32(x) __bswap32(x)
#		define htole32(x) (x)
#		define be32toh(x) __bswap32(x)
#		define le32toh(x) (x)

#		define htobe64(x) __bswap64(x)
#		define htole64(x) (x)
#		define be64toh(x) __bswap64(x)
#		define le64toh(x) (x)

#	elif BYTE_ORDER == BIG_ENDIAN

#		define htobe16(x) (x)
#		define htole16(x) __bswap16(x)
#		define be16toh(x) (x)
#		define le16toh(x) __bswap16(x)

#		define htobe32(x) (x)
#		define htole32(x) __bswap32(x)
#		define be32toh(x) (x)
#		define le32toh(x) __bswap32(x)

#		define htobe64(x) (x)
#		define htole64(x) __bswap64(x)
#		define be64toh(x) (x)
#		define le64toh(x) __bswap64(x)

#	else

#		error byte order not supported

#	endif

#else

#	error platform not supported

#endif

#ifndef __ntohll

#if ( __INT_WIDTH__ == 32 )

static inline
uint64_t NTOHLL(uint64_net_t x){
    uint64_t tmp = __ntohl(x & 0xfffffffful);
    return (tmp << 32) | __ntohl(x>>32);
}

#define __ntohll(x)    NTOHLL(x)
#define __htonll(x)    NTOHLL(x)

#elif ( __INT_WIDTH__ == 64 )

#   if BYTE_ORDER == LITTLE_ENDIAN
#define __ntohll(x)    __bswap64(x)
#define __htonll(x)    __bswap64(x)
#else
#define __ntohll(x)    (x)
#define __htonll(x)    (x)
#endif

#elif ( __INT_WIDTH__ == 16 )

static inline
uint64_t NTOHLL(uint64_t x){
    typedef  union {
        uint64_t    value;
        uint16_t    w[4];
    } u64w;
    u64w tmpx.value = x;
    u64w tmpy;
    tmpy.w[0] = __ntohl(w[3]);
    tmpy.w[1] = __ntohl(w[2]);
    tmpy.w[2] = __ntohl(w[1]);
    tmpy.w[3] = __ntohl(w[0]);
    return tmpy.value;
}

#define __ntohll(x)    NTOHLL(x)
#define __htonll(x)    NTOHLL(x)

#endif //( __INT_WIDTH__ == 16 )

#endif //__ntohll

#endif
