/*
 * memoryx.h
 *
 *  Created on: 13/08/2021.
 *      Author: alexrayne <alexraynepe196@gmail.com>
  ------------------------------------------------------------------------
    Copyright (c) alexrayne

   All rights reserved.
   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are met:
   - Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   - Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   - Neither the name of ARM nor the names of its contributors may be used
     to endorse or promote products derived from this software without
     specific prior written permission.
   *
   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDERS AND CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. *
  ------------------------------------------------------------------------
      memory operations, extends stdlib memoru.h
 */

#ifndef LIB_MEMORYX_H_
#define LIB_MEMORYX_H_

#include <memory.h>
#include <stdbool.h>
#include <string.h>

#include <c_compat.h>



void memseti(int* x, int value, size_t cnt);
void memsetl(long* x, long value, size_t cnt);

INLINE
void memsetu(unsigned* x, unsigned value, size_t cnt){
    memseti((int*)x, (int)value, cnt);
}

INLINE
void memsetul(unsigned long* x, unsigned long value, size_t cnt){
    memsetl((long*)x, (long)value, cnt);
}


void  memcpyi( int *  __restrict y,  const int*  __restrict x, unsigned cnt);
void  memcpyl( long * __restrict y, const long* __restrict x, unsigned cnt);

INLINE
void  memcpyu( unsigned * y,  const unsigned* x, unsigned cnt){
    memcpyi( (int*)y, (const int*)x, cnt);
}

INLINE
void  memcpyul( unsigned long * y, const unsigned long* x, unsigned cnt){
    memcpyl( (long*)y, (const long*)x, cnt);
}


/// @return == 0 same
///         > 0 len of same array, when a[len-1] > b[len-1]
///         < 0 -len of same array, when a[len-1] < b[len-1]
int  memcmp_l( const long * __restrict a, const long* __restrict b, unsigned cnt);

static inline
int  memcmp_ul( const unsigned long * a, const unsigned long* b, unsigned cnt){
    return memcmp_l( (const long *)a, (const long *)b, cnt );
}


/// @return == 0 same
///         > 0 - mismatched position [1..
int  memcmp_b( const void * a, const void* b, unsigned bytes);

/// @return == 0 same
///         > 0 - mismatched position [1..
unsigned mismatch_b( const void * a, const void* b, unsigned bytes);


/// @brief compares arrays[cnt] a==b
/// @return true is same
INLINE
bool is_same_l( const long*a, const long* b, unsigned cnt){
    return memcmp_l(a, b, cnt) == 0;
}

INLINE
bool is_same_ul( const unsigned long* a, const unsigned long* b, unsigned cnt){
    return memcmp_l((const long*)a, (const long*)b, cnt) == 0;
}


/// @brief seek byte != fill in src
/// @return - point of difference
char*    memnchr(const void* src, int fill, size_t size);

/// @brief seek x at src[size]
static inline
unsigned long*   memscanul(const void* src, unsigned long x, size_t size){
    const unsigned long* pl = (const unsigned long*)src;
    for ( ; size > 0; --size, ++pl )
        if(*pl == x)
            return (unsigned long*)pl;
    return NULL;
}


/* @brief The strnlen() function returns the number of bytes in the string
 * pointed to by s, excluding the terminating null byte ('\0'), but
 * at most maxlen.  In doing this, strnlen() looks only at the first
 * maxlen characters in the string pointed to by s and never beyond
 * s[maxlen-1]
 *
 * @return The strnlen() function returns strlen(s), if that is less than
       maxlen, or maxlen if there is no null terminating ('\0') among
       the first maxlen characters pointed to by s.
 */
size_t   strnlen(const char *s, size_t limit);


/// @return max of a or b - accorded to max abs value
long    max_absl( long a, long b);

/// @return true - if  x -> [buf .. +size]
bool    ptr_in( const void* x, const void* buf, unsigned size);

static inline
bool PTR_IN( const void* x, const void* buf, unsigned size){
    const char* px   = (const char*) x;
    const char* pbuf = (const char*) buf;

    return (px >= pbuf) && ( px < (pbuf+size));
}



/// @brief поиск позиции min/max в массиве
int  idxmax_u( const unsigned * x, unsigned cnt);
int  idxmin_u( const unsigned * x, unsigned cnt);
int  idxmax_ul( const unsigned long * x, unsigned cnt);
int  idxmin_ul( const unsigned long * x, unsigned cnt);



#endif /* LIB_MEMORYX_H_ */
