/*
 * rtimer_agt.h
 *
 *  Created on: 16/04/2021.
 *      Author: alexrayne <alexraynepe196@gmail.com>
  ------------------------------------------------------------------------
    Copyright (c) alexrayne

   All rights reserved.
   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are met:
   - Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   - Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   - Neither the name of ARM nor the names of its contributors may be used
     to endorse or promote products derived from this software without
     specific prior written permission.
   *
   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDERS AND CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. *
  ------------------------------------------------------------------------
    this is implementation of rtimer at AGT0 for ContikiOS
 */

#ifndef BSP_CPU_RTIMER_AGT_H_
#define BSP_CPU_RTIMER_AGT_H_

#include <project-conf.h>
#include <sys/rtimer.h>


//#define RTIMER_AGT_SCLK_HZ  32768
//#define RTIMER_AGT_LCLK_HZ  32768

//#define RTIMER_AGT_PRI    1



/*---------------------------------------------------------------------------*/
//* this is hardware specified default RTC sub-second resolution 32768Hz
#define RTIMER_AGT_SECOND_NORM     (1000)

#ifdef RTIMER_CONF_AGT_SECOND
#define RTIMER_AGT_SECOND RTIMER_CONF_AGT_SECOND
#elif defined(RTIMER_CONF_ARCH_SECOND)
#define RTIMER_AGT_SECOND RTIMER_CONF_ARCH_SECOND
#else
#define RTIMER_AGT_SECOND RTIMER_AGT_SECOND_NORM
#endif
/*---------------------------------------------------------------------------*/

//* this scale used for achieve maximum arguments range, avoiding overload on int32 value
#define AGTRT_USCALC_SCALE     8u
#define AGTRT_US               1000000ul
#define AGTRT_SCALED_US        (AGTRT_US/AGTRT_USCALC_SCALE)
#define AGTRT_SCALED_SECOND    (RTIMER_AGT_SECOND/AGTRT_USCALC_SCALE)
#if ( (AGTRT_SCALED_SECOND*AGTRT_USCALC_SCALE) != (RTIMER_AGT_SECOND) )
#error please fix AGTRT_USCALC_SCALE selection for your RTIMER_AGT_SECOND!!!
#endif

#define US_TO_AGTTICKS(US)  ( ((US) >= 0) ?                        \
                               (((uint32_t)(US) * (AGTRT_SCALED_SECOND) + AGTRT_SCALED_US/2u) / AGTRT_SCALED_US) :      \
                                -(((uint32_t)(-US) * (AGTRT_SCALED_SECOND) + AGTRT_SCALED_US/2u) / AGTRT_SCALED_US)\
                                )

#define AGTTICKS_TO_US(T)   (((T) >= 0) ?                     \
                               (((uint32_t)(T) * AGTRT_SCALED_US + (AGTRT_SCALED_SECOND/2u)) / (AGTRT_SCALED_SECOND)) : \
                               -(((uint32_t)(-T) * AGTRT_SCALED_US + (AGTRT_SCALED_SECOND/2u)) / (AGTRT_SCALED_SECOND))\
                               )
/* A 64-bit version because the 32-bit one cannot handle T >= 4295 ticks.
   Intended only for positive values of T. */
#define AGTTICKS_TO_US_64(T)  ((uint32_t)(((uint64_t)(T) * 1000000 + ((RTIMER_AGT_SECOND) / 2)) / (RTIMER_AGT_SECOND)))

#if RTIMER_ARCH_SECOND >= RTIMER_AGT_SECOND
#define RTIMER_FACTOR_AGT ( RTIMER_ARCH_SECOND/RTIMER_AGT_SECOND)
#define RTIMER_TO_AGTTICKS(t) ( (t) / RTIMER_FACTOR_AGT )
#define AGT_TO_RTIMERTICKS(t) ( (t) * RTIMER_FACTOR_AGT )
#else
#define AGT_FACTOR_RTIMER (RTIMER_AGT_SECOND/RTIMER_ARCH_SECOND)
#define RTIMER_TO_AGTTICKS(t) ( (t) * AGT_FACTOR_RTIMER )
#define AGT_TO_RTIMERTICKS(t) ( (t) / AGT_FACTOR_RTIMER )
#endif

/*---------------------------------------------------------------------------*/
/* Need 4 ticks to syncronize with AON, 2 ticks to account for 2 tick RTC read
   precision and 2 more ticks to put the value in the future. */
#define AGT_MINIMAL_SAFE_SCHEDULE               3u
#define RTIMER_MINIMAL_SAFE_AGT_SCHEDULE        AGT_TO_RTIMERTICKS(AGT_MINIMAL_SAFE_SCHEDULE)
/*---------------------------------------------------------------------------*/

///////////////////////////////////////////////////////////////////////
/**
 * Initialized the architecture-dependent part of rtimer.
 */
void rtimer_agt_init(void);


#ifndef RTIMER_AGT_EXEC0
#define RTIMER_AGT_EXEC0(...)  rtimer_run_next(__VA_ARGS__)
#endif

/**
 * Schedule the call to `rtimer_agt_execN` at the time t.
 * @arg t - rtimer ticks [rtimer_clock_t]
 */
void rtimer_agt_schedule(rtimer_clock_t t);


////////////////////////////////////////////////////////////////////////

// rtimer AGT utilises this ISR routines. FSP should link it to IRQ
void rtc_atg0_isr(void);




#endif /* BSP_CPU_RTIMER_AGT_H_ */
