/***********************************************************
	encode.c -- sliding dictionary with percolating update
***********************************************************/
#include "ar.h"
#include <stdlib.h>
#include <string.h>             /* memmove() */

#define PERCOLATE  0
#define NIL        0
#define USE_LH6_METHOD 1

#undef DICBIT
#undef DICSIZ
#define DICBIT (opts.method->dicbit)
#define DICSIZ (2 << (DICBIT-1))
/* #define PERC_FLAG ((unsigned)2 << (DICBIT+1)) ??? */
#define PERC_FLAG 0x8000U

#define HASH(p, c) ((p) + 2 * DICSIZ + ((c) << (DICBIT - 9)))
#define MAX_HASH_VAL HASH(DICSIZ + UCHAR_MAX, UCHAR_MAX)

#if USE_LH6_METHOD /* dicbit > 14 */
typedef uint32_t node;          /* need (dicbit+2) bits */
#else
typedef short node;
#endif

static uint8_t *text, *childcount;
static node pos, avail, *position, *parent, *prev, *next = NULL;
static int remainder;

#if MAXMATCH <= (UCHAR_MAX + 1)
static uint8_t *level;
#else
static uint16_t *level;
#endif

extern struct lha_opts opts;

struct match_t {
    node pos;
    int len;
};

static void
allocate_memory(void)
{
    if (next != NULL)
        return;
    text = malloc(DICSIZ * 2 + MAXMATCH);
    level = malloc((DICSIZ + UCHAR_MAX + 1) * sizeof(*level));
    childcount = malloc((DICSIZ + UCHAR_MAX + 1) * sizeof(*childcount));
#if PERCOLATE
    position = malloc((DICSIZ + UCHAR_MAX + 1) * sizeof(*position));
#else
    position = malloc(DICSIZ * sizeof(*position));
#endif
    parent = malloc(DICSIZ * 2 * sizeof(*parent));
    prev = malloc(DICSIZ * 2 * sizeof(*prev));
    next = malloc((MAX_HASH_VAL + 1) * sizeof(*next));
    if (next == NULL)
        error("Out of memory.");
}

static void
init_slide(void)
{
    node i;

    for (i = DICSIZ; i <= DICSIZ + UCHAR_MAX; i++) {
        level[i] = 1;
#if PERCOLATE
        position[i] = NIL;      /* sentinel */
#endif
    }
    for (i = DICSIZ; i < DICSIZ * 2; i++)
        parent[i] = NIL;
    avail = 1;
    for (i = 1; i < DICSIZ - 1; i++)
        next[i] = i + 1;
    next[DICSIZ - 1] = NIL;
    for (i = DICSIZ * 2; i <= MAX_HASH_VAL; i++)
        next[i] = NIL;
}

static node
child(node q, uint8_t c)
        /* q's child for character c (NIL if not found) */
{
    node r;

    r = next[HASH(q, c)];
    parent[NIL] = q;            /* sentinel */
    while (parent[r] != q)
        r = next[r];
    return r;
}

static void
makechild(node q, uint8_t c, node r)
        /* Let r be q's child for character c. */
{
    node h, t;

    h = HASH(q, c);

    /*
      insert r after h.

      h next -> t

      h next -> r
                r next -> t
    */
    t = next[h];
    next[h] = r;
    next[r] = t;

    /*
      h <- prev t

      h <- prev r
                r <- prev t
    */
    prev[t] = r;
    prev[r] = h;

    parent[r] = q;
    childcount[q]++;
}

static void
replace_node(node old, node new)
{
    node t;

    /* replace old into new */
    t = prev[old];
    prev[new] = t;
    next[t] = new;

    t = next[old];
    next[new] = t;
    prev[t] = new;

    parent[new] = parent[old];
}

static void
split(struct match_t *match, node old)
{
    node new;

    new = avail;
    avail = next[new];
    childcount[new] = 0;

    replace_node(old, new);

    level[new] = match->len;
    position[new] = pos;

    /* new child -> old */
    makechild(new, text[match->pos + match->len], old);
    /* new child -> pos
                    pos child -> old */
    makechild(new, text[pos + match->len], pos);
}

static void
insert_node(struct match_t *match)
{
    node q, r, j, t;
    uint8_t c, *t1, *t2;

    if (match->len >= 4) {
        match->len--;
        r = (match->pos + 1) | DICSIZ;
        while ((q = parent[r]) == NIL)
            r = next[r];
        while (level[q] >= match->len) {
            r = q;
            q = parent[q];
        }
#if PERCOLATE
        t = q;
        while (position[t] < 0) {
            position[t] = pos;
            t = parent[t];
        }
        if (t < DICSIZ)
            position[t] = pos | PERC_FLAG;
#else
        t = q;
        while (t < DICSIZ) {
            position[t] = pos;
            t = parent[t];
        }
#endif
    }
    else {
        q = text[pos] + DICSIZ;
        c = text[pos + 1];
        if ((r = child(q, c)) == NIL) {
            makechild(q, c, pos);
            match->len = 1;
            return;
        }
        match->len = 2;
    }

    /* found match->len */
    /* scan match->pos candidate r */
    for (;;) {
        if (r >= DICSIZ) {
            j = MAXMATCH;
            match->pos = r;
        }
        else {
            j = level[r];
            match->pos = position[r] & ~PERC_FLAG; /* 0x8000U */
        }
        if (match->pos >= pos)
            match->pos -= DICSIZ;
        t1 = &text[pos + match->len];
        t2 = &text[match->pos + match->len];
        while (match->len < j) {
            if (*t1 != *t2) {
                split(match, r);
                return;
            }
            match->len++;
            t1++;
            t2++;
        }
        if (match->len >= MAXMATCH)
            break;
        position[r] = pos;
        q = r;
        if ((r = child(q, *t1)) == NIL) {
            makechild(q, *t1, pos);
            return;
        }
        match->len++;
    }
    /* replace r into pos */
    replace_node(r, pos);

    parent[r] = NIL;
    next[r] = pos;              /* special use of next[] */
}

static void
delete_node_1(node s)
{
    node t, u;

    t = prev[s];
    u = next[s];
    next[t] = u;
    prev[u] = t;
}

static void
delete_node(void)
{
#if PERCOLATE
    node q, r, s, t, u;
#else
    node r, s, t;
#endif

    if (parent[pos] == NIL)
        return;

    delete_node_1(pos);

    r = parent[pos];
    parent[pos] = NIL;

    if (r >= DICSIZ || --childcount[r] > 1)
        return;
#if PERCOLATE
    t = position[r] & ~PERC_FLAG;
#else
    t = position[r];
#endif
    if (t >= pos)
        t -= DICSIZ;
#if PERCOLATE
    s = t;
    q = parent[r];
    while ((u = position[q]) & PERC_FLAG) {
        u &= ~PERC_FLAG;
        if (u >= pos)
            u -= DICSIZ;
        if (u > s)
            s = u;
        position[q] = (s | DICSIZ);
        q = parent[q];
    }
    if (q < DICSIZ) {
        if (u >= pos)
            u -= DICSIZ;
        if (u > s)
            s = u;
        position[q] = s | DICSIZ | PERC_FLAG;
    }
#endif
    s = child(r, text[t + level[r]]);

    delete_node_1(s);
    replace_node(r, s);

    /* free r */
    parent[r] = NIL;
    next[r] = avail;
    avail = r;
}

static void
get_next_match(struct lzh_ostream *wp, FILE *rfp, struct match_t *match)
{
    int n;

    remainder--;
    if (++pos == DICSIZ * 2) {
        memmove(&text[0], &text[DICSIZ], DICSIZ + MAXMATCH);
        n = fread_crc(&text[DICSIZ + MAXMATCH], DICSIZ, rfp, &wp->crc);
        wp->origsize += n;
        remainder += n;
        pos = DICSIZ;
        if (wp->fp != stdout && opts.quiet < 1) {
            putc('.', stdout);
        }
    }
    delete_node();
    insert_node(match);
}

void
encode(struct lzh_ostream *wp, FILE *rfp)
{
    struct match_t match, lastmatch;

    allocate_memory();
    init_slide();
    huf_encode_start(wp, opts.method);
    remainder = fread_crc(&text[DICSIZ], DICSIZ + MAXMATCH, rfp, &wp->crc);
    wp->origsize += remainder;

    if (wp->fp != stdout && opts.quiet < 1) {
        putc('.', stdout);
    }
    match.len = 0;
    pos = DICSIZ;
    insert_node(&match);
    if (match.len > remainder)
        match.len = remainder;
    while (remainder > 0 && !wp->unpackable) {
        lastmatch = match;
        get_next_match(wp, rfp, &match);
        if (match.len > remainder)
            match.len = remainder;
        if (match.len > lastmatch.len || lastmatch.len < THRESHOLD)
            output(wp, text[pos - 1], 0);
        else {
            output(wp, lastmatch.len + (UCHAR_MAX + 1 - THRESHOLD),
                   (pos - lastmatch.pos - 2) & (DICSIZ - 1));
            while (--lastmatch.len > 0)
                get_next_match(wp, rfp, &match);
            if (match.len > remainder)
                match.len = remainder;
        }
    }
    huf_encode_end(wp);
}
