using System;
using System.Collections.Generic;
using System.Windows;
using System.Windows.Data;
using System.Windows.Media;
using System.Windows.Input;
using System.Windows.Threading;
using System.Windows.Controls;
using System.Windows.Controls.Primitives;

namespace Hiyoko.Themes{
	using WinForms = System.Windows.Forms;
	
	public partial class ImageBrushSettingsForm : Window{
		private ImageBrushSettings targetSettings;
		
		public ImageBrushSettingsForm(ImageBrushSettings settings){
			this.targetSettings = settings;
			this.InitializeComponent();
			
			this.tileModeBox.DataContext = new KeyValuePair<string, TileMode>[]{
				new KeyValuePair<string, TileMode>("無し", TileMode.None),
				new KeyValuePair<string, TileMode>("敷き詰める", TileMode.Tile),
				new KeyValuePair<string, TileMode>("横方向に引っ繰り返して敷き詰める", TileMode.FlipX),
				new KeyValuePair<string, TileMode>("縦方向に引っ繰り返して敷き詰める", TileMode.FlipY),
				new KeyValuePair<string, TileMode>("縦と横方向に引っ繰り返して敷き詰める", TileMode.FlipXY)
			};
			this.stretchBox.DataContext = new KeyValuePair<string, Stretch>[]{
				new KeyValuePair<string, Stretch>("無し", Stretch.None),
				new KeyValuePair<string, Stretch>("領域を埋める", Stretch.Fill),
				new KeyValuePair<string, Stretch>("縦横比を維持して領域に収める", Stretch.Uniform),
				new KeyValuePair<string, Stretch>("縦横比を維持して領域を埋める", Stretch.UniformToFill),
			};
			this.alignmentXBox.DataContext = new KeyValuePair<string, AlignmentX>[]{
				new KeyValuePair<string, AlignmentX>("左", AlignmentX.Left),
				new KeyValuePair<string, AlignmentX>("中", AlignmentX.Center),
				new KeyValuePair<string, AlignmentX>("右", AlignmentX.Right)
			};
			this.alignmentYBox.DataContext = new KeyValuePair<string, AlignmentY>[]{
				new KeyValuePair<string, AlignmentY>("上", AlignmentY.Top),
				new KeyValuePair<string, AlignmentY>("中", AlignmentY.Center),
				new KeyValuePair<string, AlignmentY>("下", AlignmentY.Bottom)
			};
			
			this.TempSettings = new ImageBrushSettings();
			this.targetSettings.CopyTo(this.TempSettings);
			this.DataContext = this.TempSettings;
		}
		
		private void OpenImage(object target, RoutedEventArgs e){
			WinForms.OpenFileDialog dlg = new WinForms.OpenFileDialog();
			dlg.Filter = "画像ファイル|*.bmp;*.jpg;*.png;*.gif";
			dlg.CheckPathExists = true;
			dlg.CheckFileExists = true;
			if(dlg.ShowDialog() == WinForms.DialogResult.OK){
				this.imageFileEdit.Text = dlg.FileName;
			}
		}
		
		private void Close_CanExecute(object target, CanExecuteRoutedEventArgs e){
			e.CanExecute = true;
		}
		
		private void Close_Execute(object target, ExecutedRoutedEventArgs e){
			this.Close();
		}
		
		private void Save_CanExecute(object target, CanExecuteRoutedEventArgs e){
			e.CanExecute = true;
		}
		
		private void Save_Execute(object target, ExecutedRoutedEventArgs e){
			this.TempSettings.CopyTo(this.targetSettings);
			this.targetSettings.Save();
			this.DialogResult = true;
			this.Close();
		}
		
		public static readonly DependencyProperty TempSettingsProperty = DependencyProperty.Register("TempSettings", typeof(ImageBrushSettings), typeof(ImageBrushSettingsForm));
		private ImageBrushSettings TempSettings{
			get{
				return (ImageBrushSettings)this.GetValue(TempSettingsProperty);
			}
			set{
				this.SetValue(TempSettingsProperty, value);
			}
		}
		
		public static bool? Show(ImageBrushSettings settings, string title){
			return Show(settings, title, null);
		}
		public static bool? Show(ImageBrushSettings settings, string title, Window owner){
			var form = new ImageBrushSettingsForm(settings);
			form.Owner = owner;
			form.Title = title;
			return form.ShowDialog();
		}
	}
	
	public static class TextBoxBehaviors{
		public static readonly DependencyProperty IsSelectAllOnFocusProperty =
			DependencyProperty.RegisterAttached("IsSelectAllOnFocus", typeof(bool), typeof(TextBoxBehaviors), new UIPropertyMetadata(false, IsSelectAllOnFocusChanged));
		
		[AttachedPropertyBrowsableForType(typeof(TextBoxBase))]
		public static bool GetIsSelectAllOnFocus(DependencyObject obj){
			return (bool)obj.GetValue(IsSelectAllOnFocusProperty);
		}
		
		[AttachedPropertyBrowsableForType(typeof(TextBoxBase))]
		public static void SetIsSelectAllOnFocus(DependencyObject obj, bool value){
			obj.SetValue(IsSelectAllOnFocusProperty, value);
		}
		
		private static void IsSelectAllOnFocusChanged(DependencyObject sender, DependencyPropertyChangedEventArgs e){
			TextBoxBase textBox = (TextBoxBase)sender;
			
			bool newValue = (bool)e.NewValue;
			bool oldValue = (bool)e.OldValue;
			if(oldValue){
				textBox.GotFocus -= TextBox_GotFocus;
			}
			if(newValue){
				textBox.GotFocus += TextBox_GotFocus;
			}
		}
		
		private static void TextBox_GotFocus(object sender, RoutedEventArgs e){
			TextBoxBase textBox = (TextBoxBase)sender;
			textBox.Dispatcher.BeginInvoke(DispatcherPriority.Render, new Action(delegate{
				textBox.SelectAll();
			}));
		}
	}
	
	public class UriToStringConverter : IValueConverter{
		public object Convert(object value, Type targetType, object parameter, System.Globalization.CultureInfo culture){
			var uri = value as Uri;
			if(uri != null){
				return uri.LocalPath;
			}else{
				return null;
			}
		}
		
		public object ConvertBack(object value, Type targetType, object parameter, System.Globalization.CultureInfo culture){
			return new Uri((string)value, UriKind.RelativeOrAbsolute);
		}
	}
}