/* Copyright (c) 2007 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

//
//  YouTubeClientMain.m
//

#import <Cocoa/Cocoa.h>
#import "YouTubeClientMain.h"
#import "GData/GData.h"
#import "GData/GDataServiceGoogleYouTube.h"
#import "GData/GDataQueryYouTube.h"
#import "GData/GDataEntryYouTubeVideo.h"
#import "GData/GDataFeedYouTubeVideo.h"
#import "GData/GDataEntryYouTubeUserProfile.h"


// The name of the server hosting the YouTube GDATA feeds
NSString *const kGDataServerYouTube = @"http://gdata.youtube.com";

// The prefix common to all standard feeds
NSString *const kStandardFeedPrefix = @"http://gdata.youtube.com/feeds/standardfeeds/";

// The URL of the "Most Recent" feed
NSString *const kMostRecentFeed = @"http://gdata.youtube.com/feeds/standardfeeds/most_recent";

// The URL of the "Top Rated" feed
NSString *const kTopRatedFeed = @"http://gdata.youtube.com/feeds/standardfeeds/top_rated";

// The URL of the "Most Viewed" feed
NSString *const kMostViewedFeed = @"http://gdata.youtube.com/feeds/standardfeeds/most_viewed";

// The URL of the "Recently Featured" feed
NSString *const kRecentlyFeaturedFeed = @"http://gdata.youtube.com/feeds/standardfeeds/recently_featured";

// The URL of the "Watch On Mobile" feed
NSString *const kWatchOnMobileFeed = @"http://gdata.youtube.com/feeds/standardfeeds/watch_on_mobile";

// The URL of the "Videos" feed
NSString *const kVideosFeed = @"http://gdata.youtube.com/feeds/videos";

// The prefix of the User Feeds
NSString *const kUserFeedPrefix = @"http://gdata.youtube.com/feeds/users/";

// The URL suffix of the test user's uploads feed
NSString *const kUploadsFeedSuffix = @"/uploads";

// The URL suffix of the test user's favorites feed
NSString *const kFavoritesFeedSuffix = @"/favorites";

// The URL suffix of the test user's subscriptions feed
NSString *const kSubscriptionsFeedSuffix = @"/subscriptions";

// The URL suffix of the test user's playlists feed
NSString *const kPlaylistsFeedSuffix = @"/playlists";

// The default user if the user does not enter one.
// TEST
//NSString *kDefaultTestUser = @"YTdebates";
NSString *kDefaultTestUser = @"yama3tomato";


@interface YouTubeClient (Private)

- (void)printVideoFeed:(NSString *)feedUrlString;
- (void)printResponsesFeed:(GDataEntryYouTubeVideo *)videoEntry;
- (void)printCommentsFeed:(GDataEntryYouTubeVideo *)videoEntry;

@end

@implementation YouTubeClient 

- (id)init {
  self = [super init];
  if (self) {
    service_ = [[GDataServiceGoogleYouTube alloc] init];
  }
  return self;
}

- (void)dealloc {
  [super dealloc];
  [service_ release];
}

// deleteResource calls don't return data or an error, so we'll use
// a global int for the callbacks to increment to say they're done
// (because NSURLConnection is performing the fetches, all this 
// will be safely executed on the same thread)
static int gFetchCounter = 0; 

- (void)waitForFetch {
  
  int fetchCounter = gFetchCounter;
  
  // Give time for the fetch to happen, but give up if 
  // 10 seconds elapse with no response
  NSDate* giveUpDate = [NSDate dateWithTimeIntervalSinceNow:10.0];
  
  while ((!fetchedObject_ && !fetcherError_) 
         && fetchCounter == gFetchCounter
         && [giveUpDate timeIntervalSinceNow] > 0) {
    
    NSDate *stopDate = [NSDate dateWithTimeIntervalSinceNow:0.001];
    [[NSRunLoop currentRunLoop] runUntilDate:stopDate]; 
  }  
}

- (void)resetFetchResponse {
  [fetchedObject_ release];
  fetchedObject_ = nil;
  
  [fetcherError_ release];
  fetcherError_ = nil;
  
  [ticket_ release];
  ticket_ = nil;
}

// successfully
- (void)fetchTicketDidEnd:(GDataServiceTicket *)ticket
               finishedWithFeed:(GDataFeedBase *)object {
  fetchedObject_ = [object retain]; // save the fetched object
  ++gFetchCounter;
}

// failed
- (void)fetchTicketDidEnd:(GDataServiceTicket *)ticket
                failedWithError:(NSError *)error {
    NSLog(@"failed");
  fetcherError_ = [error retain]; // save the error
  ++gFetchCounter;
}

/**
 * Prints a list of all standard feeds.
 *
 */
- (void)printStandardFeeds 
{
	[self printVideoFeed:kMostViewedFeed];
//	[self printVideoFeed:kTopRatedFeed];
//	[self printVideoFeed:kRecentlyFeaturedFeed];
//	[self printVideoFeed:kWatchOnMobileFeed];
}

/**
 * Prints a String, a newline, and a number of '-' characters equal to the
 * String's length.
 *
 * @param feedTitle - the title to print underlined
 */
- (void)printUnderlined:(NSString *)feedTitle 
{
    NSLog(feedTitle);
	NSString *underline = @"";
    for (int i = 0; i < [feedTitle length]; ++i) {
		underline = [underline stringByAppendingString:@"-"];
	}
    NSLog(underline);
}

/**
 * Prints a VideoEntry, optionally showing its responses and comment feeds.
 *
 * @param prefix                   a string to be shown before each entry
 * @param videoEntry               the GDataEntryYouTubeVideo to be printed
 */
- (void)printVideoEntry:(GDataEntryYouTubeVideo *)videoEntry prefix:(NSString *)prefix 
{
    NSLog(prefix);
    if ([videoEntry title] != nil) {
		NSLog(@"Title: %@", [[videoEntry title] stringValue]);
    }
    if ([videoEntry summary] != nil) {
		NSLog(@"Summary: %@", [[videoEntry summary] stringValue]);
    }
    GDataMediaGroup *mediaGroup = [videoEntry mediaGroup];
    if (mediaGroup != nil) {
		// still unsupported: MediaPlayer
		// GDataMediaPlayer *mediaPlayer = [mediaGroup mediaPlayer];
		// NSLog(@"Web Player URL: %@", mediaPlayer);
		
		NSString *keywords = @"";
		GDataMediaKeywords *mediaKeywords = [mediaGroup mediaKeywords];
		NSEnumerator *keywordsEtor = [[mediaKeywords keywords] objectEnumerator];
		NSString *keyword;
		while ((keyword = [keywordsEtor nextObject]) != nil) {
			keywords = [keywords stringByAppendingString:keyword];
			keywords = [keywords stringByAppendingString:@", "];
		}
		NSLog(@"\tKeywords: %@", keywords);
		
		NSLog(@"\tThumbnails:");
		NSArray *mediaThumbnails = [mediaGroup mediaThumbnails];
		NSEnumerator *mediaEtor = [mediaThumbnails objectEnumerator];
		GDataMediaThumbnail *thumbnail;
		while ((thumbnail = [mediaEtor nextObject]) != nil) {
			NSLog(@"\t\tThumbnail URL: %@", [thumbnail URLString]);
			NSLog(@"\t\tThumbnail Time Index: %@", [[thumbnail time] HHMMSSString]);
			NSLog(@"");
		}
		
		NSLog(@"\tMedia:");
		NSArray *mediaContents = [mediaGroup mediaContents];
		NSEnumerator *contentsEtor = [mediaContents objectEnumerator];
		GDataMediaContent *content;
		while ((content = [contentsEtor nextObject]) != nil) {
        	NSLog(@"\t\tMedia Location: %@", [content URLString]);
        	NSLog(@"\t\tMedia Type: %@", [content type]);
        	NSLog(@"\t\tDuration: %d", [[content duration] intValue]);
			NSLog(@"");
		}
		NSLog(@"");
	}
/*    if (showCommentsAndResponses) {
//		printResponsesFeed(videoEntry);
		NSLog(@"");
//		printCommentsFeed(videoEntry);
		NSLog(@"");
		NSLog(@"");
    }
*/
}

/**
 * Prints the responses feed of a GDataEntryYouTubeVideo.
 *
 * @param videoEntry the GDataEntryYouTubeVideo whose responses are to be printed
 */
- (void)printResponsesFeed:(GDataEntryYouTubeVideo *)videoEntry {
/*    if ([videoEntry videoResponsesLink] != nil) {
		NSString *videoResponsesFeedUrl = [[videoEntry videoResponsesLink] href];
		NSLog(@"");
		printVideoFeed(videoResponsesFeedUrl, NO);
    }
*/
}

/**
 * Prints the comments feed of a GDataEntryYouTubeVideo.
 *
 * @param videoEntry the GDataEntryYouTubeVideo whose comments are to be printed
 */
- (void)printCommentsFeed:(GDataEntryYouTubeVideo *)videoEntry {
/*
    GDataComment *comments = [videoEntry comments];
    if (comments != nil && [comments feedLink] != nil) {
		NSLog(@"\tComments:");
		printFeed([[comments feedLink] href], "Comment");
    }
*/
}

/**
 * Prints a basic feed, such as the comments or responses feed, retrieved from
 * a feedUrlString.
 *
 * @param feedUrlString the url of the feed
 * @param prefix  a prefix string to be printed in front of each entry field
 * @throws ServiceException
 *                     If the service is unable to handle the request.
 * @throws IOException error sending request or reading the feed.
 */
/*
- (void)printFeed(NSString *feedUrlString, NSString *prefix) {
    GDataFeed *feed = [service() feed(feedUrlString), Feed.class);
	
    for (Entry e : feed.getEntries()) {
		printEntry(e, prefix);
    }
}

- (void)printFeed(NSString *feedUrlString, NSString *prefix) {
	NSURL *feedURL = [NSURL URLWithString:feedUrlString];
	GDataServiceTicket *ticket = [service() fetchYouTubeFeedWithURL:feedURL delegate:self didFinishSelector:@selector(feedFetchTicket:finishedWithFeed:) didFailSelector:@selector(vfeedFetchTicket:failedWithError:)];
}

// finished feed successfully
- (void)feedFetchTicket:(GDataServiceTicket *)ticket
               finishedWithFeed:(GDataFeedBase *)object {
	GDataFeedYouTubeVideo *videoFeed = object;
    NSString *title = [videoFeed title];
    printUnderlined(title);
}

// failed
- (void)feedFetchTicket:(GDataServiceTicket *)ticket
                failedWithError:(NSError *)error {
	// notihing
}
*/

/**
 * Prints a basic Entry, usually from a comments or responses feed.
 *
 * @param entry      the entry to be printed
 * @param prefix a prefix to be printed before each entry attribute
 */
/*
- (void)printEntry(Entry entry, NSString *prefix) {
    NSLog(@"\t\t" + prefix + " Title: " 
		  + entry.getTitle().getPlainText());
    if (entry.getContent() != nil) {
		TextContent content = (TextContent) entry.getContent();
		NSLog(@"\t\t" + prefix + " Content: " 
			  + content.getContent().getPlainText());
    }
    NSLog(@"\t\tURL: " + entry.getHtmlLink().getHref());
}
*/

/**
 * Prints a PlaylistEntry by retrieving the Description of the PlayList,
 * followed by the Titles and URLs of each entry in the feed.
 *
 * @param prefix               a string to be printed before each entry
 * @param playlistLinkEntry    the PlaylistEntry to be printed
 * @param showPlaylistContents if YES, show the list of videos in the
 *                             playlist
 * @throws ServiceException
 *                             If the service is unable to handle the
 *                             request.
 * @throws IOException error sending request or reading the feed.
 */
/*
- (void)printPlaylistEntry(NSString *prefix,
							   PlaylistLinkEntry playlistLinkEntry, 
							   BOOL showPlaylistContents) {
    NSLog(prefix);
    NSLog(@"Description: %s\n", playlistLinkEntry.getDescription());
    if (showPlaylistContents) {
		FeedLink feedLink = playlistLinkEntry.getFeedLink();
		printPlaylist(playlistLinkEntry.getService(), feedLink.getHref());
    }
}
 */

/**
 * Prints a playlist feed as a series of Titles and URLs.
 *
 * @param service     a GDataServiceGoogleYouTube object
 * @param playlistUrl the url of the playlist
 * @throws ServiceException
 *                     If the service is unable to handle the request.
 * @throws IOException error sending request or reading the feed.
 */
/*
- (void)printPlaylist(Service service, NSString *playlistUrl) {
    PlaylistFeed playlistFeed = service.getFeed(new URL(playlistUrl),
												PlaylistFeed.class);
    if (playlistFeed != nil) {
		for (PlaylistEntry e : playlistFeed.getEntries()) {
			NSLog(@"\tPlaylist Entry: " + e.getTitle().getPlainText());
			NSLog(@"\tPlaylist URL: " + e.getHtmlLink().getHref());
		}
    }
}
 */

/**
 * Prints a FeedLinkEntry as a Title and URL String.
 *
 * @param feedLinkEntry the FeedLinkEntry to be printed
 */
/*
- (void)printFeedLinkEntry(FeedLinkEntry feedLinkEntry) {
    if (feedLinkEntry.getTitle() != nil) {
		NSLog(@"Title: %s\n", feedLinkEntry.getTitle().getPlainText());
    }
    NSLog(@"FeedLink: " + feedLinkEntry.getFeedLink().getHref());
}
 */

/**
 * Prints a SubscriptionEntry, which is a FeedLink entry.
 *
 * @param subEntry the SubscriptionEntry to be printed
 */
/*
- (void)printSubscriptionEntry(SubscriptionEntry subEntry) {
    printFeedLinkEntry(subEntry);
}
 */

/**
 * Reads a line of text from the standard input.
 * 
 * @return a line of text read from the standard input
 */
static NSString *readLine() {
	char buf[2];
	fgets(buf, 2, stdin);
	return [NSString stringWithCString:buf];
}

/**
 * Reads a line of text from the standard input, and returns the parsed
 * integer representation.
 * 
 * @return an integer read from the standard input
 */
static int readInt() {
	NSString *input = readLine();
	return [input intValue];
}

static void printUsage() {
	NSLog(@"Choose one of the following demo options:");
	NSLog(@"\t1) Print Standard Feeds");
	NSLog(@"\t2) Print Search Feed");
	NSLog(@"\t3) Print Keyword Search Feed");
	NSLog(@"\t4) Print Uploads Feed");
	NSLog(@"\t5) Print Favorites Feed");
	NSLog(@"\t6) Show Comments and Responses Feed for a single Video");
	NSLog(@"\t7) Print Playlists Feed");
	NSLog(@"\t8) Display a playlist");
	NSLog(@"\t9) Print Subscriptions Feed");
	NSLog(@"\t10) Print User Profile Feed");
	NSLog(@"\t0) Exit");
	NSLog(@"\nEnter Number (0-10): ");
}

/**
 * Prompts the user to enter his user name on the standard input.
 * @return a username entered by the user
 */
NSString *promptForUser() {
    NSLog(@"\nEnter YouTube username [default %@ ]: ", kDefaultTestUser);
// TEST
//    NSString *line = readLine();
    NSString *line = @"";
	line = [line stringByTrimmingCharactersInSet:[NSCharacterSet whitespaceCharacterSet]];
    if (line == nil || [line isEqualToString:@""]) {
		line = kDefaultTestUser;
    } else {
		kDefaultTestUser = line;
    }
    return line;
}

/**
 * Fetchs a user's profile feed and prints out most of the attributes.
 */
- (void)printUserProfile {
    NSString *testUser = promptForUser();
    NSString *title = [NSString stringWithFormat:@"User Profile for '%@'", testUser];
    [self printUnderlined:title];

	NSString *feedUrlString = [kUserFeedPrefix stringByAppendingString:testUser];
	NSURL *feedURL = [NSURL URLWithString:feedUrlString];
	GDataServiceTicket *ticket = [service_ fetchYouTubeFeedWithURL:feedURL 
		delegate:self 
		didFinishSelector:@selector(fetchTicketDidEnd:finishedWithFeed:) 
		didFailSelector:@selector(fetchTicketDidEnd:failedWithError:)];
	ticket_ = [ticket retain];

	[self waitForFetch];
  
	GDataEntryYouTubeUserProfile *userProfileEntry = (GDataEntryYouTubeUserProfile *)fetchedObject_;

    NSLog(@"Username: %@", [userProfileEntry username]);
    NSLog(@"Age     : %@", [userProfileEntry age]);
    NSLog(@"Gender  : %@", [userProfileEntry gender]);
    NSLog(@"Single? : %@", [userProfileEntry relationship]);
    NSLog(@"Books   : %@", [userProfileEntry books]);
    NSLog(@"Company : %@", [userProfileEntry company]);
    NSLog(@"Describe: %@", [userProfileEntry description]);
    NSLog(@"Hobbies : %@", [userProfileEntry hobbies]);
    NSLog(@"Hometown: %@", [userProfileEntry hometown]);
    NSLog(@"Location: %@", [userProfileEntry location]);
    NSLog(@"Movies  : %@", [userProfileEntry movies]);
    NSLog(@"Music   : %@", [userProfileEntry music]);
    NSLog(@"Job     : %@", [userProfileEntry occupation]);
    NSLog(@"School  : %@", [userProfileEntry school]);

	[self resetFetchResponse];
}

/**
 * Prints a user's playlists feed.
 *
 * @param service              a GDataServiceGoogleYouTube object.
 * @param showPlaylistContents if YES, print only the first playlist,
 *                             followed by all of its contained entries
 * @throws ServiceException
 *                     If the service is unable to handle the request.
 * @throws IOException error sending request or reading the feed.
 */
- (void)printPlaylists {
/*
    NSString *testUser = promptForUser();
    printPlaylistsFeed(service, USER_FEED_PREFIX + testUser
					   + PLAYLISTS_FEED_SUFFIX, showPlaylistContents);
 */
}

/**
 * Prints a user's subscriptions feed.
 * 
 * @throws ServiceException If the service is unable to handle the request.
 * @throws IOException error sending request or reading the feed.
 */
- (void)printSubscriptions {
/*
    NSString *testUser = promptForUser();
    printSubscriptionsFeed(service, USER_FEED_PREFIX + testUser 
						   + SUBSCRIPTIONS_FEED_SUFFIX);
 */
}

/**
 * Prints a user's favorites feed.
 */
- (void)printFavorites {
    NSString *testUser = promptForUser();
	NSString *feedURL = [NSString stringWithFormat:@"%@%@%@", kUserFeedPrefix, testUser, kFavoritesFeedSuffix];
    [self printVideoFeed:feedURL];
}

/**
 * Prints a user's uploads feed.
 */
- (void)printUploads {
    NSString *testUser = promptForUser();
	NSString *feedURL = [NSString stringWithFormat:@"%@%@%@", kUserFeedPrefix, testUser, kUploadsFeedSuffix];
    [self printVideoFeed:feedURL];
}

/**
 * Fetches a feed known to be a VideoFeed, printing each GDataEntryYouTubeVideo with in
 * a numbered list, optionally prompting the user for the number of a video
 * entry which should have its comments and responses printed.
 *
 * @param feedUrlString the url of the video feed to print
 */
- (void)printVideoFeed:(NSString *)feedUrlString {
	NSURL *feedURL = [NSURL URLWithString:feedUrlString];
	GDataServiceTicket *ticket = [service_ fetchYouTubeFeedWithURL:feedURL 
		delegate:self 
		didFinishSelector:@selector(fetchTicketDidEnd:finishedWithFeed:) 
		didFailSelector:@selector(fetchTicketDidEnd:failedWithError:)];
	ticket_ = [ticket retain];

	[self waitForFetch];
  
	GDataFeedYouTubeVideo *videoFeed = (GDataFeedYouTubeVideo *)fetchedObject_;
    NSString *title = [[videoFeed title] stringValue];
    [self printUnderlined:title];

	NSArray *videoEntries = [videoFeed entries];
	NSEnumerator *etor = [videoEntries objectEnumerator];
	id entry = nil;
    int count = 1;
	while ((entry = [etor nextObject]) != nil) {
		NSString *prefix = [NSString stringWithFormat:@"(Video #%d)", count];
		[self printVideoEntry:entry prefix:prefix];
		count++;
	}
    NSLog(@"");

	[self resetFetchResponse];
}

/**
 * Prints a user's playlists feed.
 *
 * @param service              a GDataServiceGoogleYouTube object
 * @param feedUrlString              the url of the feed
 */
- (void)printPlaylistsFeed:(NSString *)feedUrlString {
/*
    PlaylistLinkFeed playlistLinkFeed = service.getFeed(new URL(feedUrlString),
														PlaylistLinkFeed.class);
    NSString *title = playlistLinkFeed.getTitle().getPlainText();
    if (showPlaylistContents) {
		title += " with playlist content.");
    }
    printUnderlined(title);
    List<PlaylistLinkEntry> playlistEntries = playlistLinkFeed.getEntries();
    int count = 1;
    for (PlaylistLinkEntry pe : playlistEntries) {
		printPlaylistEntry("(Playlist #" + count + ")", pe, NO);
		count++;
    }
    if (showPlaylistContents) {
		NSLog(@"\nWhich playlist do you want to see? (1-%d): \n",
			   count - 1);
		int whichVideo = readInt();
		printPlaylistEntry("", playlistEntries.get(whichVideo - 1), YES);
    }
    NSLog();
 */
}

/**
 * Prints a user's subscriptions feed.
 * 
 * @param feedUrlString the url of the feed
 */
- (void)printSubscriptionsFeed:(NSString *)feedUrlString 
{
/*
    SubscriptionFeed subscriptionFeed = service.getFeed(new URL(feedUrlString),
														SubscriptionFeed.class);
    printUnderlined(subscriptionFeed.getTitle().getPlainText());
    for (SubscriptionEntry se : subscriptionFeed.getEntries()) {
		printSubscriptionEntry(se);
    }
    NSLog();
 */
}

/**
 * Searches the VIDEOS_FEED for search terms and print each resulting
 * GDataEntryYouTubeVideo.
*/
- (void)searchFeed {
	NSURL *feedURL = [NSURL URLWithString:kVideosFeed];
	GDataQueryYouTube *query = [GDataQueryYouTube youTubeQueryWithFeedURL:feedURL];

    // order results by the number of views (most viewed first)
//	[query setOrderBy:kGDataYouTubeOrderByViewCount];
	[query setOrderBy:@"viewCount"];

    // do not exclude restricted content from the search results 
    // (by default, it is excluded) 
    [query setIncludeRacy:YES];

    NSLog(@"\nEnter search terms: ");
//    NSString *searchTerms = readLine();
    NSString *searchTerms = @"nuko";
	
	[query setVideoQuery:searchTerms];
	
    NSString *title = [NSString stringWithFormat:@"Running Search for '%@'", searchTerms];
    [self printUnderlined:title];
	
	GDataServiceTicket *ticket = [service_ fetchYouTubeQuery:query 
		delegate:self 
		didFinishSelector:@selector(fetchTicketDidEnd:finishedWithFeed:) 
		didFailSelector:@selector(fetchTicketDidEnd:failedWithError:)];
	ticket_ = [ticket retain];

	[self waitForFetch];

	GDataFeedYouTubeVideo *videoFeed = (GDataFeedYouTubeVideo *)fetchedObject_;
	
	NSArray *videoEntries = [videoFeed entries];
	NSEnumerator *etor = [videoEntries objectEnumerator];
	id entry = nil;
    int count = 1;
	while ((entry = [etor nextObject]) != nil) {
		NSString *prefix = [NSString stringWithFormat:@"(Video #%d)", count];
		[self printVideoEntry:entry prefix:prefix];
		count++;
	}
    NSLog(@"");
}

/**
 * Searches the VIDEOS_FEED for category keywords and prints each resulting
 * GDataEntryYouTubeVideo.
 */
- (void)searchFeedWithKeywords {
	NSURL *feedURL = [NSURL URLWithString:kVideosFeed];
	GDataQueryYouTube *query = [GDataQueryYouTube youTubeQueryWithFeedURL:feedURL];

    // order results by the number of views (most viewed first)
//	[query setOrderBy:kGDataYouTubeOrderByViewCount];
	[query setOrderBy:@"viewCount"];

    // do not exclude restricted content from the search results 
    // (by default, it is excluded) 
    [query setIncludeRacy:YES];

    // a category filter holds a collection of categories to limit the search
    GDataCategoryFilter *categoryFilter = [GDataCategoryFilter categoryFilter];
	
    NSString *keywordTerm = nil;
    NSString *allKeywords = @"";
	
// TEST
//    do {
		NSLog(@"\nEnter keyword or empty line when done: ");
// TEST
//		keywordTerm = readLine();
		keywordTerm = @"test";
      // creates categories whose scheme is kGDataSchemeKeyword
//		GDataCategory *category = [GDataCategory categoryWithScheme:kGDataSchemeKeyword term:keywordTerm];
		GDataCategory *category = [GDataCategory categoryWithScheme:nil term:keywordTerm];
										 
		[categoryFilter addCategory:category];
      // keeps track of concatenated list of keywords entered so far
		if (![keywordTerm isEqualToString:@""]) {
			allKeywords = [allKeywords stringByAppendingString:keywordTerm];
			allKeywords = [allKeywords stringByAppendingString:@", "];
		}
  //  } while (![keywordTerm isEqualToString:@""]);
	
    // adds the collection of keyword categories to the query
    [query addCategoryFilter:categoryFilter];
	
    NSString *title = [NSString stringWithFormat:@"Running Search for '%@'", allKeywords];
    [self printUnderlined:title];

	GDataServiceTicket *ticket = [service_ fetchYouTubeQuery:query 
		delegate:self 
		didFinishSelector:@selector(fetchTicketDidEnd:finishedWithFeed:) 
		didFailSelector:@selector(fetchTicketDidEnd:failedWithError:)];
	ticket_ = [ticket retain];

	[self waitForFetch];

	GDataFeedYouTubeVideo *videoFeed = (GDataFeedYouTubeVideo *)fetchedObject_;
	
	NSArray *videoEntries = [videoFeed entries];
	NSEnumerator *etor = [videoEntries objectEnumerator];
	id entry = nil;
    int count = 1;
	while ((entry = [etor nextObject]) != nil) {
		NSString *prefix = [NSString stringWithFormat:@"(Video #%d)", count];
		[self printVideoEntry:entry prefix:prefix];
		count++;
	}
    NSLog(@"");
}

@end

/**
 * YouTubeClient is a sample command line application that demonstrates
 * many features of the YouTube data API using the Java Client library.
 *
 * @param args not used
 */
int main(int argc, char*argv[]) 
{
	NSAutoreleasePool*ap = [[NSAutoreleasePool alloc] init];

	YouTubeClient *client = [[YouTubeClient alloc] init];
	
	BOOL exit = NO;
	
    while (!exit) {
        printUsage();
// TEST
 //       int choice = readInt();
        int choice = 10;
		
        switch (choice) {
			case 1:
				// Fetches and prints the standard feeds.
				[client printStandardFeeds];
				break;
			case 2:
				// Searches the VIDEO_FEED for user supplied search terms.
				[client searchFeed];
				break;
			case 3:
				// Searches the VIDEO_FEED for user supplied category keyword 
				// terms.
				[client searchFeedWithKeywords];
				break;
			case 4:
				// Fetches and prints a user's uploads feed.
				[client printUploads];
				break;
			case 5:
				// Fetches and prints a user's favorites feed.
				[client printFavorites];
				break;
			case 6:
				// Prompts the user for a username and displays the uploads 
				// feed for that username as a numbered list. The user is asked
				// to choose the number of a video entry for which the comments
				// and responses feeds should be displayed.
				[client printUploads];
				break;
			case 7:
				// Fetches and prints a list of a user's playlists feed.
				[client printPlaylists];
				break;
			case 8:
				// Fetches and prints a numbered list of entries in a user's
				// playlists feed. The user is then asked to choose the number
				// of a playlist he wishes to see the contents of.
				[client printPlaylists];
				break;
			case 9:
				// Fetches and prints a user's subscriptions feed.
				[client printSubscriptions];
				break;
			case 10:
				// Fetches and prints a user's profile feed.
				[client printUserProfile];
				break;
			case 0:
			default:
				exit = YES;
				break;
        }
// TEST
        readInt();
	}
	
	[client release];
	
	[ap release];
}
