/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package jp.sourceforge.damstation_dl;

import java.io.IOException;
import java.io.Reader;
import java.text.ParseException;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.swing.text.MutableAttributeSet;
import javax.swing.text.html.HTML;
import javax.swing.text.html.HTML.Tag;
import javax.swing.text.html.HTMLEditorKit.ParserCallback;
import javax.swing.text.html.parser.ParserDelegator;
import jp.sourceforge.damstation_dl.data.ResultData;
import jp.sourceforge.damstation_dl.data.ResultDate;
import jp.sourceforge.damstation_dl.data.SongData;
import jp.sourceforge.damstation_dl.data.SongId;

/**
 *
 * @author ｄ
 */
public class SeimitsuPageParser {

    /** 正常時のTD要素の出現回数 */
    private static final int TD_ELEMENT_COUNT = 23;
    /** 日付取得用のパターン */
    private static final Pattern DATE_PATTERN = Pattern.compile("\\d{4}/\\d{2}/\\d{2} \\d{2}:\\d{2}:\\d{2}");
    /** ファイル名(拡張子なし)取得用のパターン */
    private static final Pattern FILENAME_PATTERN = Pattern.compile("/(.)+?[.]gif$");
    /** ビブラート時間取得用のパターン */
    private static final Pattern VIBRATO_TIME_PATTERN = Pattern.compile("^[0-9.]+");
    /** 必要な情報の領域の始まりを見分けるためのクラス名 */
    private static final String SEARCH_CLASSNAME = new String("list_marking");
    
    private Map<SongId, SongData> songDataList = new HashMap<SongId, SongData>();;
    private Map<ResultDate, ResultData> resultDataList = new HashMap<ResultDate, ResultData>();
    
    /**
     * 精密データのパーサ
     */
    private class SeimitsuPageParserCallback extends ParserCallback {
        
        /** matcher 正規表現の結果 */
        private Matcher matcher = null;
        /** img単独タグ検出回数 */
        private int imgCount = 0;
        /** TD開始タグ検出回数 */
        private int tdCount = 0;
        /** テキストノード出現回数(TD毎にリセット) */
        private int textCount = 0;
        /** 現在の読み込み位置が曲名かどうか */
        private boolean isTitle = false;
        /** 現在の読み込み位置が歌手名かどうか */
        private boolean isArtist = false;
        /** 現在、必要な情報の領域にはいっているかどうか */
        private boolean isChecking = false;
        
        /** 現在読み取り中のデータ(完了したらリストに追加する) */
        private String id;
        private String title;
        private String artist;
        private String date;
        private String score;
        private String vibratoType;
        private String vibratoTime;
        private String shakuri;
        private String interval;
        private String rhythm;
        private String modulation;


        /**
         * パース完了時のコールバック関数
         * @param eol
         */
        @Override
        public void handleEndOfLineString(String eol) {
            super.handleEndOfLineString(eol);
            this.matcher = null;
        }

        /**
         * 終了タグ検出時のコールバック関数
         * @param t タグの種類
         * @param pos 開始位置
         */
        @Override
        public void handleEndTag(Tag t, int pos) {
            super.handleEndTag(t, pos);
            if (!this.isChecking) {
                return;
            }

            if (Tag.HTML.TABLE.equals(t)) {
                this.endCheck();
            } else if (Tag.HTML.A.equals(t) && this.isTitle) {
                this.isTitle = false;
            } else if (Tag.HTML.P.equals(t) && this.isArtist) {
                this.isArtist = false;
            } else if (Tag.HTML.TD.equals(t)) {
                this.textCount = 0;
            }
        }

        /**
         * 単独タグ検出時のコールバック関数
         * @param t タグの種類
         * @param a 属性
         * @param pos 開始位置
         */
        @Override
        public void handleSimpleTag(Tag t, MutableAttributeSet a, int pos) {
            super.handleSimpleTag(t, a, pos);
            if (!this.isChecking) {
                return;
            }
            
            if (Tag.HTML.IMG.equals(t)) {
                this.imgCount++;

                switch (this.imgCount) {
                    case 1:
                        // リズム
                        this.matcher = FILENAME_PATTERN.matcher((String) a.getAttribute(HTML.Attribute.SRC));
                        if (this.matcher.find() && (this.matcher.groupCount() >= 1)) {
                            this.rhythm = this.matcher.group(1);
                        }
                        break;
                    case 2:
                        // 抑揚
                        this.matcher = FILENAME_PATTERN.matcher((String) a.getAttribute(HTML.Attribute.SRC));
                        if (this.matcher.find() && (this.matcher.groupCount() >= 1)) {
                            this.modulation = this.matcher.group(1);
                        }
                        break;
                    default:
                        break;
                }
            }
        }

        /**
         * 開始タグ検出時のコールバック関数
         * @param t タグの種類
         * @param a 属性
         * @param pos 開始位置
         */
        @Override
        public void handleStartTag(Tag t, MutableAttributeSet a, int pos) {
            super.handleStartTag(t, a, pos);
            if (Tag.HTML.TABLE.equals(t) && SEARCH_CLASSNAME.equals(a.getAttribute(HTML.Attribute.CLASS))) {
                this.startCheck();
            }

            if (this.isChecking) {
                if (Tag.HTML.TD.equals(t)) {
                    this.tdCount++;
                } else if (Tag.HTML.A.equals(t) && (this.tdCount == 2)) {
                    this.isTitle = true;
                } else if (Tag.HTML.P.equals(t) && (this.tdCount == 2)) {
                    this.isArtist = true;
                }
            }
        }

        /**
         * テキストノード検出時のコールバック関数(前後の空白などは無視)
         * @param data テキストノードの文字列
         * @param pos 開始位置
         */
        @Override
        public void handleText(char[] data, int pos) {
            super.handleText(data, pos);
            if (!this.isChecking) {
                return;
            }
            
            this.textCount++;
            switch (this.tdCount) {
                case 1:
                    // 日時
                    this.matcher = DATE_PATTERN.matcher(new String(data));
                    if (this.matcher.find()) {
                        try {
                            this.date = ResultDate.damStationDateToXmlDateTime(this.matcher.group(0));
                        } catch(ParseException e) {
                            this.date = null;
                        }
                    }
                    break;
                case 2:
                    // 新曲の場合は曲名と歌手名が存在しないケースがある
                    // その場合に対処するため、isArtistとisTitleを使用している
                    if (this.isTitle) {
                        // 曲名
                        this.title = new String(data);
                    } else if (this.isArtist) {
                        // 歌手
                        this.artist = new String(data);
                    } else {
                        // ID
                        this.id = new String(data).replace("(", "").replace(")", "");
                    }
                    break;
                case 3:
                    if (this.textCount == 2) {
                        // 得点
                        this.score = new String(data).replace("点", "");
                    }
                    break;
                case 9:
                    switch (this.textCount) {
                        case 1:
                            // ビブラート時間
                            this.matcher = VIBRATO_TIME_PATTERN.matcher(new String(data));
                            if (this.matcher.find()) {
                                this.vibratoTime = this.matcher.group(0);
                            }
                            break;
                        case 2:
                            // ビブラートタイプ
                            this.vibratoType = new String(data);
                            break;
                    }
                    break;
                case 10:
                    // しゃくり
                    this.shakuri = new String(data).replace("回", "");
                    break;
                case 11:
                    // 音程
                    this.interval = new String(data).replace("%", "");
                    break;
                default:
                    break;
            }
        }

        /**
         * 読み込み位置が必要な情報の領域に入ったときの処理
         */
        private void startCheck() {
            // データの初期化
            this.imgCount = 0;
            this.tdCount = 0;
            this.textCount = 0;
            this.isTitle = false;
            this.isArtist = false;
            this.isChecking = true;
            
            this.id = null;
            this.title = null;
            this.artist = null;
            this.date = null;
            this.score = null;
            this.vibratoType = null;
            this.vibratoTime = null;
            this.shakuri = null;
            this.interval = null;
            this.rhythm = null;
            this.modulation = null;
        }

        /**
         * 読み込み位置が必要な情報の領域から抜けたときの処理
         */
        private void endCheck() {
            
            /*
            // テストコード
            System.out.println("tdCount: " + tdCount);
            System.out.println("id: " + this.id);
            System.out.println("title: " + this.title);
            System.out.println("artist: " + this.artist);
            System.out.println("date: " + this.date);
            System.out.println("score: " + this.score);
            System.out.println("vibratoType: " + this.vibratoType);
            System.out.println("vibratoTime: " + this.vibratoTime);
            System.out.println("shakuri: " + this.shakuri);
            System.out.println("interval: " + this.interval);
            System.out.println("rhythm: " + this.rhythm);
            System.out.println("modulation: " + this.modulation);
            */

            this.isChecking = false;
            
            // TD要素の数をチェックする
            if (this.tdCount == TD_ELEMENT_COUNT) {
                // NULLチェック
                if ((this.id == null) || (this.date == null) || (this.score == null)
                        || (this.vibratoType == null) || (this.vibratoTime == null)
                        || (this.shakuri == null) || (this.interval == null)
                        || (this.rhythm == null) || (this.modulation == null)) {
                    return;
                }
                
                // IDと日付の妥当性をチェックする
                if (!SongId.isValid(this.id) || !ResultDate.isValid(this.date)) {
                    return;
                }
                
                SongId songId = SongId.getInstance(this.id);
                // 重複データでなければ曲データに追加するための処理を開始する
                if (!songDataList.containsKey(songId)) {
                    
                    // 曲名かアーティストが取得できなかった場合(新曲の場合)は曲データに追加しない
                    if ((this.title != null) && (this.artist != null)) {
                        // 曲データを追加する
                        songDataList.put(songId, new SongData(this.title, this.artist));
                    }
                }
                
                ResultDate resultDate = ResultDate.getInstance(this.date);
                ResultData resultData = null;
       
                // 重複データでなければ結果データを追加するための処理を開始する
                if (!resultDataList.containsKey(resultDate)) {
                    try {

                        // 文字列から数値に変換する
                        double score = Double.parseDouble(this.score);
                        int vibratoType = SeimitsuPageParser.parseVibratoTypeInteger(this.vibratoType);
                        double vibratoTime = Double.parseDouble(this.vibratoTime);
                        int shakuri = Integer.parseInt(this.shakuri);
                        int interval = Integer.parseInt(this.interval);
                        int rhythm = Integer.parseInt(this.rhythm);
                        int modulation = Integer.parseInt(this.modulation);

                        // 各値の妥当性チェックを行う
                        if (ResultData.isValidScore(score) && ResultData.isValidVibratoType(vibratoType) && ResultData.isValidVibratoTime(vibratoTime) && ResultData.isValidShakuri(shakuri) && ResultData.isValidInterval(interval) && ResultData.isValidRhythm(rhythm) && ResultData.isValidModulation(modulation)) {
                            resultData = new ResultData(songId, score, vibratoType, vibratoTime, shakuri, interval, rhythm, modulation);

                            // 結果データを追加する
                            resultDataList.put(resultDate, resultData);
                        }
                    } catch (NumberFormatException e) {
                    } catch (ParseException e) {
                    }
                }
            }
        }
    }
    
    /**
     * A-1等のビブラートの種類を表す文字列を数値に変換する
     * @param vibratoType
     * @return
     * @throws java.text.ParseException
     */
    public static int parseVibratoTypeInteger(String vibratoType) throws ParseException {
        if (vibratoType == null) {
            throw new NullPointerException("SeimitsuPageParser.parseVibratoTypeInteger\n\tvibratoType=" + vibratoType);
        }
        
        if ("無し".equals(vibratoType)) {
            return 0;
        } else if ("A-1".equals(vibratoType)) {
            return 1;
        } else if ("A-2".equals(vibratoType)) {
            return 2;
        } else if ("A-3".equals(vibratoType)) {
            return 3;
        } else if ("B-1".equals(vibratoType)) {
            return 4;
        } else if ("B-2".equals(vibratoType)) {
            return 5;
        } else if ("B-3".equals(vibratoType)) {
            return 6;
        } else if ("C-1".equals(vibratoType)) {
            return 7;
        } else if ("C-2".equals(vibratoType)) {
            return 8;
        } else if ("C-3".equals(vibratoType)) {
            return 9;
        } else {
            // 意図しない文字列の場合は例外を返す
            throw new ParseException("SeimitsuPageParser.parseVibratoTypeInteger\n\tvibratoType=" + vibratoType, 0);
        }
    }
    
    /**
     * コンストラクタ
     * @param reader
     * @throws java.io.IOException
     */
    public SeimitsuPageParser(Reader reader) throws IOException {
        ParserDelegator parserDelegator = new ParserDelegator();
        ParserCallback parserCallback = new SeimitsuPageParserCallback();
        
        try {
            parserDelegator.parse(reader, parserCallback, true);
        } catch (IOException e) {
            throw e;
        }
    }
    
    /**
     * 解析した曲データリストを取得する
     * @return
     */
    public Map<SongId, SongData> getSongDataList() {
        return songDataList;
    }
    
    /**
     * 解析した結果データリストを取得する
     * @return
     */
    public Map<ResultDate, ResultData> getResultDataList() {
        return resultDataList;
    }
}
