/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp;

import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import net.morilib.lisp.subr.UnaryArgs;

public final class RegexPattern extends Datum {
	
	public static class Match extends Datum {
		
		//
		private Matcher match;
		//private int ptr = 0;
		
		
		private Match(Matcher x) {
			match = x;
		}
		
		
		public Datum group() {
			return new LispString(match.group());
		}
		
		
		public Datum group(int no) {
			String res = match.group(no);
			
			return (res != null) ? new LispString(res) : null;
		}
		
		
		/*private boolean hasNext() {
			return ptr <= match.groupCount();
		}
		
		
		private Datum next() {
			return group(ptr++);
		}
		
		
		private void remove() {
			throw new UnsupportedOperationException();
		}*/
		
		
		public String toStringRepl() {
			return match.toString();
		}
		
	}
	
	
	public final static class CompileRe extends UnaryArgs {
		
		@Override
		protected Datum execute(
				Datum c1a, Environment env, LispMessage mesg) {
			if(c1a instanceof LispString) {
				String  tok = ((LispString)c1a).getString();
				Matcher mch = REGEX_PAT.matcher(tok);
				
				if(!mch.matches()) {
					throw mesg.getError("err.regex.syntax", tok);
				}
				//mch.find();
				
				String  re  = mch.group(1);
				String  fl  = mch.group(2);
				
				try {
					return new RegexPattern(re, fl);
				} catch(PatternSyntaxException e) {
					throw mesg.getError("err.regex.syntax", tok);
				}
			} else {
				throw mesg.getError("err.require.string", c1a);
			}
		}
		
	}
	
	
	//
	private static final Pattern REGEX_PAT =
		Pattern.compile("/(.*)/([iuc]*)");
	
	//
	private Pattern pat;
	private String patternString;
	
	
	public RegexPattern(String regex, String fl) {
		int flg = 0;
		
		for(int i = 0; i < fl.length(); i++) {
			switch(fl.charAt(i)) {
			case 'u':
				flg |= Pattern.UNICODE_CASE;
				break;
			case 'i':
				flg |= Pattern.CASE_INSENSITIVE;
				break;
			case 'c':
				flg |= Pattern.CANON_EQ;
				break;
			default:
				throw new IllegalArgumentException(fl);
			}
		}
		pat = Pattern.compile(regex, flg);
		patternString = "#/" + regex + "/" + fl;
	}
	
	
	public Datum getMatch(String str) {
		Matcher mt = pat.matcher(str);
		
		return mt.matches() ? new Match(mt) : LispBoolean.FALSE;
	}
	
	
	public String getPatternString() {
		return patternString;
	}
	
}
