/*  esecannaserver --- pseudo canna server that wraps another IME.
 *  Copyright (C) 1999-2000 Yasuhiro Take
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <pwd.h>
#include <sys/types.h>
#include <signal.h>
#include <dlfcn.h>

#include "def.h"

#include "misc.h"

#define IW_ERROR16(_buf) { \
  cannaheader_t *he; \
  he = (cannaheader_t *)(_buf); \
  he->datalen = LSBMSB16(2); \
  he->err.e16 = LSBMSB16(-1); \
}

#define IW_ERROR8(_buf) { \
  cannaheader_t *he; \
  he = (cannaheader_t *)(_buf); \
  he->datalen = LSBMSB16(1); \
  he->err.e8 = -1; \
}

typedef int (*imefunc_t)();

static char *funcsymbols[] = {
  "_finalize",
  "_create_context",
  "_duplicate_context",
  "_close_context",
  "_define_word",
  "_delete_word",
  "_begin_convert",
  "_end_convert",
  "_get_candidacy_list",
  "_get_yomi",
  "_subst_yomi",
  "_store_yomi",
  "_store_range",
  "_get_lastyomi",
  "_flush_yomi",
  "_remove_yomi",
  "_get_simplekanji",
  "_resize_pause",
  "_get_hinshi",
  "_get_lex",
  "_get_status",
  "_set_locale",
  "_auto_convert",
  "_initialize",
  "_init_rootclient",
  "_end_client",
  "_end_rootclient",
  "_clear_client_data"
};

/* ʥߥå󥯤ؿֹȤбդ */
/*
  _finalize 0
  _create_context 1
  _duplicate_context 2
  _close_context 3
  _define_word 4
  _delete_word 5
  _begin_convert 6
  _end_convert 7
  _get_candidacy_list 8
  _get_yomi 9
  _subst_yomi 10
  _store_yomi 11
  _store_range 12
  _get_lastyomi 13
  _flush_yomi 14
  _remove_yomi 15
  _get_simplekanji 16
  _resize_pause 17
  _get_hinshi 18
  _get_lex 19
  _get_status 20
  _set_locale 21
  _auto_convert 22
  _initialize 23
  _init_rootclient 24
  _end_client 25
  _end_rootclient 26
  _clear_client_data 27
  */

#define F_initialize 23
#define F_init_rootclient 24
#define F_end_client 25
#define F_end_rootclient 26
#define F_clear_client_data 27

#define F_COMMON_END 27

#define VJE_clear_vjeid 28
#define VJE_restarted 29
#define VJE_connect_unix 30
#define VJE_close_connection 31

static imefunc_t vjefunc[32];
static imefunc_t atokfunc[28];
static imefunc_t wnnfunc[28];

static imefunc_t *imefunc[IME_END] = {vjefunc, vjefunc, atokfunc, wnnfunc};
static char ime_connected_flag[IME_END];
static void *ime_dl_handler[IME_END];

char *ime_dl_name[] = {
  "vje25.so", "vje30.so", "atok.so", "wnn6.so"
};

extern char *protocol_name[], *e_protocol_name[];
extern client_t client[];

/*
 * ե졼եɤߡɤ IME ³뤫֤
 */

static char *iw_get_conf_file_path(char *home)
{
  char *path;

  if ((path = m_makepath(home, ".esecannarc")) == NULL) {
    m_msg("out of memory!\n");
    return NULL;
  }

  if (access(path, R_OK)) {
    free(path);
    if ((path = strdup(ESECANNA_RC_PATH)) == NULL) {
      m_msg("out of memory!\n");
      return NULL;
    }

    if (access(path, R_OK)) {
      m_msg("No conffile found.\n");
      return NULL;
    }
  }

  m_msg("Config file %s\n", path);

  return path;
}

static int iw_read_conf_file(char *path)
{
  FILE *fp;
  char buf[1024];
  char *ope, *val;
  int ret = IME_NON;

  if ((fp = fopen(path, "r")) == NULL) {
    m_msg("Cannot open Conffile %s.\n", path);
    return IME_NON;
  }
  
  while (fgets(buf, 1024, fp)) {
    if (buf[0] != '#' && m_conf1_parse(buf, &ope, &val) == 0) {
      if (m_conf_isequal(ope, "IME", val, "VJE25") == 2)
	ret = IME_VJE25;
      if (m_conf_isequal(ope, "IME", val, "VJE30") == 2)
	ret = IME_VJE30;
      if (m_conf_isequal(ope, "IME", val, "ATOK") == 2)
	ret = IME_ATOK;
      if (m_conf_isequal(ope, "IME", val, "WNN6") == 2)
	ret = IME_WNN;
      if (ret == IME_NON && m_conf_isequal(ope, "IME", val, "dummy"))
	m_msg("Unsupported IME: [%s]\n", val);
      
      if (ret != IME_NON)
	break;
    }
  }

  fclose(fp);

  return ret;
}

/*
 * ʥߥå⥸塼ɤ
 */

static int iw_retrieve_vje_symbols(void *handler)
{
  int i;
  char buf[100];
  int (*dl_start_func)();

  for (i = 0; i <= F_COMMON_END; i++) {
    strcpy(buf, "vjewrapper");
    strcat(buf, funcsymbols[i]);
    
    vjefunc[i] = dlsym(handler, buf);
  }

  vjefunc[VJE_clear_vjeid] = dlsym(handler, "vjewrapper_clear_vjeid");
  vjefunc[VJE_restarted] = dlsym(handler, "vjewrapper_restarted");
  vjefunc[VJE_connect_unix] = dlsym(handler, "vje_socket_connect_unix");
  vjefunc[VJE_close_connection] = dlsym(handler,"vje_socket_close_connection");

  dl_start_func = dlsym(handler, "vjewrapper_dl_started");

  return dl_start_func(client);
}

static int iw_retrieve_atok_symbols(void *handler)
{
  extern char debugmode;
  int i;
  char buf[100];
  int (*dl_start_func)();

  for (i = 0; i <= F_COMMON_END; i++) {
    strcpy(buf, "atokwrapper");
    strcat(buf, funcsymbols[i]);
    
    atokfunc[i] = dlsym(handler, buf);
  }

  dl_start_func = dlsym(handler, "atokwrapper_dl_started");

  return dl_start_func(client, debugmode);
}

static int iw_retrieve_wnn_symbols(void *handler)
{
  int i;
  char buf[100];
  int (*dl_start_func)();

  for (i = 0; i <= F_COMMON_END; i++) {
    strcpy(buf, "wnnwrapper");
    strcat(buf, funcsymbols[i]);
    
    wnnfunc[i] = dlsym(handler, buf);
  }

  dl_start_func = dlsym(handler, "wnnwrapper_dl_started");

  return dl_start_func(client);
}

static int iw_load_library(int ime)
{
  char buf[1024];
  
  if (ime == IME_VJE25 && ime_dl_handler[IME_VJE30 - 1] != NULL) {
    m_msg("vje30.so already loaded so vje25 not loaded.\n");
    return -1;
  }

  if (ime == IME_VJE30 && ime_dl_handler[IME_VJE25 - 1] != NULL) {
    m_msg("vje25.so already loaded so vje30 not loaded.\n");
    return -1;
  }

  if (ime_dl_handler[ime - 1] == NULL) {
    sprintf(buf, "%s/%s", ESECANNA_DL_PATH, ime_dl_name[ime - 1]);
    
    if ((ime_dl_handler[ime - 1] = dlopen(buf, RTLD_NOW)) == NULL) {
      m_msg("unable to load module %s\n", ime_dl_name[ime - 1]);
      m_msg(dlerror());

      return -1;
    }

    m_msg("Module %s loaded.\n", ime_dl_name[ime - 1]);

    switch (ime) {
      case IME_VJE25:
      case IME_VJE30:
	return iw_retrieve_vje_symbols(ime_dl_handler[ime - 1]);
      case IME_ATOK:
	return iw_retrieve_atok_symbols(ime_dl_handler[ime - 1]);
      case IME_WNN:
	return iw_retrieve_wnn_symbols(ime_dl_handler[ime - 1]);
    }
  }

  return 0;
}

/*
 * ʥߥå⥸塼򥢥ɤ
 */

static int iw_unload_library(int ime)
{
  if (ime_dl_handler[ime - 1]) {
    dlclose(ime_dl_handler[ime - 1]);

    ime_dl_handler[ime - 1] = NULL;

    m_msg("Module %s unloaded.\n", ime_dl_name[ime - 1]);
  }

  return 0;
}    

/*
 *  饤Ȥνλ򤹤 
 */

int imewrapper_end_client(int id)
{
  int ime = client[id].ime;
  imefunc_t *func;
  
  if (ime > 0 && ime_dl_handler[ime - 1]) {
    func = imefunc[ime - 1];
    (func[F_end_client])(id);
  }
  
  return 0;
}

/*
 *  IME νλ򤹤롣
 */
 
int imewrapper_end_rootclient(int ime)
{
  imefunc_t *func;
  
  if (ime > 0 && ime_dl_handler[ime - 1] != NULL) {
    /* ʥߥå⥸塼뤬ɤƤΤʤ齪λ򤹤 */
    func = imefunc[ime - 1];
    (func[F_end_rootclient])();
    
    /* ⥸塼 */
    iw_unload_library(ime);

    /* ³ե饰򥯥ꥢ */
    ime_connected_flag[ime - 1] = FALSE;
  }
  
  return 0;
}

/*
 * client[] Υǡ򥯥ꥢؿƤ
 */

int imewrapper_clear_client_data(int id)
{
  int ime = client[id].ime;
  imefunc_t *func;
  
  if (ime > 0 && ime_dl_handler[ime - 1]) {
    func = imefunc[ime - 1];
    (func[F_clear_client_data])(id);
  }
  
  MYFREE(client[id].host);
  MYFREE(client[id].homedir);
  
  client[id].ime = IME_NON;

  memset(&(client[id]), 0, sizeof(client_t));
  
  client[id].sockfd = -1;

  return 0;
}
      


/*
 * IME ۾ｪλȤνطδؿs
 */

static int iw_vje_aborted(int ime)
{
#if 0
  /* Ф餯Ƶư disable */
  /* FIXME: δؿϡvjewrapper.c ˤ٤ */
  
  (vjefunc[VJE_close_connection])();

  if (client[VJE_ROOT_CLIENT].restart_vjed && client[VJE_ROOT_CLIENT].restart
      && access(client[VJE_ROOT_CLIENT].restart, R_OK | X_OK) == 0) {
    /* vjed Ƶư³ */
    m_msg("iw_vje_aborted(): vjed down. Restarting...\n");
    m_system(client[VJE_ROOT_CLIENT].restart);
    sleep(1);

    /* vjeid ƥꥢBeginConvert λ
       Ƥ vjeid 褦ˤ */
    (vjefunc[VJE_clear_vjeid])();
    
    if ((vjefunc[VJE_connect_unix])() == 0) {
      /* 餯ƵưƤΤ vjeerror 򥯥ꥢ */
      (vjefunc[VJE_restarted])();
      
      if ((vjefunc[F_init_rootclient])() == 0) {
	return 0;
      } else {
	(vjefunc[VJE_close_connection])();
      }
    }

    m_msg("failed.\n");
  } else {
    m_msg("vjed down.\n");
  }

#endif

  /* Ƶư˼Ԥϡmain.c Ǥ IME ³Ƥ
   * Ƥ(rootclient ޤ)饤Ȥνλ򤹤뤿ᡢ
   * -ime ֤
   */
  
  return -ime;
}

static int iw_wnn_aborted(int ime)
{
  return -ime;
}

static int iw_atok_aborted(int ime)
{
  return -ime;
}

int imewrapper_ime_aborted(int ime)
{
  switch (ime) {
    case IME_VJE25:
    case IME_VJE30:
      return iw_vje_aborted(ime);
    case IME_ATOK:
      return iw_atok_aborted(ime);
    case IME_WNN:
      return iw_wnn_aborted(ime);
  }

  return -1;
}

/*
 * ʤλ뤿δؿ
 */

static int iw_send_term_signal()
{
  raise(SIGTERM);

  return 0;
}

/*
 * VJE ˤޤ³Ƥʤ˸ƤФ롣³롣Ԥʤ顢
 * ⥸塼롣
 */

static int iw_vje_connect(int ime)
{
  /* ⥸塼 */
  if (iw_load_library(ime) == 0) {

    /* VJE ³ */
    if ((vjefunc[VJE_connect_unix])() == 0) {
      
      /* ³ξͤ restarted Ƥǡե饰򥯥ꥢ */
      (vjefunc[VJE_restarted])();
      
      /*  */
      if ((vjefunc[F_init_rootclient])() == 0) {
	/* VJE ³ & */
	
	client[VJE_ROOT_CLIENT].ime = ime;
	ime_connected_flag[ime - 1] = TRUE;

	return 0;
      } else {
	/* ˤϼԡ³ڤ */

	(vjefunc[F_end_rootclient])();
      }
    }
    /* ³˼ԡ⥸塼 */
      
    iw_unload_library(ime);
  }

  return -1;
}

/*
 * ATOK ˤޤ³Ƥʤ˸ƤФ
 */

static int iw_atok_connect(int ime)
{
  /* ⥸塼 */
  if (iw_load_library(ime) == 0) {
    /*  */
    if ((atokfunc[F_init_rootclient])() == 0) {
      /* ATOK ³ & */
      
      client[ATOK_ROOT_CLIENT].ime = ime;
      ime_connected_flag[ime - 1] = TRUE;
      
      return 0;
    } else {
      (atokfunc[F_end_rootclient])();
    }

    iw_unload_library(ime);
  }

  return -1;
}

/*
 * Wnn ˤޤ³Ƥʤ˸ƤФ
 */

static int iw_wnn_connect(int ime)
{
  /* ⥸塼 */
  if (iw_load_library(ime) == 0) {
    /*  */
    if ((wnnfunc[F_init_rootclient])() == 0) {
      /* Wnn ³ & */
      
      client[WNN_ROOT_CLIENT].ime = ime;
      ime_connected_flag[ime - 1] = TRUE;
      
      return 0;
    } else {
      (wnnfunc[F_end_rootclient])();
    }

    iw_unload_library(ime);
  }

  return -1;
}
    
/*
 * IME ˤޤ³Ƥʤ˸ƤФ
 */

static int iw_ime_connect(int ime)
{
  switch (ime) {
    case IME_VJE25:
    case IME_VJE30:
      return iw_vje_connect(ime);
      break;
    case IME_ATOK:
      return iw_atok_connect(ime);
      break;
    case IME_WNN:
      return iw_wnn_connect(ime);
      break;
  }

  return -1;
}
      
/*
 * ʥץȥ wrap ؿs
 */

int imewrapper_initialize(int id, buffer_t *cbuf)
{
  int ime = IME_NON, errflag, *ip = (int *)cbuf->buf;
  short *sp = (short *)cbuf->buf;
  short cx_num;
  char *p, *major_p, *minor_p, *user = NULL, *home;
  short major, minor;
  struct passwd *pwd;
  char *conffile = NULL;
  imefunc_t *func;

  errflag = 0;

  major_p = &(cbuf->buf[0]); /* Initialize ˸¤, cbuf->buf ˥إå
				ʤ */
  home = NULL;
  major = minor = -1;

  if ((p = strchr(major_p, '.')) == NULL)
    errflag = 1;
  else {
    *p = 0;
    
    minor_p = p + 1;
    
    if ((p = strchr(minor_p, ':')) == NULL)
      errflag = 2;
    else {
      *p = 0;
      user = p + 1;

      if (user[0] == 0)
	errflag = 3;
      else {
	if ((pwd = getpwnam(user)) == NULL)
	  errflag = 4;
	else {
	  if ((home = pwd->pw_dir) == NULL)
	    errflag = 5;
	  else {
	    if ((conffile = iw_get_conf_file_path(home)) == NULL)
	      errflag = 6;

	    major = atoi(major_p);
	    minor = atoi(minor_p);
	  }
	}
      }
    }
  }

  /* FIXME: major, minor ǾˤäƤϥ顼֤褦 */

  if (errflag)
    m_msg("Header invalid. Maybe server version mismatch?\n");

  if (errflag == 0) {
    if ((ime = client[id].ime = iw_read_conf_file(conffile)) != 0) {
      if (ime_connected_flag[ime - 1] == FALSE) {
	/* ³׵ᤵ줿 IME ˤޤ³Ƥʤ */

	iw_ime_connect(ime);
      }
    } else {
      m_msg("IME not determined.\n");
    }
  }

  if (errflag == 0 && ime > 0 && ime_connected_flag[ime - 1] == TRUE) {
    /* client[]  user ̾, ۡǥ쥯ȥΥѥݴ */
    strncpy(client[id].user, user, 10);
    client[id].user[9] = 0;
    client[id].homedir = strdup(home);

    func = imefunc[ime - 1];

    cx_num = (func[F_initialize])(id, conffile);
  } else {
    cx_num = -1;
  }

  if (cx_num == -1) {
    m_msg("Initialize failed. #%d %s@%s refused.\n", id, user ? user : "",
	  client[id].host);
    
    /* ꥽եɤ߹ʳǤμԡޤϥƥȤγ
       ԡ饤ȤȤ³ڤäƤޤ */
    client[id].need_terminate = TRUE; /* main.c ǽλ򤷤Ƥ餦 */
    *ip = LSBMSB32(-1);
  } else {  /* Success */
    sp[0] = LSBMSB16(3);
    sp[1] = LSBMSB16(cx_num);
  }

  if (conffile)
    free(conffile);

  return 1;
}

#define CALLFUNC(_num) imefunc_t *func; func = imefunc[client[id].ime - 1]; \
  return (func[_num])(id, cbuf); \


int imewrapper_finalize(int id, buffer_t *cbuf)
{
  CALLFUNC(0);
}

int imewrapper_create_context(int id, buffer_t *cbuf)
{
  CALLFUNC(1);
}

int imewrapper_duplicate_context(int id, buffer_t *cbuf)
{
  CALLFUNC(2);
}

int imewrapper_close_context(int id, buffer_t *cbuf)
{
  CALLFUNC(3);
}

int imewrapper_get_dictionary_list(int id, buffer_t *cbuf)
{
  cannaheader_t *header = (cannaheader_t *)cbuf->buf;

  header->datalen = LSBMSB16(2);
  header->err.e16 = 0;
  
  return 1;
}

int imewrapper_get_directory_list(int id, buffer_t *cbuf)
{
  cannaheader_t *header = (cannaheader_t *)cbuf->buf;

  header->datalen = LSBMSB16(2);
  header->err.e16 = 0;
  
  return 1;
}

int imewrapper_mount_dictionary(int id, buffer_t *cbuf)
{
  cannaheader_t *header = (cannaheader_t *)cbuf->buf;
  
  header->datalen = LSBMSB16(1);
  header->err.e8 = 0;
  
  return 1;
}

int imewrapper_unmount_dictionary(int id, buffer_t *cbuf)
{
  cannaheader_t *header = (cannaheader_t *)cbuf->buf;
  
  header->datalen = LSBMSB16(1);
  header->err.e8 = 0;

  return 1;
}

int imewrapper_remount_dictionary(int id, buffer_t *cbuf)
{
  cannaheader_t *header = (cannaheader_t *)cbuf->buf;

  header->datalen = LSBMSB16(1);
  header->err.e8 = 0;
  
  return 1;
}

int imewrapper_get_mountdictionary_list(int id, buffer_t *cbuf)
{
  cannaheader_t *header = (cannaheader_t *)cbuf->buf;

  header->datalen = LSBMSB16(2);
  header->err.e16 = 0;
  
  return 1;
}

int imewrapper_query_dictionary(int id, buffer_t *cbuf)
{
  cannaheader_t *header = (cannaheader_t *)cbuf->buf;

  header->datalen = LSBMSB16(1);
  header->err.e8 = -1;
  
  return 1;
}

int imewrapper_define_word(int id, buffer_t *cbuf)
{
  CALLFUNC(4);
}

int imewrapper_delete_word(int id, buffer_t *cbuf)
{
  CALLFUNC(5);
}

int imewrapper_begin_convert(int id, buffer_t *cbuf)
{
  CALLFUNC(6);
}

int imewrapper_end_convert(int id, buffer_t *cbuf)
{
  CALLFUNC(7);
}

int imewrapper_get_candidacy_list(int id, buffer_t *cbuf)
{
  CALLFUNC(8);
}

int imewrapper_get_yomi(int id, buffer_t *cbuf)
{
  CALLFUNC(9);
}

int imewrapper_subst_yomi(int id, buffer_t *cbuf)
{
  CALLFUNC(10);
}

int imewrapper_store_yomi(int id, buffer_t *cbuf)
{
  CALLFUNC(11);
}

int imewrapper_store_range(int id, buffer_t *cbuf)
{
  CALLFUNC(12);
}

int imewrapper_get_lastyomi(int id, buffer_t *cbuf)
{
  CALLFUNC(13);
}

int imewrapper_flush_yomi(int id, buffer_t *cbuf)
{
  CALLFUNC(14);
}

int imewrapper_remove_yomi(int id, buffer_t *cbuf)
{
  CALLFUNC(15);
}

int imewrapper_get_simplekanji(int id, buffer_t *cbuf)
{
  CALLFUNC(16);
}

int imewrapper_resize_pause(int id, buffer_t *cbuf)
{
  CALLFUNC(17);
}

int imewrapper_get_hinshi(int id, buffer_t *cbuf)
{
  CALLFUNC(18);
}

int imewrapper_get_lex(int id, buffer_t *cbuf)
{
  CALLFUNC(19);
}

int imewrapper_get_status(int id, buffer_t *cbuf)
{
  CALLFUNC(20);
}

int imewrapper_set_locale(int id, buffer_t *cbuf)
{
  CALLFUNC(21);
}

int imewrapper_auto_convert(int id, buffer_t *cbuf)
{
  CALLFUNC(22);
}

int imewrapper_query_extensions(int id, buffer_t *cbuf)
{
  cannaheader_t *header = (cannaheader_t *)cbuf->buf;

  header->datalen = LSBMSB16(1);
  header->err.e8 = 0;
  
  return 1;
}

int imewrapper_set_applicationname(int id, buffer_t *cbuf)
{
  cannaheader_t *header = (cannaheader_t *)cbuf->buf;

  header->datalen = LSBMSB16(1);
  header->err.e8 = 0;
  
  return 1;
}


int imewrapper_notice_groupname(int id, buffer_t *cbuf)
{
  cannaheader_t *header = (cannaheader_t *)cbuf->buf;

  header->datalen = LSBMSB16(1);
  header->err.e8 = 0;
  
  return 1;
}

int imewrapper_through(int id, buffer_t *cbuf)
{
  cannaheader_t *header = (cannaheader_t *)cbuf->buf;

  header->datalen = LSBMSB16(2);
  header->err.e16 = 0;
  
  return 1;
}

int imewrapper_kill_server(int id, buffer_t *cbuf)
{
  cannaheader_t *header = (cannaheader_t *)cbuf->buf;
  int err = 0;
  char buf[128];
  struct passwd *pw;
  uid_t pid;

  if (gethostname(buf, 128))
    buf[0] = 0;
  pw = getpwnam(client[id].user);
  pid = getuid();
  
  if (strcmp(client[id].host, "UNIX") && strcmp(client[id].host, buf) &&
      strcmp(client[id].host, "localhost"))
    err = -111; /* NOTUXSRV */
  else if (pw->pw_uid != 0 && pw->pw_gid != 0 && pw->pw_uid != pid)
    err = -112; /* NOTOWNSRV */

  header->datalen = LSBMSB16(1);
  header->err.e8 = err;

  if (err == 0) {
    signal(SIGALRM, (void(*)())iw_send_term_signal);
    alarm(1);

    m_msg("KillServer from %s@%s accepted.\n",
	  client[id].user, client[id].host);
  }
  
  return 1;
}

int imewrapper_get_serverinfo(int id, buffer_t *cbuf)
{
  cannaheader_t *header = (cannaheader_t *)cbuf->buf;
  int protocol_num, e_protocol_num, datalen, protocol_datalen, pnt;
  int i;
  uint ui;
  short s;

  for (protocol_num = 0; protocol_name[protocol_num] != NULL; protocol_num++);
  for (e_protocol_num = 0; e_protocol_name[e_protocol_num] != NULL;
       e_protocol_num++);
  
  protocol_datalen = 0;
  for (i = 0; i < protocol_num; i++)
    protocol_datalen += strlen(protocol_name[i]) + 1;
  for (i = 0; i < e_protocol_num; i++)
    protocol_datalen += strlen(e_protocol_name[i]) + 1;
  protocol_datalen++;
  
  datalen = 17 + protocol_datalen + (protocol_num + e_protocol_num) * 4;

  buffer_check(cbuf, datalen + 4);
  header = (cannaheader_t *)cbuf->buf;
  
  header->type = 0x01;
  header->extra = 0x01;
  header->datalen = LSBMSB16(datalen);

  cbuf->buf[4] = 0;
  cbuf->buf[5] = 3; /* Major Server Version */
  cbuf->buf[6] = 5; /* Minor Server Version */
  
  ui = (uint)time(NULL); ui = LSBMSB32(ui);
  memcpy(&(cbuf->buf[7]), &ui, 4);

  i = protocol_num + e_protocol_num;
  s = LSBMSB16(i);
  memcpy(&(cbuf->buf[11]), &s, 2);

  s = LSBMSB16(protocol_datalen);
  memcpy(&(cbuf->buf[13]), &s, 2);

  pnt = 15;
  for (i = 0; i < protocol_num; i++) {
    strcpy(&(cbuf->buf[pnt]), protocol_name[i]);
    pnt += strlen(protocol_name[i]) + 1;
  }

  for (i = 0; i < e_protocol_num; i++) {
    strcpy(&(cbuf->buf[pnt]), e_protocol_name[i]);
    pnt += strlen(e_protocol_name[i]) + 1;
  }
  
  cbuf->buf[pnt++] = 0;

  memset(&(cbuf->buf[pnt]), 0, 4 * (protocol_num + e_protocol_num));
  pnt += 4 * (protocol_num + e_protocol_num);

  memset(&(cbuf->buf[pnt]), 0, 6);
  pnt += 6;

  return 1;
}

int imewrapper_get_access_control_list(int id, buffer_t *cbuf)
{
  cannaheader_t *header = (cannaheader_t *)cbuf->buf;

  header->datalen = LSBMSB16(2);
  header->err.e16 = LSBMSB16(-1);
  
  return 1;
}

int imewrapper_create_dictionary(int id, buffer_t *cbuf)
{
  cannaheader_t *header = (cannaheader_t *)cbuf->buf;

  header->datalen = LSBMSB16(1);
  header->err.e8 = -1;
  
  return 1;
}

int imewrapper_delete_dictionary(int id, buffer_t *cbuf)
{
  cannaheader_t *header = (cannaheader_t *)cbuf->buf;

  header->datalen = LSBMSB16(1);
  header->err.e8 = -1;
  
  return 1;
}

int imewrapper_rename_dictionary(int id, buffer_t *cbuf)
{
  cannaheader_t *header = (cannaheader_t *)cbuf->buf;

  header->datalen = LSBMSB16(1);
  header->err.e8 = -1;
  
  return 1;
}

int imewrapper_get_wordtext_dictionary(int id, buffer_t *cbuf)
{
  cannaheader_t *header = (cannaheader_t *)cbuf->buf;

  header->datalen = LSBMSB16(2);
  header->err.e16 = LSBMSB16(-1);
  
  return 1;
}

int imewrapper_list_dictionary(int id, buffer_t *cbuf)
{
  cannaheader_t *header = (cannaheader_t *)cbuf->buf;

  header->datalen = LSBMSB16(1);
  header->err.e8 = -1;
  
  return 1;
}

int imewrapper_sync(int id, buffer_t *cbuf)
{
 cannaheader_t *header = (cannaheader_t *)cbuf->buf;

  header->datalen = LSBMSB16(1);
  header->err.e8 = 0;
  
  return 1;
}

int imewrapper_chmod_dictionary(int id, buffer_t *cbuf)
{
 cannaheader_t *header = (cannaheader_t *)cbuf->buf;

  header->datalen = LSBMSB16(1);
  header->err.e8 = -1;
  
  return 1;
}

int imewrapper_copy_dictionary(int id, buffer_t *cbuf)
{
 cannaheader_t *header = (cannaheader_t *)cbuf->buf;

  header->datalen = LSBMSB16(1);
  header->err.e8 = -1;
  
  return 1;
}
